import acm.graphics.GContainer;

/**
 * A full description of the state of a Beast. Why have this separate class
 * rather than looking directly at the Beast? Well, any method that gets a
 * reference to a Beast can actually CHANGE its state, which a beast brain
 * should be able to do. So, instead, we give the beast brains this info which
 * is immutable (just as a String is immutable).
 * 
 * @author CPSC 111 instructors
 */
public class BeastInfo
{
    /**
     * The beast's unique ID. Every beast in a given simulation should have its
     * own distinct ID.
     */
    private int id;

    /** The beast's age in steps taken since creation. */
    private int age;

    /** The beast's x location. */
    private double x;

    /** The beast's x location. */
    private double y;

    /** The beast's current mood. */
    private BeastMood mood;

    /** The beast's current direction. (The direction of its LAST step.) */
    private Direction direction;

    /** The beast's brain. */
    private IBeastBrain brain;

    /**
     * Create a new BeastInfo from the given Beast.
     * 
     * @param beast the beast to "scrape" info from, non-null
     */
    public BeastInfo( Beast beast )
    {
        // Note: it's maybe a bit illegitimate to have BeastInfo scrounging
        // around in a Beast's parent container for an ID. However, it otherwise
        // makes sense to have a BeastInfo build itself from a Beast.

        // Pick the id as the location in the parent's order of Beasts.
        GContainer container = beast.getParent();
        int id = -1;
        if( container != null )
        {
            for( int i = 0; i < container.getElementCount(); i++ )
            {
                if( container.getElement( i ) == beast )
                {
                    id = i;
                    break;
                }
            }
        }

        this.id = id;
        this.age = beast.getAge();
        this.brain = beast.getBrain();
        this.direction = new Direction( beast.getDirection() );
        this.mood = beast.getMood();
        this.x = beast.getX();
        this.y = beast.getY();
    }

    /**
     * Get the beast's unique ID (or -1 if no unique ID could be found for the
     * beast).
     * 
     * @return an ID
     */
    public int getId()
    {
        return this.id;
    }

    /**
     * Get the beast's age in steps since creation.
     * 
     * @return the beast's age
     */
    public int getAge()
    {
        return this.age;
    }

    /**
     * Get the beast's brain.
     * 
     * @return the beast's brain
     */
    public IBeastBrain getBrain()
    {
        return this.brain;
    }

    /**
     * Get the beast's direction
     * 
     * @return the beast's direction
     */
    public Direction getDirection()
    {
        return this.direction;
    }

    /**
     * Get the beast's mood.
     * 
     * @return the beast's mood
     */
    public BeastMood getMood()
    {
        return this.mood;
    }

    /**
     * Get the beast's x location in the simulation.
     * 
     * @return the beast's x location
     */
    public double getX()
    {
        return this.x;
    }

    /**
     * Get the beast's y location in the simulation.
     * 
     * @return the beast's y location
     */
    public double getY()
    {
        return this.y;
    }


    public boolean equals( Object obj )
    {
        // Fob off responsibility on someone else if the object is not a
        // BeastInfo
        if( !( obj instanceof BeastInfo ) )
            return super.equals( obj );

        // Cast the object to a BeastInfo.
        BeastInfo other = (BeastInfo) obj;

        // Check that all fields match.
        return other.age == this.age && other.brain.equals( this.brain )
                && other.direction.equals( this.direction )
                && other.id == this.id && other.mood.equals( this.mood )
                && other.x == this.x && other.y == this.y;
    }
}
