#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#ifdef __APPLE__
#include <GLUT/glut.h>
#else
#include <GL/glut.h>
#endif

// globals for rendering
int fullscreen_mode=0;
int width=800, height=600;
int saved_width, saved_height;
int file_count=0;
int xslice=0, yslice=0, zslice=0, axis=0;
int start=0;

// globals for data
int frame=0;
const char *directory=0;
int nx=0, ny=0, nz=0;
float *s=0;
#define S(i,j,k) (s[((i)*ny+(j))*nz+(k)])

// prototypes
void read_frame(int new_frame);
void handle_display();
void handle_reshape(int w, int h);
void handle_keypress(unsigned char key, int x, int y);
void toggle_fullscreen();
void display_smoke();
void save_screen(const char *filename);

// definitions
int main(int argc, char *argv[])
{
   // initialize GLUT stuff 
   glutInit(&argc, argv);
   glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGBA|GLUT_DEPTH);
   glutInitWindowSize(width,height);
   saved_width=width;
   saved_height=height;
   glutCreateWindow("smoke slice viewer (hw6)");
   glutDisplayFunc(handle_display);
   glutReshapeFunc(handle_reshape);
   glutKeyboardFunc(handle_keypress);

   // initialize OpenGL stuff
   glEnable(GL_DEPTH_TEST);
   glClearColor(0, 0, 0, 0);
   glClearDepth(1);
   glPixelStorei(GL_PACK_ALIGNMENT, 1);
   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

   if(argc<2){
      fprintf(stderr, "Expecting <directory> as argument\n");
      return 1;
   }
   directory=argv[1];

   // initialize first frame
   read_frame(0);

   // let's go
   glutPostRedisplay();
   glutMainLoop();
   return 0;
}

void read_frame(int new_frame)
{
   char *filename;
   FILE *fp;
   int i;

   if(new_frame<0) return;
   filename=malloc(strlen(directory)+30);
   sprintf(filename, "%s/smoke%04d", directory, new_frame);
   fp=fopen(filename, "rt");
   if(fp){
      int newnx, newny, newnz;
      fscanf(fp, "%d %d %d", &newnx, &newny, &newnz);
      if(newnx!=nx || newny!=ny || newnz!=nz || s==0){
         free(s);
         nx=newnx;
         ny=newny;
         nz=newnz;
         s=(float*)malloc(nx*ny*nz*sizeof(float));
      }
      frame=new_frame;
      for(i=0; i<nx*ny*nz; ++i){
         fscanf(fp, "%f", s+i);
      }
      fclose(fp);
   }
   free(filename);
}

void handle_display()
{
   glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   switch(axis){
      case 0:
         if(ny/(float)nz > height/(float)width){
            float aspect=width/(float)height;
            gluOrtho2D(0.5*nz-0.5*aspect*ny, 0.5*nz+0.5*aspect*ny, 0, ny);
         }else{
            float aspect=height/(float)width;
            gluOrtho2D(0, nz, 0.5*ny-0.5*aspect*nz, 0.5*ny+0.5*aspect*nz);
         }
         break;
      case 1:
         if(nz/(float)nx > height/(float)width){
            float aspect=width/(float)height;
            gluOrtho2D(0.5*nx-0.5*aspect*nz, 0.5*nx+0.5*aspect*nz, 0, nz);
         }else{
            float aspect=height/(float)width;
            gluOrtho2D(0, nx, 0.5*nz-0.5*aspect*nx, 0.5*nz+0.5*aspect*nx);
         }
         break;
      case 2:
         if(ny/(float)nx > height/(float)width){
            float aspect=width/(float)height;
            gluOrtho2D(0.5*nx-0.5*aspect*ny, 0.5*nx+0.5*aspect*ny, 0, ny);
         }else{
            float aspect=height/(float)width;
            gluOrtho2D(0, nx, 0.5*ny-0.5*aspect*nx, 0.5*ny+0.5*aspect*nx);
         }
         break;
   }

   display_smoke();

   glutSwapBuffers();
   if(start==0){
      start=1;
      display_smoke();
      glutSwapBuffers();
   }
}

void handle_reshape(int w, int h)
{
   width=w;
   height=h;
   glViewport(0, 0, (GLsizei)width, (GLsizei)height);
   glutPostRedisplay();
}

void handle_keypress(unsigned char key, int x, int y)
{
   switch(key){
   case '.':
      read_frame(frame+1);
      break;
   case ',':
      read_frame(frame-1);
      break;
   case '>':
      read_frame(frame+10);
      break;
   case '<':
      read_frame(frame-10);
      break;
   case 'a':
      axis=(axis+1)%3;
      printf("axis: %d\n", axis);
      break;
   case '[':
      switch(axis){
         case 0: --xslice; if(xslice<0) xslice=0; printf("x: %d\n", xslice); break;
         case 1: --yslice; if(yslice<0) yslice=0; printf("y: %d\n", yslice); break;
         case 2: --zslice; if(zslice<0) zslice=0; printf("z: %d\n", zslice); break;
      }
      break;
   case ']':
      switch(axis){
         case 0: ++xslice; if(xslice>=nx) xslice=nx-1; printf("x: %d\n", xslice); break;
         case 1: ++yslice; if(yslice>=ny) yslice=ny-1; printf("y: %d\n", yslice); break;
         case 2: ++zslice; if(zslice>=nz) zslice=nz-1; printf("z: %d\n", zslice); break;
      }
      break;
   case 'f':
      toggle_fullscreen();
      break;
   case 'q':
      exit(0);
   case 's':
      {
         char filename[30];
         sprintf(filename, "hw6_%04d.ppm", file_count);
         save_screen(filename);
         ++file_count;
      }
      break;
   default:
      return;
   }
   glutPostRedisplay();
}

void toggle_fullscreen()
{
   if(fullscreen_mode){
      fullscreen_mode=0;
      width=saved_width;
      height=saved_height;
      glutReshapeWindow(saved_width, saved_height);
   }else{
      fullscreen_mode=1;
      saved_width=width;
      saved_height=height;
      glutFullScreen();
   }
}

void draw_x()
{
   int j, k;
   if(xslice>nx-1) xslice=nx-1;
   for(j=0; j<ny-1; ++j){
      glBegin(GL_QUAD_STRIP);
      for(k=0; k<nz; ++k){
         float g=S(xslice,j,k);
         glColor3f(g,g,g);
         glVertex2i(k,j);
         g=S(xslice,j+1,k);
         glColor3f(g,g,g);
         glVertex2i(k,j+1);
      }
      glEnd();
   }
}

void draw_y()
{
   int i, k;
   if(yslice>ny-1) yslice=ny-1;
   for(i=0; i<nx-1; ++i){
      glBegin(GL_QUAD_STRIP);
      for(k=0; k<nz; ++k){
         float g=S(i,yslice,k);
         glColor3f(g,g,g);
         glVertex2i(i,k);
         g=S(i+1,yslice,k);
         glColor3f(g,g,g);
         glVertex2i(i+1,k);
      }
      glEnd();
   }
}

void draw_z()
{
   int i, j;
   if(zslice>nz-1) zslice=nz-1;
   for(i=0; i<nx-1; ++i){
      glBegin(GL_QUAD_STRIP);
      for(j=0; j<ny; ++j){
         float g=S(i,j,zslice);
         glColor3f(g,g,g);
         glVertex2i(i,j);
         g=S(i+1,j,zslice);
         glColor3f(g,g,g);
         glVertex2i(i+1,j);
      }
      glEnd();
   }
}

void display_smoke()
{
   switch(axis){
      case 0: draw_x(); break;
      case 1: draw_y(); break;
      case 2: draw_z(); break;
   }
}

void save_screen(const char *filename)
{
   FILE *fp;
   GLubyte *image_buffer;

   // get the pixels
   image_buffer=malloc(3*width*height*sizeof(GLubyte));
   glReadBuffer(GL_FRONT);
   glReadPixels(0, 0, width, height, GL_RGB, GL_UNSIGNED_BYTE, image_buffer);
  
   // write them to the file
   fp=fopen(filename, "wb");
   if(fp){
      int i;
      fprintf(fp, "P6\n%d %d 255\n", width, height);
      for(i=1; i<=height; ++i){
         fwrite(image_buffer+3*width*(height-i), 1, 3*width, fp);
      }
   }
   
   // clean up
   free(image_buffer);
}

