#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#ifdef __APPLE__
#include <GLUT/glut.h>
#else
#include <GL/glut.h>
#endif

// globals for rendering
int fullscreen_mode=0;
int width=800, height=600;
int saved_width, saved_height;
int oldmousex=0, oldmousey=0;
enum {DONOTHING, ROTATE, ZOOM, MOVETARGET} mousemode=DONOTHING;
float camdist=6;
const float NEARCLIP_FACTOR=0.1, FARCLIP_FACTOR=10;
float nearclip, farclip;
float target[3]={0, 1, 0};
float view_target_timer=0;
float old_time=0;
const float fovy=45;
float heading=M_PI/3, pitch=0;
float campos[3];
int file_count=0;
int wireframe_mode=0;

// globals for data
int frame=0;
const char *directory=0;
int num_tiles=1;
int nx=0, nz=0;
float *h=0, *normals=0;

// prototypes
void read_frame(int new_frame);
void compute_normals();
void find_reasonable_camera();
void handle_display();
void handle_reshape(int w, int h);
void handle_keypress(unsigned char key, int x, int y);
void handle_click(int button, int state, int x, int y);
void handle_drag(int x, int y);
void handle_idle();
void toggle_fullscreen();
void compute_camera_position();
void display_heightfields();
void display_target();
void save_screen(const char *filename);
void init_lighting();
void init_material();

// definitions
int main(int argc, char *argv[])
{
   // initialize GLUT stuff 
   glutInit(&argc, argv);
   glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGBA|GLUT_DEPTH);
   glutInitWindowSize(width,height);
   saved_width=width;
   saved_height=height;
   glutCreateWindow("heightfield viewer (hw5)");
   glutDisplayFunc(handle_display);
   glutReshapeFunc(handle_reshape);
   glutKeyboardFunc(handle_keypress);
   glutMouseFunc(handle_click);
   glutMotionFunc(handle_drag);
   glutIdleFunc(handle_idle);

   // initialize OpenGL stuff
   glEnable(GL_DEPTH_TEST);
   glClearColor(0, 0, 0, 0);
   glClearDepth(1);
   glPixelStorei(GL_PACK_ALIGNMENT, 1);
   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   init_lighting();
   glEnable(GL_LIGHTING);
   glDisable(GL_CULL_FACE);
   glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, 1);
   init_material();

   if(argc<2){
      fprintf(stderr, "Expecting <directory> as argument\n");
      return 1;
   }
   directory=argv[1];
   if(argc>2)
      sscanf(argv[2], "%d", &num_tiles);

   // initialize first frame
   read_frame(0);
   find_reasonable_camera();
   nearclip=NEARCLIP_FACTOR*camdist;
   farclip=FARCLIP_FACTOR*camdist;
   compute_camera_position();

   // let's go
   old_time=clock()/(float)CLOCKS_PER_SEC;
   glutPostRedisplay();
   glutMainLoop();
   return 0;
}

void init_lighting()
{
   {
      GLfloat ambient[4] = {.3, .3, .3, 1};
      glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);
   }{
      GLfloat color[4] = {.7, .7, .7, 1};
      glLightfv(GL_LIGHT0, GL_DIFFUSE, color);
      glLightfv(GL_LIGHT0, GL_SPECULAR, color);
      glEnable(GL_LIGHT0);
   }
   /*
   {
      GLfloat color[4] = {.4, .4, .4, 1};
      glLightfv(GL_LIGHT1, GL_DIFFUSE, color);
      glLightfv(GL_LIGHT1, GL_SPECULAR, color);
      glEnable(GL_LIGHT1);
   }{
      GLfloat color[4] = {.2, .2, .2, 1};
      glLightfv(GL_LIGHT2, GL_DIFFUSE, color);
      glLightfv(GL_LIGHT2, GL_SPECULAR, color);
      glEnable(GL_LIGHT2);
   }
   */
}

void init_material()
{
   GLfloat f_ambient[4]={0}, f_diffuse[4]={0}, f_specular[4]={0};

   f_ambient[0]=0.1;
   f_ambient[1]=0.2;
   f_ambient[2]=0.2;
   f_diffuse[0]=0;
   f_diffuse[1]=0.6;
   f_diffuse[2]=0.9;
   f_specular[0]=0.2;
   f_specular[1]=0.2;
   f_specular[2]=0.2;

   glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, f_ambient);
   glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, f_diffuse);
   glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, f_specular);
   glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 64);
}

void read_frame(int new_frame)
{
   char *filename;
   FILE *fp;
   int i;

   if(new_frame<0) return;
   filename=malloc(strlen(directory)+30);
   sprintf(filename, "%s/heightfield%04d", directory, new_frame);
   fp=fopen(filename, "rt");
   if(fp){
      int newnx, newnz;
      fscanf(fp, "%d %d", &newnx, &newnz);
      if(newnx!=nx || newnx!=nz || h==0){
         free(h); free(normals);
         nx=newnx;
         nz=newnz;
         h=(float*)malloc(nx*nz*sizeof(float));
         normals=(float*)malloc(3*nx*nz*sizeof(float));
      }
      frame=new_frame;
      for(i=0; i<nx*nz; ++i){
         fscanf(fp, "%f", h+i);
      }
      fclose(fp);
      compute_normals();
   }
   free(filename);
}

void find_reasonable_camera()
{
   float xx=nx*num_tiles;
   float zz=nz*num_tiles;
   float y0=h[0], y1=h[0];
   int i;
   for(i=1; i<nx*nz; ++i)
      if(h[i]>y1) y1=h[i];
      else if(h[i]<y0) y0=h[i];
   target[0]=0.25*xx;
   target[1]=y1+0.02*(y1-y0);
   target[2]=0.25*zz;
   camdist=0.1*sqrt(xx*xx+zz*zz);
}

void handle_display()
{
   glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

   glMatrixMode (GL_MODELVIEW);
   glLoadIdentity();
   gluLookAt(campos[0], campos[1], campos[2], target[0], target[1], target[2], 0, 1, 0);

   if(wireframe_mode) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   display_heightfields();
   if(wireframe_mode) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   if(view_target_timer>0) display_target();

   glutSwapBuffers();
}

void handle_reshape(int w, int h)
{
   width=w;
   height=h;
   glViewport(0, 0, (GLsizei)width, (GLsizei)height);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   gluPerspective(fovy, width/(float)height, nearclip, farclip);
   glutPostRedisplay();
}

void handle_keypress(unsigned char key, int x, int y)
{
   switch(key){
   case '.':
      read_frame(frame+1);
      break;
   case ',':
      read_frame(frame-1);
      break;
   case 'f':
      toggle_fullscreen();
      break;
   case 'q':
      exit(0);
   case 's':
      {
         char filename[30];
         sprintf(filename, "hw5_%04d.ppm", file_count);
         save_screen(filename);
         ++file_count;
      }
      break;
   case 'w':
      wireframe_mode=!wireframe_mode;
      break;
   default:
      return;
   }
   glutPostRedisplay();
}

void handle_click(int button, int state, int x, int y)
{
   int mods=glutGetModifiers();
   if(state==GLUT_UP) mousemode = DONOTHING;
   else if(mods&GLUT_ACTIVE_SHIFT){
      view_target_timer=1;
      mousemode=MOVETARGET;
   }else{
      if(button==GLUT_RIGHT_BUTTON || (button==GLUT_LEFT_BUTTON && mods&GLUT_ACTIVE_CTRL))
         mousemode=ZOOM;
      else if(button==GLUT_LEFT_BUTTON)
         mousemode=ROTATE;
   }
   oldmousex=x;
   oldmousey=y;
}

void handle_drag(int x, int y)
{
   switch(mousemode){
   case ROTATE:
      heading+=0.007*(oldmousex-x);
      if(heading<-M_PI) heading+=2*M_PI;
      else if(heading>M_PI) heading-=2*M_PI;
      pitch+=0.007*(oldmousey-y);
      if(pitch<-.48*M_PI) pitch=-.48*M_PI;
      else if(pitch>.48*M_PI) pitch=.48*M_PI;
      break;
   case ZOOM:
      camdist*=pow(1.01, y-oldmousey);
      nearclip=NEARCLIP_FACTOR*camdist;
      farclip=FARCLIP_FACTOR*camdist;
      break;
   case MOVETARGET:
      target[0]-=(0.002*camdist)*cos(heading)*(oldmousex-x);
      target[1]-=(0.002*camdist)*(oldmousey-y);
      target[2]-=(0.002*camdist)*sin(heading)*(oldmousex-x);
      break;
   case DONOTHING:
      break;
   }
   oldmousex=x;
   oldmousey=y;
   compute_camera_position();
   glutPostRedisplay();
}

void handle_idle()
{
   float new_time=clock()/(float)CLOCKS_PER_SEC;
   float delta_time=new_time-old_time;
   int target_turned_off=0;
   if(delta_time<0) delta_time=0; // this can happen due to roll-over
   old_time=new_time;
   if(view_target_timer>0){
      view_target_timer-=delta_time;
      if(view_target_timer<=0){
         target_turned_off=1;
         view_target_timer=0;
      }
   }
   if(target_turned_off)
      glutPostRedisplay();
}

void toggle_fullscreen()
{
   if(fullscreen_mode){
      fullscreen_mode=0;
      width=saved_width;
      height=saved_height;
      glutReshapeWindow(saved_width, saved_height);
   }else{
      fullscreen_mode=1;
      saved_width=width;
      saved_height=height;
      glutFullScreen();
   }
}

void compute_camera_position()
{
   float ch=cos(heading), sh=sin(heading), cp=cos(pitch), sp=sin(pitch);
   campos[0]=target[0]-camdist*sh*cp;
   campos[1]=target[1]-camdist*sp;
   campos[2]=target[2]-camdist*ch*cp;
}

static inline void normalize(float v[3])
{
   float overmag=1/sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
   v[0]*=overmag;
   v[1]*=overmag;
   v[2]*=overmag;
}

void compute_normals()
{
   int i, j;
   float hpx, hmx, hpz, hmz;
   for(i=0; i<nx; ++i){
      for(j=0; j<nz; ++j){
         if(i+1<nx) hpx=h[(i+1)*nz+j]; else hpx=h[j];
         if(i-1>=0) hmx=h[(i-1)*nz+j]; else hpx=h[(nx-1)*nz+j];
         if(j+1<nz) hpz=h[i*nz+j+1];   else hpz=h[i*nz];
         if(j+1>=0) hmz=h[i*nz+j-1];   else hmz=h[i*nz+nz-1];
         normals[3*(i*nz+j)]=hmx-hpx;
         normals[3*(i*nz+j)+1]=2;
         normals[3*(i*nz+j)+2]=hmz-hpz;
         normalize(normals+3*(i*nz+j));
      }
   }
}

void display_heightfields()
{
   int i, j;
   int i0, i1, j0, j1;
   glBegin(GL_TRIANGLES);
   for(i=0; i<nx*num_tiles; ++i){
      for(j=0; j<nz*num_tiles; ++j){
         i0=i%nx; i1=(i+1)%nx;
         j0=j%nz; j1=(j+1)%nz;
         glNormal3fv(normals+3*(i0*nz+j0));
         glVertex3f(i, h[i0*nz+j0], j);
         glNormal3fv(normals+3*(i1*nz+j1));
         glVertex3f(i+1, h[i1*nz+j1], j+1);
         glNormal3fv(normals+3*(i1*nz+j0));
         glVertex3f(i+1, h[i1*nz+j0], j);
         glNormal3fv(normals+3*(i0*nz+j0));
         glVertex3f(i, h[i0*nz+j0], j);
         glNormal3fv(normals+3*(i0*nz+j1));
         glVertex3f(i, h[i0*nz+j1], j+1);
         glNormal3fv(normals+3*(i1*nz+j1));
         glVertex3f(i+1, h[i1*nz+j1], j+1);
      }
   }
   glEnd();
}

void display_target()
{
   float logcd=log(camdist)/log(20.);
   float smallsize=pow(20.0, floor(logcd-.5));
   glDisable(GL_LIGHTING);
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glTranslatef(target[0], target[1], target[2]);
   glColor3f(1, 1, 1);
   glutWireCube(smallsize);
   glutWireCube(20*smallsize);
   glPopMatrix();
   glEnable(GL_LIGHTING);
}

void save_screen(const char *filename)
{
   FILE *fp;
   GLubyte *image_buffer;

   // get the pixels
   image_buffer=malloc(3*width*height*sizeof(GLubyte));
   glReadBuffer(GL_FRONT);
   glReadPixels(0, 0, width, height, GL_RGB, GL_UNSIGNED_BYTE, image_buffer);
  
   // write them to the file
   fp=fopen(filename, "wb");
   if(fp){
      int i;
      fprintf(fp, "P6\n%d %d 255\n", width, height);
      for(i=1; i<=height; ++i){
         fwrite(image_buffer+3*width*(height-i), 1, 3*width, fp);
      }
   }
   
   // clean up
   free(image_buffer);
}

