function initialConditionsTest1D()
% initialConditionsTest1D: test initial condition routines in 1 dimension.
%
%   initialConditionsTest1D (no arguments)
%
% This function (basically a script file) generates a sequence of 
%   shapes built by constructive solid geometry methods
%
% In 1D, basically all the implicit surfaces are (possibly infinite) intervals.
%   Showing the intervals themselves would be pretty dull, so we show the
%   entire implicit surface function.

% Copyright 2004 Ian M. Mitchell (mitchell@cs.ubc.ca).
% This software is used, copied and distributed under the licensing 
%   agreement contained in the file LICENSE in the top directory of 
%   the distribution.
%
% Ian Mitchell, 6/23/04

run('../addPathToKernel');

%---------------------------------------------------------------------------
% Standard output file id.
fid = 1;

%---------------------------------------------------------------------------
fprintf(fid, '\n%s %s\n%s', ...
             'A sequence of implicit surface functions in 1D', ...
             'along with their descriptions.', ...
             'Press any key after each display to move to the next');

figure;

%---------------------------------------------------------------------------
% All 1D implicit surfaces are intervals, but we should check that the
%   methods work (stupid Matlab stupid no 1D arrays).

% A 1D grid.
g1.dim = 1;
g1.min = -2;
g1.max = +2;
g1.N = 101;
g1 = processGrid(g1);

%---------------------------------------------------------------------------
% Some basic shapes.
sphere1 = shapeSphere(g1, 0.2, 0.9);
show1D(g1, sphere1);
fprintf(fid, '\nInterval centered on %f, width %f, generated by %s', ...
             0.2, 0.9, 'shapeSphere');
pause;

rectangle1 = shapeRectangleByCorners(g1);
show1D(g1, rectangle1);
fprintf(fid, '\nInterval centered on %f, width %f, generated by %s', ...
             0.5, 0.5, 'shapeRectangleByCorners');
pause;

rectangle2 = shapeRectangleByCenter(g1);
show1D(g1, rectangle2);
fprintf(fid, '\nInterval centered on %f, width %f, generated by %s', ...
             0, 0.5, 'shapeRectangleByCenter');
pause;

hyperplane1 = shapeHyperplane(g1, -1, -0.3);
show1D(g1, hyperplane1);
fprintf(fid, '\nHalfspace to the %s, starting at %f, generated by %s', ...
             'right', -0.3, 'shapeHyperplane');
pause;

%---------------------------------------------------------------------------
% Now try constructive geometry operations.
union1 = shapeUnion(rectangle1, rectangle2);
show1D(g1, union1);
fprintf(fid, '\nInterval from %f to %f, generated by %s (%s)', ...
             -0.5, 1.0, 'shapeUnion', 'NOTE: not signed distance');
pause;

intersection1 = shapeIntersection(rectangle1, rectangle2);
show1D(g1, intersection1);
fprintf(fid, '\nInterval from %f to %f, generated by %s', ...
             0.0, 0.5, 'shapeIntersection');
pause;

complement1 = shapeComplement(hyperplane1);
show1D(g1, complement1);
fprintf(fid, '\nHalfspace to the %s, starting at %f, generated by %s', ...
             'left', -0.3, 'shapeComplement');
pause;

difference1 = shapeDifference(sphere1, hyperplane1);
show1D(g1, difference1);
fprintf(fid, '\nInterval from %f to %f, generated by %s', ...
             -0.7, -0.3, 'shapeDifference');


%---------------------------------------------------------------------------
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%---------------------------------------------------------------------------
function h = show1D(g, data)
% show1D: display a 1D implicit surface function.
%
%   show1D(g, data)
%
% 1D implicit surface functions are plotted value vs state.
%   The interval that they represent are those portions of the state
%   where their value is negative.
%
% parameters:
%   g   	Grid structure (see processGrid.m for details).
%   data        Array containing the implicit surface function.

plot(g.xs{1}, data, 'b-');
grid on;
xlabel('x');  ylabel('\phi(x)');
