function hs = figureAir3D(g, data, data0, superimpose)
% figureAir3D: Generate a figure for 3D aircraft collision avoidance example.
%
%   hs = figureAir3D(g, data, data0, superimpose)
%  
% Depending on the inputs, this function will generate one or two
%   figures with three dimensional isosurfaces showing the backwards
%   reachable set for the 3D aircraft collision avoidance example 
%   (also called the game of two identical vehicles).
%
% The g, data and data0 parameters are normally generated by running
%   air3D.m first.
%
% Parameters:
%
%   g            Grid structure on which data was computed.
%   data         Implicit surface function for reachable set.
%   data0        Implicit surface function for target set (optional).
%   superimpose  Boolean specifying whether to superimpose the target and
%                  reachable sets on a single figure using transparent
%                  isosurfaces (optional, default = 0).
%
%   hs           Handles to any isosurfaces that are generated.

% Copyright 2004 Ian M. Mitchell (mitchell@cs.ubc.ca).
% This software is used, copied and distributed under the licensing 
%   agreement contained in the file LICENSE in the top directory of 
%   the distribution.
%
% Ian Mitchell, 4/18/04

%---------------------------------------------------------------------------
if(nargin < 2)
  error('g and data parameters are required');
end
if(nargin < 3)
  plotTarget = 0;
else
  plotTarget = 1;
end
if(nargin < 4)
  superimpose = 0;
end

%---------------------------------------------------------------------------
% What level set should we view?
level = 0;

% Visualize the angular dimension a little bigger.
aspectRatio = [ 1 1 0.4 ];

% A small number to avoid speckle in overlapped reachable sets.
small = 0.01;

%---------------------------------------------------------------------------
% Visualize the reachable set.

  figure;
  hR = patch(isosurface(g.xs{1}, g.xs{2}, g.xs{3}, data, level));
  %isonormals(g.xs{1}, g.xs{2}, g.xs{3}, data, hR);
  set(hR, 'FaceColor', 'red', 'EdgeColor', 'none');
  lighting phong;
  axis(g.axis);
  daspect(aspectRatio);
  view(75,10)
  camlight left;  camlight right;
  grid on;
  xlabel('x_1');  ylabel('x_2');  zlabel('x_3');

%---------------------------------------------------------------------------
% If necessary, visualize the target set.
if(plotTarget)
  if(superimpose)
    hold on;
  else
    figure;
  end

  % Offset the target set slightly so that in the superimposed case
  %   the two sets won't overlap exactly and lead to speckle.
  hT = patch(isosurface(g.xs{1}, g.xs{2}, g.xs{3}, data0 + small, level));
  %isonormals(g.xs{1}, g.xs{2}, g.xs{3}, data, hT);

  if(superimpose)
    set(hR, 'FaceAlpha', 0.75);
    set(hT, 'FaceColor', 'blue', 'EdgeColor', 'none');
  else
    set(hT, 'FaceColor', 'red', 'EdgeColor', 'none');
    lighting phong;
    axis(g.axis);
    daspect(aspectRatio);
    view(75,10)
    camlight left;
    grid on;
    xlabel('x_1');  ylabel('x_2');  zlabel('x_3');
  end
 
  hs = [ hR, hT ];

else
  hs = hR;
end
