// multibinset.cpp: implementation of the MultiBinSet class.
//
//////////////////////////////////////////////////////////////////////

#include "defines.h"

#ifdef MULTI_UNIT

#include "multibinset.h"
#include <assert.h>
#include <math.h>
#include <string.h>

// global var's
extern VECTOR_DATA_TYPE *max_array;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

// constructor
MultiBinSet::MultiBinSet(int num_bins)
{
	this->num_bins = num_bins;
	bin = new MultiBin*[num_bins];
	for (int t=0; t<num_bins; t++)
		bin[t] = NULL;
	current_bin = 0;
	num_bids = 0;
	
	// singletons -- do I still need these now that I have dynamic programming?
	singleton = new unsigned[num_bins];
	memset(singleton,0,num_bins * sizeof(singleton[0]));

	// allocate binstack
	binstack = new int[num_bins];
	binstack_pointer = -1;
}

// destructor
MultiBinSet::~MultiBinSet()
{
	for (int t=0; t<num_bins; t++)
		delete bin[t];
	delete[] bin;
	delete[] singleton;
}

void printArray(VECTOR_DATA_TYPE* array)
{
  for (unsigned i = 0; i < NUM_ITEMS; i++)
    {
      printf(" %d", array[i]);
    }
  printf ("\n");
}

// put the bids into bins
void MultiBinSet::populateBins(InputFile &infile)
{
	unsigned i,j;
	
	// reorder max_array
	VECTOR_DATA_TYPE *max_array2 = new VECTOR_DATA_TYPE[NUM_ITEMS];
	for (i=0;i<NUM_ITEMS;i++)
		max_array2[i] = max_array[infile.mapBack(i)];
	for (i=0;i<NUM_ITEMS;i++)
		max_array[i] = max_array2[i];
	//printArray(max_array);
	delete[] max_array2;

	// allocate the bins
	for (i=0; i<NUM_ITEMS; i++)
		bin[i] = new MultiBin(i,infile.getNumBids(i)*FUDGE_FACTOR,bin,max_array[i]);

	// extract bids from the input parameter and populate the lists
	for (i = 0; i < infile.num_bids; i++) 
	{
		// don't add it to a bin if it's dominated
		if (infile.bid[i]->deleted) continue;
		num_bids++;
	
		// make the bid
		Bid *temp_bid = makeBid(infile, infile.bid[i]);
		
		// construct and add slaves
		if (infile.bid[i]->num_slaves != 0)
		{
			temp_bid->slaves = new BidHistory(infile.bid[i]->num_slaves);
			for (j=0;j<infile.bid[i]->num_slaves;j++)
				temp_bid->slaves->append(makeBid(infile, infile.bid[infile.bid[i]->slaves[j]]));
		}
		this->bin[temp_bid->bin_number]->add(temp_bid,infile.bid[i]->numGoods == 1);
	}
}

// output pruning data
void MultiBinSet::gatherPruningStats()
{
#ifdef GATHER_PRUNING_STATS
	// hits
	FILE *fp = fopen("pruning_hit.csv","at");
	for (i=0;i<NUM_ITEMS;i++)
		fprintf (fp,"%d, ",this->prune_hits[i]);
	fprintf (fp,"\n");
	fclose(fp);
	
	// prunes
	fp = fopen("pruning_prune.csv","at");
	for (i=0;i<NUM_ITEMS;i++)
		fprintf (fp,"%d, ",this->prune_bin[i]);
	fprintf (fp,"\n");
	fclose(fp);
#endif
;
}

// build the pruning tables
void MultiBinSet::constructPruningTables()
{
	signed l,k;
	
	// construct the pruning table for each good, for each bin
	for (l = num_bins - 1; l >= 0; l--)  // each bin, in descending order
	{
		MultiBin *this_bin = bin[l];
		
		// make singleton vectors
		this_bin->constructSingletonArray();
		
		// copy the previous bin's lists
		for (k=l+1;k<num_bins;k++)  // this gets skipped the first time (l==num_bins-1)
			bin[l]->getPruneTable(k)->copyFrom(bin[l+1]->getPruneTable(k));

		// add each bid to all the applicable pruning tables
		this_bin->constructNewBin();
		while (this_bin->nextBid())	// each bid in bin l
		{
			k = -1;

			// for each good in the bid
			while ((k = this_bin->currentBid()->vec->lowOrderPresent(k+1)) != -1)
			{
				// add the bid to the pruning table for that good
				this_bin->getPruneTable(k)->addBid(this_bin->currentBid());
			}

			// if this bid dominates other bids
			if (this_bin->currentBid()->slaves) 
			{
				// for each bid dominated by this bid
				for (int t=0;t<this_bin->currentBid()->slaves->num;t++)
				{
					Bid *this_bid = this_bin->currentBid()->slaves->bid[t];
					k = -1;
					while ((k = this_bid->vec->lowOrderPresent(k+1)) != -1)
					{
						// add the dominated bid to the pruning table for that good
						this_bin->getPruneTable(k)->addBid(this_bid);
					}

				}
			}
		} 

		// now, add the bin's singletons too
		// in the future, I may be able to improve on just adding them all
		for (k=0;k<this_bin->getNumSubbins()-1;k++)
			this_bin->getPruneTable(l)->addBid(this_bin->getSingleton(k));
	}
}


// set up the first dynamic bin before running allocate
void MultiBinSet::makeFirstBin(Allocation *allocation,int best_amount)
{
	Bid::firstSubbin();
	bin[0]->constructNewBin(allocation,best_amount);	
}

// sort all the base bins
void MultiBinSet::sortBins()
{
#ifdef SORT_BINS
	for (int t=0;t<num_bins;t++)
		bin[t]->sort();
#endif
	return;
}

void MultiBinSet::printSingletonArrays()
{
	for (unsigned t=0;t<NUM_ITEMS;t++)
		bin[t]->printSingletonArray();
}

Bid *MultiBinSet::makeBid(InputFile &infile, InputBid *inbid)
{
	int first_good = -1, j;
	unsigned k;
	Vector *temp_vector;

	//inbid->vec->printVector();
	// read through, good by good, in the new ordering
	// add each bid to whatever bin it belongs to: the bin corresponding to the first found good
	for (k=0; k < NUM_ITEMS; k++) 
	{
		j=infile.mapBack(k); // can't put this in the for statement or there's an uninitialized memory read
		
		// if the given bid contains the given good...
		if (inbid->vec->readArray(j)) 
		{
			if (first_good == -1)
			{
				temp_vector = new Vector(NUM_ITEMS,max_array,1);
				first_good = k;
			}
			temp_vector->set(k,inbid->vec->readArray(j));
		}
	}
	//temp_vector->printVector();
	// add the new bitvector to the bin
	Bid *temp_bid = new Bid(temp_vector,inbid->amount,inbid->bid_num,first_good,inbid);
	assert (temp_bid->vec->readArray(first_good));
	
	// return the bid
	return temp_bid;
}

#endif // multi_unit
