#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>
#include "inputfile.h"
#include "stopwatch.h"
#include "qsort.h"

extern Stopwatch sw;
extern VECTOR_DATA_TYPE *max_array;

/* constructor */
InputFile::InputFile()
{
	reordered_goods = false;
	max_array = NULL;
	first_good = -1;
	num_dummy_items = 0;
}

/* destructor */
InputFile::~InputFile()
{
	delete[] filename;
	for (unsigned int t=0; t<num_bids; t++)
		delete bid[t];
	delete[] bid;

	if (reordered_goods)
	{
		delete[] listMap;
		delete[] listMapBack;
		delete[] bidcount_temp;
		delete[] bidlength_temp;
	}

	if (max_array) delete[] max_array;
}

/* opens an input file, and leaves InputFile::fp pointing to the open file. */
bool InputFile::open(char *filename)
{
	this->filename = new char[strlen(filename)+1];
	strcpy(this->filename, filename);
	fp = fopen(filename, "rt");
	if (fp == NULL)
	{
		printf ("Error opening file \"%s\"\n",filename);
		return false;
	}
	return true;
}

/* closes an input file */
void InputFile::close()
{
	fclose(fp);
}

/* write a CPLEX input file */
// assumes that bid numbers are unique
bool InputFile::writeCPLEXFile(char *filename)
{
	FILE *fp2;
	unsigned i;

	fp2 = fopen(filename, "wt");
	if (fp == NULL)
	{
		printf ("Error writing CPLEX file \"%s\"\n",filename);
		return false;
	}

	// first section: maximize revenue
	fprintf (fp2, "max\n\n");
	for (i=0; i<num_bids; i++)
	{
		fprintf (fp2, "%dx%d ", bid[i]->amount, bid[i]->bid_num);
		if (i < num_bids - 1) fprintf (fp2, "+ ");
		if (i%10 == 9) fprintf(fp2,"\n");
	}

	// second section: constraints
	fprintf (fp2,"\n\nst\n\n");
	for (i=0; i<num_items; i++)
	{
		bool make_a_plus = false;
		int printed = 0;
		for (unsigned j=0; j<num_bids; j++)
		{
			// list each bid that names the current good
			if (bid[j]->vec->readArray(i))
			{
				if (make_a_plus) fprintf(fp2,"+ ");
				#ifdef MULTI_UNIT
					fprintf(fp2,"%dx%d ",bid[j]->vec->readArray(i),bid[j]->bid_num);
				#else
					fprintf(fp2,"x%d ",bid[j]->bid_num);
				#endif
				make_a_plus = true;
				if (printed++%10 == 9) fprintf(fp2,"\n");
			}
		}

		// finish off the constraint
		#ifdef MULTI_UNIT
			fprintf (fp2, " <= %d\n\n",max_array[i]);
		#else
			fprintf (fp2, " <= 1\n\n");
		#endif
	}

	// third section: integrality
	fprintf (fp2, "integer\n\n");
	for (i=0; i<num_bids; i++)
	{
		fprintf (fp2, "x%d ", bid[i]->bid_num);
		if (i%10 == 9) fprintf(fp2,"\n");
	}

	fprintf (fp2,"\n\nend\n");
	fclose(fp2);
	return true;
}

/* do everything in one command */
bool InputFile::parse (char *filename)
{
	assert (!this->reordered_goods);
	if (!this->open (filename))
		return false;
	this->parse();
	this->close();
	return true;
}


/* parses the input file into a bunch of internal variables */
void InputFile::parse()
{
	unsigned int temp,t;
	signed int foundGood;
	
	// read number of goods
	fileRead ("goods",&num_items);
	
	// read number of bids
	fileRead ("bids", &num_bids);

	// read number of dummy goods
	fileRead ("dummy", &num_dummy_items);

	num_items += num_dummy_items; // NOTE: num_items is *all* the items

	// allocate 'array' and 'bidcount'
	unsigned *array = new unsigned[num_items];
	bidcount = new unsigned[num_items];
	memset(bidcount,0,sizeof(unsigned)*num_items);

	
	bid = new InputBid*[num_bids + num_items];  // + num_items so that we don't have to reallocate memory to add singleton bids later
	
	// read maximums
	#ifdef MULTI_UNIT
		max_array = new VECTOR_DATA_TYPE[num_items];
		#ifdef ASSUME_SINGLE_UNITS
			for (t=0;t<num_items;t++)
				max_array[t] = SINGLE_UNITS_MAX;
		#else
			fileRead("maximums",&temp);
			max_array[0] = (VECTOR_DATA_TYPE) temp;
			for (t=1;t<num_items;t++)
				fileRead(max_array[t]);
		#endif
	#else
		// make sure it's not a multi-unit file
		long pos = ftell(fp);
		skipComments();
		fscanf(fp, "%s", buffer);
		if (!stricmp(buffer,"maximums"))
		{
			printf("Error reading input file.\nThis is a multi-unit file, and CASS is compiled in single-unit mode.\n");
			exit(1);
		}
		fseek(fp,pos,SEEK_SET);
	#endif

	// read bids
	for (t=0; t<num_bids; t++)
	{
		// create a new bid
		bid[t] = new InputBid(NULL, 0, 0);
		unsigned arrayindex = 0;
		
		// read bid number
		fileRead(&temp);
		bid[t]->bid_num = temp;
		
		// read bid price
		fileRead(&bid[t]->amount);
		
		// allocate and initialize goods array
		bid[t]->vec = new VECTOR(num_items, max_array, 1);
		
		// read goods until end-of-record encountered
		foundGood = 0;
		unsigned tt;
		for (tt=0;foundGood != -1;tt++)
		{
			foundGood = fileReadGood(bid[t]->bid_num);
			if (foundGood != -1)
			{
				#ifdef MULTI_UNIT
					#ifdef ASSUME_SINGLE_UNITS
						temp = 1;
					#else
						fileRead(&temp);
					#endif
					bid[t]->vec->set(foundGood,temp);
				#else
					bid[t]->vec->setBit(foundGood);
				#endif
				array[arrayindex++] = foundGood;
			}
		}

		// copy the array to bid
		bid[t]->array = new unsigned[arrayindex];
		for (tt=0; tt<arrayindex; tt++)
			bid[t]->array[tt] = array[tt];
		bid[t]->numGoods = arrayindex;
	}

	delete[] array;
}

/* skip over comments */
void InputFile::skipComments()
{
	int pos;
	unsigned t;
	char c;

	do
	{
		pos = ftell(fp);
		fgets (buffer, 1000, fp);
		for (t=0,c=' ';t<strlen(buffer) && isspace(c);t++)
			c=buffer[t];  // allow comments to follow whitespace
	} while (!feof(fp) && (c == '%' || isAllWhitespace(buffer,strlen(buffer))));
	fseek (fp,pos,SEEK_SET);
}
	
bool InputFile::isAllWhitespace(char *input,int len)
{
	for (int t=0; t<len; t++)
		if (!isspace(input[t])) return false;
	return true;
}

/* read an integer from the file */
void InputFile::fileRead (unsigned int *int_input)
{
	skipComments();
	fscanf(fp, "%d", int_input);
}

/* read a float from the file */
void InputFile::fileRead (float *float_input)
{
	skipComments();
	fscanf(fp, "%f", float_input);
}

/* read a vectorDataType from the file */
void InputFile::fileRead (VECTOR_DATA_TYPE &vdt_input)
{
	skipComments();
	unsigned tempint;
	fscanf(fp, "%d", &tempint);
	vdt_input = tempint;
}

/* read a label followed by an integer from the file */
void InputFile::fileRead (const char *label, unsigned int *int_input)
{
	skipComments();
	fscanf(fp, "%s", buffer);
	if (stricmp(buffer,label))
	{
		printf("Error reading input file %s.\nExpected label %s, found %s.\n",filename, label, buffer);
		exit(1);
	}
	fscanf(fp, "%d", int_input);
}

/* read a good, unless '#' is the first thing found */
signed int InputFile::fileReadGood (unsigned bidnum)
{
	unsigned goodnum;
	
	skipComments();

	// read input
	fscanf(fp, "%s", buffer);

	// we found a pound sign -- end of record
	if (buffer[0] == '#')
		return -1;

	// we found a good number
	sscanf(buffer, "%d", &goodnum);
	if (goodnum >= num_items)
	{
		printf ("Good #%d was named in bid %d, but %d is the largest possible good #.\n",
			goodnum,bidnum,num_items);
		exit(1);
	}
	return goodnum;
}

/* output what has been read, after the read is complete */
void InputFile::print(int maxrec)
{
	printf ("Goods: %d, DGoods: %d, Bids: %d\n",num_items, num_dummy_items, num_bids);
	if (maxrec == -1) maxrec = num_bids;
	
	for (unsigned t=0; t<(unsigned) maxrec; t++)
	{
		printf ("Bid#: %d, Amount: %d, Vec: ",bid[t]->bid_num, bid[t]->amount);
		bid[t]->vec->printVector();
	}
}

extern double randNeg();
extern double randFactor(double variance);
extern double good_rand_factor;

/* construct "listMap" and "listMapBack", which order goods */
void InputFile::orderGoods()
{
	unsigned int l, i,j;

	listMap = new unsigned int[num_items];
	listMapBack = new unsigned int[num_items];
	bidcount_temp = new unsigned int[num_items];
	bidlength_temp = new unsigned int[num_items];

	// initialize
	for (i=0;i<num_items;i++)
	{
		listMap[i] = (unsigned) -1;
		listMapBack[i] = (unsigned) -1;
	}

	/* determine good ordering */
//	for (l = 0; l < num_items ; l++)  // each level
	for (l = 0; l < num_items - num_dummy_items; l++)  // each level
	{
		memset(bidcount_temp,0,sizeof(bidcount_temp[0])*num_items);
		memset(bidlength_temp,0,sizeof(bidlength_temp[0])*num_items);

		for (i = 0; i < num_bids; i++) // each bid
		{
			assert (bid[i]->numGoods <= num_items);
			if (bid[i]->deleted) continue;

			for (j = 0; j < bid[i]->numGoods; j++) // each good
			{
				if (listMap[bid[i]->array[j]] < (unsigned)l) // the bid contains a conflicting item
					goto conflict;
			}

				// if we reach this point (we don't take the goto above) then this bid is non-conflicting
			for (j = 0; j<bid[i]->numGoods; j++) // each good in this bid, again
			{
				// keep track of number of bids and length of bids
				bidcount_temp[bid[i]->array[j]]++;
				bidlength_temp[bid[i]->array[j]] += bid[i]->vec->getNumSet();
			}

			// the  bid contained a conflicting item--go to the next bid
			conflict:  ; 
		}
		
		// record the new numbering of goods
		int index = 0;
		double bestscore = (double)LARGE_NUMBER;
		for (unsigned t=0;t<num_items-num_dummy_items;t++)
		{
			if (!bidcount_temp[t]) continue;
			#ifdef MULTI_UNIT
				double score = (double)(bidcount_temp[t] * max_array[t])/((double)bidlength_temp[t]/(double)bidcount_temp[t]);
			#else
				double score = 1.0/((double)bidcount_temp[t]*(double)bidlength_temp[t]/(double)bidcount_temp[t]);
//				double score = (double)bidcount_temp[t]/((double)bidlength_temp[t]/(double)bidcount_temp[t]);
//				double score = (double)bidcount_temp[t];
			#endif
			#ifdef RANDOMIZE
				score *= randFactor(good_rand_factor);
			#endif
			if (score < bestscore)
			{
				bestscore = score;
				index = t;
			}
		}
			
		#ifdef RANDOMIZE_FIRST_BIN_CONSTANT
				if (first_good != -1 && l == 0) index = first_good; 
		#endif
		listMap[index] = l;
		listMapBack[l] = index;
//		bidcount[index] = bidcount_temp[index];  // count the number of bids in each layer
		bidcount[l] = bidcount_temp[index];  // count the number of bids in each layer
	}


//#ifdef NOTHING
	// dummies
	for (l = num_items-num_dummy_items; l < num_items; l++)  // each level
	{
		listMap[l] = l;
		listMapBack[l] = l;
//		bidcount[index] = bidcount_temp[index];  // count the number of bids in each layer
		bidcount[l] = 1;  // count the number of bids in each layer
	}
//#endif

	reordered_goods = true;
	first_good = listMapBack[0];
}

// qsort compare function for scramble function
int InputFile::scrambleCompare(const void *in1, const void *in2)
{
	return (rand()-(RAND_MAX/2) > 0 ? 1 : -1);
}

// scramble an array
void InputFile::scramble(unsigned int *input, int size)
{
	for (int t=0;t<20;t++)
		Qsort(input,size,sizeof(unsigned int),scrambleCompare);
}

/* construct "listMap" and "listMapBack", which order goods */
void InputFile::randomlyOrderGoods()
{
	unsigned int i,j, lowest;

	listMap = new unsigned int[num_items];
	listMapBack = new unsigned int[num_items];
	bidcount_temp = new unsigned int[num_items];
	bidlength_temp = new unsigned int[num_items];

	// produce random ordering
	for (i=0;i<num_items;i++)
		listMap[i] = i;
	scramble (listMap,num_items);

	// make sure that the first good is first_good
	#ifdef RANDOMIZE_FIRST_BIN_CONSTANT
		int now_zero;
		for (i=0;i<num_items;i++)
		{
			if (listMap[i] == 0)
			{
				now_zero = i;
				break;
			}
		}
		listMap[now_zero] = listMap[first_good];
		listMap[first_good] = 0;
	#endif

	for (i=0;i<num_items;i++)
		listMapBack[listMap[i]] = i;
			
	// count goods in each bin
	for (i = 0; i < num_bids; i++) // each bid
	{
		if (bid[i]->deleted) continue;
		assert (bid[i]->numGoods <= num_items);
		
		lowest = num_items + 10;

		for (j = 0; j < bid[i]->numGoods; j++) // each good
			lowest = min(lowest,listMap[bid[i]->array[j]]);
		bidcount[lowest]++;   // note: BidCount is stored in the NEW ordering
	}
	reordered_goods = true;
}


int InputFile::deleteDominatedBids(unsigned t,bool singleton_cleared)
{
	// check for dominated bids
	for (unsigned tt=(singleton_cleared ? 0 : t+1); tt<num_bids; tt++)
	{
		// skip deleted bids
		if (bid[t]->deleted) continue;
		#ifdef MULTI_UNIT
			if (bid[tt]->numGoods == 1) continue;
		#endif

		// don't check a bid against itself
		if (tt == t) continue;

		// amount(tt) is more, tt subset_of t
		if (bid[tt]->amount >= bid[t]->amount && bid[tt]->vec->subsetequalof(bid[t]->vec))
		{
			#ifdef MULTI_UNIT
				appendDominatedBid(tt,t);
				num_killed++;
			#else
				removeBid(t);
				//t--;
				num_killed++;
				//goto next_bid;  // because we just deleted bid t--now we have to compare it to everything.
				return -1; // instead of the commented goto
			#endif
		}

		// amount(t) is more, t subset_of tt
		if (bid[t]->amount >= bid[tt]->amount && bid[t]->vec->subsetequalof(bid[tt]->vec))
		{
			#ifdef MULTI_UNIT
				appendDominatedBid(t,tt);
			#else
				removeBid(tt);
				tt--; // check the new bid that has been substituted in for tt
			#endif
			num_killed++;
		}
#ifdef NOTHING
		// amount(tt) is more, tt equal to t
		if (bid[tt]->vec->equalTo(bid[t]->vec))
		{
			if (bid[tt]->amount > bid[t]->amount)
				appendDominatedBid(tt,t);
		
			// amount(t) is more or equal, t equal to tt
			else
				appendDominatedBid(t,tt);

			num_killed++;
		}
#endif
	}
	
	return 0;
}


// go through the list of bids and clear dummy goods that are only referenced in a single bid.
// if such bids are found, delete any bids that this bid now dominates, or delete this bid if it is dominated.
bool InputFile::removeSingletonDummyGoods()
{
	int dummy;
	bool pair, got_one = false;
	
	// look at each bid
	for (unsigned t=0;t<num_bids-1;t++)
	{
		if (bid[t]->deleted) continue;

		do
		{
			dummy = bid[t]->vec->lowOrderPresent(num_items);

			if (dummy != -1)
			{
				pair = false;
				
				for (unsigned tt = t+1;!pair && tt<num_bids;tt++)
				{
					if (bid[tt]->deleted) continue;

					int dummy2 = bid[tt]->vec->lowOrderPresent(num_items);
					if (dummy2 == dummy)
						pair = true;
				}

				if (!pair) 
				{
					bid[t]->vec->clearBit(dummy);
					int old_num_killed = num_killed;
					deleteDominatedBids(t,true);  // do we have to decrement t here?
					if (num_killed > old_num_killed) got_one = true;
				}
			}
		} while (dummy != -1);
	}

	return got_one;
}


// add singleton bids, and remove dominated bids
void InputFile::refine(bool dom_check)
{
	bool *singletons = new bool[num_items];
	unsigned t, num_singletons = 0;
	
	num_killed=0;

	// output
#ifdef REFINE_OUTPUT
	printf ("Refining bids...");
#endif

	// initialize singletons array
	#ifndef MULTI_UNIT
		for (t=0; t<num_items; t++)
			singletons[t] = false;
	#endif

	// first, delete dominated bids once
	// consider each bid
	for (t=0; t<num_bids-1; t++)
	{
		// skip deleted bids
		if (bid[t]->deleted) continue;
		#ifdef MULTI_UNIT
			if (bid[t]->numGoods == 1) continue;
		#endif

		if (dom_check) t += deleteDominatedBids(t,false);
	}

	// delete bids over and over as long as dummies are removed
	while (removeSingletonDummyGoods());

	// consider each bid
	for (t=0; t<num_bids-1; t++)
	{
		// skip deleted bids
		if (bid[t]->deleted) continue;
		#ifdef MULTI_UNIT
			if (bid[t]->numGoods == 1) continue;
		#endif
	
		// keep track of singletons
		#ifndef MULTI_UNIT
			if (bid[t]->numGoods == 1)
			{
				unsigned singletonindex = bid[t]->array[0];
				if (!singletons[singletonindex])
				{
					singletons[singletonindex] = true;
					num_singletons++;
				}
			}

		//next_bid: ;

		#endif
	}

	// if singleton bids must be added
	// NOTE: singletons don't have to be added for dummies IFF these goods are ordered at the *end*
	#ifdef NOTHING
		if (num_singletons < num_items - num_dummy_items)
		{
			for (t=0; t<num_items - num_dummy_items; t++)
			{
				if (!singletons[t])
					createSingletonBid(t);
			}
		}

		// free memory
		delete[] singletons;
	#endif
	#ifndef MULTI_UNIT
		if (num_singletons < num_items)
		{
			for (t=0; t<num_items; t++)
			{
				if (!singletons[t])
					createSingletonBid(t);
			}
		}

		// free memory
		delete[] singletons;
	#endif

	// output
#ifdef REFINE_OUTPUT
	printf ("deleted %u bid%s",num_killed, (num_killed==1?"":"s"));
	#ifndef MULTI_UNIT
		printf (", added %u singleton%s",num_items-num_dummy_items-num_singletons,(num_items-num_dummy_items-num_singletons==1?"":"s"));
	#endif
	printf (" (%lf)\n",sw.Lap());
#endif
}

// remove the bid 'index'
void InputFile::removeBid(int index)
{
	// error check
	if (!num_bids)
	{
		printf ("Error: InputFile::removeBid tried to remove a bid from an empty list!\n");
		return;
	}

	// remove the bid
	num_bids--;
	delete bid[index];
	bid[index] = bid[num_bids];  // if these are equal, so what?
	bid[num_bids] = NULL;
}

// create a bid of length 1, only good "index" allocated, price 0
void InputFile::createSingletonBid(int index)
{
#ifndef MULTI_UNIT
	unsigned b = num_bids++;
	bid[b] = new InputBid(new VECTOR(num_items, max_array,1),0,-1);
	#ifdef MULTI_UNIT
		bid[b]->vec->set(index,1);
	#else
		bid[b]->vec->setBit(index);
	#endif
	bid[b]->numGoods = 1;
	bid[b]->array = new unsigned[1];
	bid[b]->array[0] = index;
#endif
}

int InputFile::getNumBids(int index)
{
	if (!reordered_goods)
	{
		printf ("InputFile::getNumBids error: Goods have not been reordered yet!\n");
		return -1;
	}
	return bidcount[index];
}



// bidcount is zero for every index except 0!!




int InputFile::mapBack(int index)
{
	if (!reordered_goods)
	{
		return index;
	}
	return listMapBack[index];
}

int InputFile::map(int index)
{
	if (!reordered_goods)
	{
		return index;
	}
	return listMap[index];
}

// master dominates slave.  Add slave to master's list
void InputFile::appendDominatedBid(int master, int slave)
{
	// slaves array is not yet allocated
	if (bid[master]->num_slaves == 0)
	{
		bid[master]->slaves = new unsigned[NUM_SLAVES];
		bid[master]->max_num_slaves = NUM_SLAVES;
	}

	// slaves array is full--resize and copy over
	if (bid[master]->num_slaves + bid[slave]->num_slaves + 1 > bid[master]->max_num_slaves)
	{
		unsigned *slaves2 = new unsigned[NUM_SLAVES + bid[slave]->num_slaves + 1 + bid[master]->num_slaves];
		bid[master]->max_num_slaves = NUM_SLAVES + bid[slave]->num_slaves + 1 + bid[master]->num_slaves;
		memcpy(slaves2,bid[master]->slaves,bid[master]->num_slaves*sizeof(unsigned));
		delete[] bid[master]->slaves;
		bid[master]->slaves = slaves2;
	}

	// add this bid to the slaves array
	bid[master]->slaves[bid[master]->num_slaves] = slave;
	bid[master]->num_slaves++;

	// copy over the slave's slaves, if any
	for (unsigned t=0;t<bid[slave]->num_slaves;t++)
	{
		bid[master]->slaves[bid[master]->num_slaves] = bid[slave]->slaves[t];
		bid[master]->num_slaves++;
	}

	// delete the slave
	bid[slave]->deleted = true;
}

bool InputFile::reorderedGoods()
{
	return this->reordered_goods;
}
