//////////////////////////////////////
/* The class that does all the work */
/*
*///////////////////////////////////////

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "defines.h"
#include "cass.h"
#include "cacheitem.h"
#include "linkedlist.h"
#include "inputfile.h"
#include "qsort.h"
#include "stats.h"
#include "vector_include.h"

/* global variables */
unsigned int NUM_ITEMS, NUM_UNITS, NUM_DUMMY_ITEMS;
VECTOR_DATA_TYPE *max_array;
Stopwatch sw;
Stats *stats;

/* constructor */
Cass::Cass(InputFile *infile, int cache_size)
{
	// create max_array
	NUM_ITEMS = infile->num_items;
	NUM_DUMMY_ITEMS = infile->num_dummy_items;
#ifdef MULTI_UNIT
	NUM_UNITS=0;
	for (unsigned t=0;t<NUM_ITEMS;t++)
		NUM_UNITS+=max_array[t];
#else 
	NUM_UNITS = NUM_ITEMS;
#endif

	// normal variables
	main_path = new BidHistory(NUM_UNITS*FUDGE_FACTOR);
	progress=101;
	best = new Allocation(new VECTOR(NUM_ITEMS,max_array,1), 0);
	best->path = new BidHistory(NUM_UNITS);
	stats = new Stats();
	already_searched_ctr = 0;
	abort_time = 0;
	usedTimeLimit = false;

	// propagate_amounts
	#ifdef PROPAGATE_AMOUNTS
		propagate_unallocated = new int[NUM_ITEMS];
	#endif

	// error check
	#ifdef _DEBUG
		if (NUM_UNITS > (VECTOR_COUNTING_TYPE) -1)
			printf ("VECTOR_COUNTING_TYPE must be increased for NUM_UNITS=%d\n",NUM_UNITS);
	#endif

	// cache
	#ifdef CACHING_ENABLED
		cache = new Cache(cache_size,best->vec->getIntNum());
	#endif
}

/* destructor */
Cass::~Cass()
{
	// normal variables
	delete main_path;
	delete best;
	delete stats;
	delete binset;

	// propagate_amounts
	#ifdef PROPAGATE_AMOUNTS
		delete[] propagate_unallocated;
	#endif
	
	// caching
	#ifdef CACHING_ENABLED
		if (cache->getLargePrime()) 
			delete cache;
	#endif
}

bool no_progress_before_abort;
double bin_rand_factor=0,good_rand_factor=0;

// Set up and run the CASS algorithm, given the parameters in param
int Cass::run (InputFile& infile) 
{
	// set up variables
	double abort_increment = ABORT_INCREMENT;
	int calls = LARGE_NUMBER,t;
	double num_bids[NUM_BIDS_HORIZON];
	bool max_random_cutoff = false;
	
	// setup
	srand(17);
	sw.Start();
	Allocation *allocation = new Allocation(new VECTOR(NUM_ITEMS,max_array,1), 0);
	for (t=0; t<NUM_BIDS_HORIZON;t++)
		num_bids[t] = LARGE_NUMBER;

	// process the incoming bids
	#ifdef REFINE_ENABLED
		infile.refine();
	#endif

// debug
#ifdef PRINT_AFTER_REFINE
		infile.print();
#endif
	
		
	double k = ((log(ABORT_FIRST)/log(2))/sqrt(NUM_ITEMS));

	best->amount = 0;  	// maybe replace this with a lower bound -- NOTE: see below, I've added one now.

	// this loop only executes more than once in randomized mode
	do
	{
		aborted_during_search = false;
		#ifdef RANDOMIZE
			double randomization_factor;
			
			// prepare to search
			no_progress_before_abort = true;
			calls = (int)pow(2,k * sqrt(NUM_ITEMS));
			if (calls >= RANDOMIZATION_LIMIT || max_random_cutoff) 
			  calls = LARGE_NUMBER;
			abort_time = calls;
			randomization_factor = 1.0 - (1.0 / RANDOMIZATION_LIMIT) * calls;
			if (randomization_factor < 0.0) randomization_factor = 0.0;
			bin_rand_factor = BIN_RAND_FACTOR * randomization_factor;
			good_rand_factor = GOOD_RAND_FACTOR * randomization_factor;

			if (!infile.reorderedGoods())
			{
				infile.orderGoods();
//				already_searched = new unsigned[infile.getNumBids(0)]; // the number of goods in bin 1 will always be fixed
			}
			else
				#ifdef RANDOMIZE_GOOD_ORDERING
					infile.randomlyOrderGoods();
				#else
					infile.orderGoods();
				#endif

		#else
			infile.orderGoods();
		#endif

		// construct the set of bins
		binset = new BINSET(NUM_ITEMS);
		binset->populateBins(infile);
		binset->constructPruningTables();

		//if (num_bids_avg == -1)	num_bids_avg = binset->num_bids + 10;

		#ifdef PRINT_SINGLETONS
			binset->printSingletonArrays();
		#endif
		binset->sortBins();
		binset->makeFirstBin(allocation,this->best->amount);
		
		// lower bound:
		#ifdef MULTI_UNIT
#ifdef DISABLED
			Allocation *temp_hist = new Allocation(NULL, 0);
			temp_hist->path = new BidHistory(NUM_ITEMS+1);
			for (t =0; t<(signed) NUM_ITEMS; t++)
			{
				temp_hist->path->append(this->binset->bin[t]->singleton[max_array[t]-1]);
				temp_hist->amount += this->binset->bin[t]->singleton[max_array[t]-1]->amount;
			}
			if (temp_hist->amount > best->amount)
			{
				delete best;
				best = temp_hist;
			}
			else delete temp_hist;
#endif
		#endif

		// set the lower bound for the decision version
		#ifdef DECISION_PROBLEM
			best->amount = decision_cutoff;
		#endif

		printf ("Starting to search (%lf) - %d calls, %d bids\n",sw.Lap(), abort_time, binset->num_bids);

		// do the search
		this->allocate(allocation);

		// clean up
		if (aborted_during_search)
		{
			progress = 0;
			#ifdef CACHING_ENABLED
				cache->reset();
			#endif
			main_path->reset();
			delete binset;
			allocation->amount = 0;
			allocation->vec->reset();
			if (allocation->path)
				allocation->path->reset();

			for (int t=NUM_BIDS_HORIZON-1; t>0;t--)
				num_bids[t] = num_bids[t-1];
			num_bids[0] = binset->num_bids;
			if (num_bids[NUM_BIDS_HORIZON-1] - num_bids[0] < MIN_NUM_BIDS_CHANGE) 
			{
				printf ("Old: %lf, new: %lf -- doing max search\n",num_bids[NUM_BIDS_HORIZON-1],num_bids[0]);
				max_random_cutoff = true;
			}

			else if (num_bids[NUM_BIDS_INCREASE_HORIZON-1] - num_bids[0] < MIN_NUM_BIDS_CHANGE)
			{
				abort_increment *= ABORT_INCREASE_FACTOR;
				printf ("no progress: increment now %f\n",abort_increment);
			}
//			if (no_progress_before_abort && calls >= MIN_CALLS_BEFORE_ABORT_INCREASE)
//			{
//				abort_increment *= ABORT_INCREASE_FACTOR;
//				printf ("no progress: increment now %f\n",abort_increment);
//			}

			k += abort_increment;
		}
	} while (aborted_during_search);
	
	// done searching
	stats->time = sw.Lap();
	printf ("---- Finished ----\n");
	stats->outputAnytimeStats(infile);
	delete allocation;

	return 1;
}

// output the list of winning bids, and do some error checking
void Cass::reportWinningBids(char *filename)
{
	FILE *fp=NULL;
	VECTOR *vec=NULL;
	unsigned revenue, *bids=NULL, bids_index=0;

	#ifdef WINNING_BIDS_FILE
		fp = fopen (filename, "wt");
	#endif
	
	printf("Winning bids (%d):",best->path->num);

	#ifdef _DEBUG
		revenue = 0;
		vec = new VECTOR(NUM_ITEMS,max_array,1);
		bids = new unsigned[best->path->num * 3];
		bids_index=0;
	#endif

	// each bid
	revenue = reportWinningBids(fp, best->path,vec,bids_index,bids);

	printf ("\n");
	#ifdef _DEBUG
		// check revenue
	        assert (revenue == best->amount);
		
		// check non-repetition
		for (unsigned t=0;t<bids_index;t++)
		{
			for (unsigned tt=t+1;tt<bids_index;tt++)
				assert (bids[t] != bids[tt]);
		}

		printf ("  ---> verified: bids disjunct and non-repeating, $ %d, %d units\n",revenue,vec->getNumSet());
		delete vec;
		delete[] bids;
	#endif
	#ifdef WINNING_BIDS_FILE
		fclose(fp);
	#endif
}

// internal function used by the overloaded reportWinningBids()
unsigned Cass::reportWinningBids(FILE *fp,BidHistory *history, VECTOR *vec,unsigned &bids_index,unsigned *bids)
{
	unsigned revenue = 0;
	
	// each bid
	for (int t=0;t<history->num;t++)
	{
		// don't output bids that were created internally
		if (history->bid[t]->bid_num == -2)
			revenue += reportWinningBids(fp,history->bid[t]->bid_array,vec,bids_index,bids);
		
		else 
		{
			#ifdef _DEBUG
				revenue += history->bid[t]->amount;
				if (history->bid[t]->bid_num >=0)
				   bids[bids_index++]=history->bid[t]->bid_num;
				if
				  (vec->disjunctFrom(history->bid[t]->vec,0))
				  {
					vec->unionWith(history->bid[t]->vec);
					
				  }
				else
				{
					printf ("\nCass::reportWinningBids -- winning bids are not disjunct!\n");
					exit(1);
				}
			#endif
			if (history->bid[t]->bid_num != -1)
			{
				printf (" %d",history->bid[t]->bid_num);
				#ifdef WINNING_BIDS_FILE
					fprintf (fp,"%d%s",best->path->bid[t]->bid_num,NEWLINE);
				#endif
			}
		}
	}

	return revenue;
}

	
void Cass::outputResults(char *filename)
{
	if(usedTimeLimit) {
		printf("--- TIME LIMIT REACHED ---\n");
	}
	printf ("Revenue:  %d\n",best->amount);
	printf ("Time:     %f\n",stats->time);
	#ifdef GATHER_STATS
		printf ("Examined: %d\n",stats->num_examined);
	#ifdef CACHING_ENABLED
	if (cache->getLargePrime())
	{
		printf ("Cache hit: %d\n",stats->num_cache);
		printf ("Cache usage: %d%% (%d of %d)\n",(cache->getCacheUsage()*100)/cache->getLargePrime(),cache->getCacheUsage(),cache->getLargePrime());
		printf ("Total recursive calls: %d\n",stats->num_recursive_calls);
	}
	#ifdef PROPAGATE_AMOUNTS
		printf ("Propagation stores: %d\n",stats->num_propagated);
	#endif
	#endif

	#ifdef COUNT_MIS
		//long unsigned int high = (long unsigned int) (stats->mis_sum >> 32);
		//long unsigned int low = (long unsigned int) ((stats->mis_sum << 32)>>32);
//		double test = (double) (long unsigned int) (stats->mis_sum >> 32) * (double)4294967296L + (double)(long unsigned int) ((stats->mis_sum << 32)>>32);
//		printf ("test: %lf\n",(double) (long unsigned int) (stats->mis_sum >> 32) * (double)4294967296L + (double)(long unsigned int) ((stats->mis_sum << 32)>>32));

		//double mean=_64to2(stats->mis_sum)/_64to2(stats->mis_num);
		//double stdev=sqrt(_64to2(stats->mis_sum_sq) / _64to2(stats->mis_num) - _64to2(stats->mis_sum)/_64to2(stats->mis_num));

		//printf("test:%lf   %lf\n",mean, stdev);

		printf ("MIS Max/Min/Num/Mean/StdDev: (%d,%d,%llu,",
			stats->mis_max,stats->mis_min,stats->mis_num);
		printf( "%lf,%lf)\n",_64to2(stats->mis_sum)/_64to2(stats->mis_num),
			sqrt(_64to2(stats->mis_sum_sq) / _64to2(stats->mis_num) - _64to2(stats->mis_sum)/_64to2(stats->mis_num) * _64to2(stats->mis_sum)/_64to2(stats->mis_num)));
	#endif

	#ifdef PRUNING_ENABLED
		if (stats->num_prune_billions)
			printf ("Prune: %d%09d\n",stats->num_prune_billions,stats->num_prune);
		else
			printf ("Prune: %d\n",stats->num_prune);
	#endif
	#endif

	// Bids to take
	#ifndef RANDOMIZE
		reportWinningBids(filename);
	#endif
}

