//BitVector2 -- Inline functions from Bitvector

#ifndef BITVECTOR2
#define BITVECTOR2

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "bitvector.h"
#include "defines.h"

/* duplicate */
INLINE BitVector *BitVector::duplicate()
{
	BitVector *temp = new BitVector(num,NULL,0);
	copyTo(temp);
	return temp;
}

/* copyTo */
INLINE void BitVector::copyTo(BitVector *dest)
{
#if (MAX_NUM_GOODS > INT_BITS)
	memcpy(dest->array,this->array, int_num*sizeof(VECTOR_DATA_TYPE));
#else
	dest->array = this->array;
#endif
	dest->num_set = this->num_set;
}

/* UnionWith */
INLINE BitVector *BitVector::unionWith(BitVector *new_array, BitVector *list2)
{
	/* this function operates on integers instead of characters so that it is four times faster */
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
		new_array->array[t] = array[t] | list2->array[t]; // logical OR
#else
	new_array->array = array | list2->array; // logical OR
#endif

	// update num_set
#ifdef BV_ASSUME_DISJUNCT
	new_array->num_set = this->num_set + list2->num_set;
#else
	new_array->num_set = -1;
#endif
	return new_array;
}

/* UnionWithSelf */
INLINE void BitVector::unionWith(BitVector *list2)
{
	/* this function operates on integers instead of characters so that it is four times faster */
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
		array[t] = array[t] | list2->array[t]; // logical OR
#else
	array = array | list2->array; // logical OR
#endif

	// update num_set
#ifdef BV_ASSUME_DISJUNCT
	num_set = this->num_set + list2->num_set;
#else
	num_set = -1;
#endif
}

/* intersect one bitvector with another */
INLINE void BitVector::intersectWith(BitVector *input)
{
	/* this function operates on integers instead of characters so that it is four times faster */
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
		array[t] = array[t] & input->array[t]; // logical OR
#else
	array = array & input->array; // logical OR
#endif

	// update num_set
#ifdef BV_ASSUME_DISJUNCT
	num_set += input->num_set;
#else
	num_set = -1;
#endif
}

/* intersect one bitvector with another */
/* WARNING: num_set is corrupted by this function */
INLINE void BitVector::intersectWithNot(BitVector *input)
{
	/* this function operates on integers instead of characters so that it is four times faster */
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
		array[t] = array[t] & ~(input->array[t]); // logical OR
#else
	array = array & ~(input->array); // logical OR
#endif

	// update num_set -- this is a weird function, so always reset this.
	num_set = -1;
}

/* disjunctFrom */
INLINE bool BitVector::disjunctFrom(BitVector *list2, int i)
{
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=i/INT_BITS;t<int_num;t++)
	{
		if (array[t] & list2->array[t])
			return false;  // no, not disjunct
	}
	return true;
#else
	return (array & list2->array == 0)
#endif
}

/* notDisjunctFrom -- better than !disjunctFrom(...) for long bitvectors */
INLINE bool BitVector::notDisjunctFrom(BitVector *input,int start_at)
{
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=start_at/INT_BITS;t<int_num;t++)
	{
		if (array[t] & input->array[t])
			return true;  // has intersection
	}
	return false;
#else
	return (array & input->array != 0);
#endif
}

/* lowOrderMissing */
INLINE int BitVector::lowOrderMissing(int start_at)
{
	for (VECTOR_COUNTING_TYPE t=start_at; t<num; t++)
		if (!readArray(t)) return t;
	return -1; 
}

/* lowOrderPresent */
INLINE int BitVector::lowOrderPresent(int start_at)
{
	for (VECTOR_COUNTING_TYPE t=start_at; t<num; t++)
		if (readArray(t)) return t;
	return -1; 
}

/* printVector */
INLINE void BitVector::printVector(void)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
		printf ("%d",readArray(t));
	printf ("\n");
}

/* readArray */
INLINE bool BitVector::readArray(int index)
{
#if (MAX_NUM_GOODS > INT_BITS)
	return (array[index/INT_BITS] & ((VECTOR_DATA_TYPE) 1 << (index % INT_BITS))) != 0;
#else
	return (array & ((VECTOR_DATA_TYPE) 1 << index)) != 0;
#endif
}

/* set a bit to true */
INLINE void BitVector::setBit(int index)
{
#if (MAX_NUM_GOODS > INT_BITS)
	array[index/INT_BITS] |= ((VECTOR_DATA_TYPE) 1 << (index % INT_BITS));
#else
	array |= ((VECTOR_DATA_TYPE) 1 << index);
#endif
	num_set++;
}

/* clear the whole bitvector */
INLINE void BitVector::reset()
{
#if (MAX_NUM_GOODS > INT_BITS)
	for(VECTOR_COUNTING_TYPE i=0;i<int_num;i++)
		array[i] = 0;
//	memset(array,0x0,int_num*sizeof(VECTOR_DATA_TYPE));
#else
	array = 0;
#endif
	num_set = 0;
}

/* set a bit to false */
INLINE void BitVector::clearBit(int index)
{
#if (MAX_NUM_GOODS > INT_BITS)
	array[index/INT_BITS] &= (VECTOR_DATA_TYPE)(-1) - ((VECTOR_DATA_TYPE) 1 << (index % INT_BITS));
#else
	array &= (VECTOR_DATA_TYPE)(-1) - ((VECTOR_DATA_TYPE) 1 << index);
#endif
	num_set--;
}

/* recalculate num_set, if corrupted by bitwise operations */
INLINE void BitVector::countNumSet()
{
	num_set = 0;
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
		if (readArray(t)) num_set++;
}

// access function for int_num
INLINE 	VECTOR_COUNTING_TYPE BitVector::getIntNum()
{
	return int_num;
}

/* access function for num_set */
INLINE VECTOR_COUNTING_TYPE BitVector::getNumSet()
{
	// if num_set was flagged as corrupted, recalculate it
	if (num_set == -1)
		countNumSet();
#ifdef DEBUG_NUMSET
	else
	{
		int temp;
		temp = num_set;
		countNumSet();
		assert(temp == num_set);
	}
#endif
	return num_set;
}

/* hash function */
INLINE unsigned BitVector::hash()
{
	unsigned temp=hash(array[0]);
	for (VECTOR_COUNTING_TYPE t=1; t<int_num; t++)
		temp ^= hash(array[t]) + (temp >> 2);
	return temp;
}

// http://www.concentric.net/~Ttwang/tech/inthash.htm
INLINE unsigned BitVector::hash(unsigned key)
{
  key += ~(key << 16);
  key ^=  (key >> 5);
  key +=  (key << 3);
  key ^=  (key >> 13);
  key += ~(key << 9);
  key ^=  (key >> 17);
  return key;
}


/* test for equality between two bitvectors */
INLINE bool BitVector::equalTo(BitVector *input)
{
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
	{
		if (array[t] != input->array[t])
			return false;  // not equal
	}
	return true;  // equal
#else
	return (array == input->array);
#endif
	
}

/* test for equality between bitvector and array */
INLINE bool BitVector::equalTo(VECTOR_DATA_TYPE *input_array)
{
#if (MAX_NUM_GOODS > INT_BITS)
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
	{
		if (this->array[t] != input_array[t])
			return false;  // not equal
	}
	return true;  // equal
#else
	return (this->array == input_array[0]);
#endif
	
}

/* this is a subset of/equal to the argument vec2 */
INLINE bool BitVector::subsetequalof(BitVector *vec2)
{
#if (MAX_NUM_GOODS > INT_BITS)
	VECTOR_DATA_TYPE a,b;
	
	for (VECTOR_COUNTING_TYPE t=0;t<int_num;t++)
	{
		a = array[t];
		b = vec2->array[t];
		if (((a & b) | (a ^ b)) != b)
			return false;  // a is not a subset of b
	}
	return true;  // subsetequalof
#else
	return ((array & input->array) | (array ^ input->array) == input->array);
#endif
}

INLINE VECTOR_DATA_TYPE *BitVector::getArray()
{
	return array;
}

#endif //BITVECTOR2
