//Vector2 -- Inline functions from Bitvector

#ifndef VECTOR2
#define VECTOR2

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "vector.h"
#include "defines.h"

// duplicate 
INLINE Vector *Vector::duplicate()
{
	Vector *temp = new Vector(num,max,1);
	copyTo(temp);
	return temp;
}

// copyTo 
INLINE void Vector::copyTo(Vector *dest)
{
	memcpy(dest->array,this->array, num*sizeof(VECTOR_DATA_TYPE));
	dest->num_set = this->num_set;
	dest->max = this->max;
}

// UnionWith 
INLINE Vector *Vector::unionWith(Vector *new_array, Vector *list2)
{
	// this function operates on integers instead of characters so that it is four times faster 
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
	{
		#ifdef _DEBUG
			if (array[t] > max[t]) 
			{
				printf ("Vector::unionWith error!\n");
				exit(1);
			}
		#endif
		new_array->array[t] = array[t] + list2->array[t];
	}

	// update num_set
	new_array->num_set = this->num_set + list2->num_set;

	return new_array;
}

// UnionWithSelf 
INLINE void Vector::unionWith(Vector *list2)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
	{
		array[t] += list2->array[t];
		#ifdef _DEBUG
			if (array[t] > max[t]) 
			{
				printf ("Vector::unionWith error!\n");
				exit(1);
			}
		#endif
	}

	// update num_set
	num_set = this->num_set + list2->num_set;
}

// intersect one vector with another 
INLINE void Vector::intersectWith(Vector *input)
{
	// not used
	exit(1);
}

// intersect one vector with another 
// WARNING: num_set is corrupted by this function 
INLINE void Vector::intersectWithNot(Vector *input)
{
	// not used
	exit(1);
}

// disjunctFrom 
INLINE bool Vector::disjunctFrom(Vector *list2, int start_at)
{
#ifdef _DEBUG
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)  // kkk - 0 should be "start_at"
	{
		assert(!(array[t] && list2->array[t] && t<start_at));
		if (array[t] + list2->array[t] > max[t])
			return false;  // no, not disjunct
	}
	return true;
#else
	for (VECTOR_COUNTING_TYPE t=start_at;t<num;t++)
	{
		if (array[t] + list2->array[t] > max[t])
			return false;  // no, not disjunct
	}
	return true;
#endif
}

// notDisjunctFrom -- better than !disjunctFrom(...) for long vectors 
INLINE bool Vector::notDisjunctFrom(Vector *input,int start_at)
{
	for (VECTOR_COUNTING_TYPE t=start_at;t<num;t++)
	{
		if (array[t] + input->array[t] > max[t])
			return true;  // has intersection
	}
	return false;
}

// lowOrderMissing 
INLINE int Vector::lowOrderMissing(int start_at)
{
	for (VECTOR_COUNTING_TYPE t=start_at; t<num; t++)
		if (readArray(t) < max[t]) return t;
	return -1; 
}

// lowOrderPresent 
INLINE int Vector::lowOrderPresent(int start_at)
{
	for (VECTOR_COUNTING_TYPE t=start_at; t<num; t++)
		if (readArray(t) > 0) return t;
	return -1; 
}

// printVector 
INLINE void Vector::printVector(void)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
		printf ("%d",readArray(t));
	printf ("\n");
}

// readArray 
INLINE VECTOR_DATA_TYPE Vector::readArray(int index)
{
	return array[index];
}

// add to a vector element 
INLINE void Vector::add(int index,VECTOR_DATA_TYPE num)
{
#ifdef _DEBUG
	if ((unsigned)num + array[index] > max[index])
	{
		printf ("Vector::add error!\n");
		printVector();
		exit(1);
	}
#endif
	array[index] += num;
	num_set += num;
}

// set a vector element
INLINE void Vector::set(int index,VECTOR_DATA_TYPE num)
{
#ifdef DEBUG_NUMSET
	if ((unsigned)num > max[index])
	{
		printf ("Vector::set error!\n");
		printVector();
		exit(1);
	}
#endif
	num_set += num - array[index];
	array[index] = num;
}

// clear the whole vector 
INLINE void Vector::reset()
{
	memset(array,0,sizeof(VECTOR_DATA_TYPE)*num);
	num_set = 0;
}

// set a bit to false 
INLINE void Vector::subtract(int index, VECTOR_DATA_TYPE num)
{
#ifdef _DEBUG
	if ((signed)array[index] - (signed)num < (signed)0)
	{ 
		printf ("Vector::subtract error!\n");
		printVector();
		exit(1);
	}
#endif
	array[index] -= num;
	num_set-=num;
}

// recalculate num_set, if corrupted by bitwise operations 
INLINE void Vector::countNumSet()
{
	num_set = 0;
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
		num_set += readArray(t);
}

// access function for num_set 
INLINE VECTOR_COUNTING_TYPE Vector::getNumSet()
{
	// if num_set was flagged as corrupted, recalculate it
	if (num_set == -1)
		countNumSet();
#ifdef DEBUG_NUMSET
	else
	{
		int temp;
		temp = num_set;
		countNumSet();
		assert(temp == num_set);
	}
#endif
	return num_set;
}

// access function for array
INLINE VECTOR_DATA_TYPE *Vector::getArray()
{
	return array;
}

// int num
INLINE VECTOR_COUNTING_TYPE Vector::getIntNum()
{
	return num;
}

// access function for max - readarray
INLINE unsigned Vector::remaining(VECTOR_COUNTING_TYPE index)
{
	return maxUnits(index) - readArray(index);
}

// access function for max
INLINE unsigned Vector::maxUnits(VECTOR_COUNTING_TYPE index)
{
	return max[index];
}

// hash function 
INLINE unsigned Vector::hash()
{
	unsigned temp=hash(array[0]);
	for (VECTOR_COUNTING_TYPE t=1; t<num; t++)
		temp ^= hash(array[t]) + (temp << 2);
	return temp;
}

// http://www.concentric.net/~Ttwang/tech/inthash.htm
INLINE unsigned Vector::hash(unsigned key)
{
  key += ~(key << 16);
  key ^=  (key >> 5);
  key +=  (key << 3);
  key ^=  (key >> 13);
  key += ~(key << 9);
  key ^=  (key >> 17);
  return key;
}


// test for equality between two vectors 
INLINE bool Vector::equalTo(Vector *input)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
	{
		if (array[t] != input->array[t])
			return false;  // not equal
	}

	return true;  // equal	
}

// test for equality between vector and array 
INLINE bool Vector::equalTo(VECTOR_DATA_TYPE *input_array)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
	{
		if (array[t] != input_array[t])
		  {
		    //printArray25(input_array, num);
		    return false;  // not equal
		  }
	}
	//this->printVector();
	
	
	return true;  // equal
}

// this is a subset of/equal to the argument vec2 
INLINE bool Vector::subsetequalof(Vector *vec2)
{
	for (VECTOR_COUNTING_TYPE t=0;t<num;t++)
	{
		if (array[t] > vec2->array[t] ||              // a is bigger than b
			(array[t] == 0 && vec2->array[t] != 0))   // a is zero, b is nonzero  (this is just for VECTOR's)
			return false;  // a is not a subset of b
	}
	return true;  // subsetequalof
}

#endif //VECTOR2
