////////////////////////////////////////////////////////
/* Store a multibin of bids; used for the array of bins */
////////////////////////////////////////////////////////

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#include "defines.h"

#ifdef MULTI_UNIT

#include "multibin.h"
#include "qsort.h"
#include "vector.h"

extern VECTOR_DATA_TYPE *max_array;

/* constructor */
MultiBin::MultiBin(int bin_number, int maxnum, MultiBin **binvec, int num_subbins)
{
	unsigned t;
	
	// parameters
	this->bin_number = bin_number;
	this->maxnum = maxnum + 1;  // to make room for one singleton bid
	this->bin = binvec;
	num_subbins++;
	this->num_subbins = num_subbins;
	
	// memory allocation
	subbin = new Bid**[num_subbins];
	current_bid = new int[num_subbins];
	num = new int[num_subbins];
	temp_allocation = new Allocation * [num_subbins];
	for (t=0;t<(unsigned)num_subbins;t++)
	{
		subbin[t] = new Bid*[max(num_subbins,maxnum)*FUDGE_FACTOR];
		current_bid[t] = -1;
		num[t]=0;
		temp_allocation[t] = new Allocation(new Vector(NUM_ITEMS,max_array,0),0);
	}
	temp_vec = new Vector(NUM_ITEMS,max_array,1);
	singleton = new Bid*[max(num_subbins,maxnum)*FUDGE_FACTOR];
	
	// initialization
	progress = 0;
	sing_num = 0;

	// allocate pruning table
	prune_table = new LinkedList *[NUM_ITEMS];
	for (t=bin_number; t<NUM_ITEMS; t++)
	{
		prune_table[t] = new LinkedList(bin_number);
	}
}

/* destructor */
MultiBin::~MultiBin()
{
	unsigned t;
	
	delete temp_vec;
	for (t=0;t<(unsigned)num[0];t++)
		delete subbin[0][t];
	for (t=0;t<(unsigned)num_subbins;t++)
	{
		delete temp_allocation[t];
		delete[] subbin[t];
//		if (singleton[t]->delete_flag == 2 || singleton[t]->delete_flag == 3)
//			delete singleton; // merged bid or dummy bid
	}
	delete[] singleton;
	delete[] temp_allocation;
	delete[] subbin;
	delete[] num;
	delete[] current_bid;

	// delete pruning table
	for (t=bin_number; t<NUM_ITEMS; t++)
	{
		delete prune_table[t];
	}
	delete[] prune_table;
}


// the compare function required by the standard C quicksort function
// sorts bids in descending order of expected revenue.
// expected revenue is actual revenue from bid plus overestimated further revenue from pruning
int MultiBin::qsortCompare(const void *bid1, const void *bid2)
{
	static unsigned amt1, amt2;

	// calculate the amounts to sort on
	amt1 = (*(Bid **)bid1)->pruneAvg() + 2*(*(Bid **)bid1)->average;
	amt2 = (*(Bid **)bid2)->pruneAvg() + 2*(*(Bid **)bid2)->average;

	// note: these comparisons are backwards to get qsort to sort in descending order
	if (amt1 < amt2) return 1;
	else if (amt1 > amt2) return -1;
	else return 0;
}

// the compare function required by the standard C quicksort function
// sorts bids in descending order of expected revenue.
// expected revenue is actual revenue from bid plus overestimated further revenue from pruning
int MultiBin::qsortCompare2(const void *bid1, const void *bid2)
{
	static unsigned amt1, amt2;

	// calculate the amounts to sort on
	amt1 = (*(Bid **)bid1)->average;
	amt2 = (*(Bid **)bid2)->average;

	// note: these comparisons are backwards to get qsort to sort in descending order
	if (amt1 < amt2) return 1;
	else if (amt1 > amt2) return -1;
	else return 0;
}

// sort the whole base multibin
void MultiBin::sort()
{
	Qsort((void *)(subbin[0]),num[0],sizeof(Bid *),qsortCompare2);
	//printf ("multibin sorting is disabled at the moment.\n");
}

// two dimensional array indexing -- reference arrays starting from 1 (!!!)
#define DP_ARRAY(a,b) (dp_array[n_subbins*(a-1)+(b-1)])

void MultiBin::constructSingletonArray()
{
	int sn = max(sing_num,1);
	int n_subbins = num_subbins - 1;
	Bid **dp_array = new Bid*[n_subbins*sn];
	unsigned num_set;

	memset (dp_array,NULL,sizeof(Bid *)*n_subbins*sn);

	// do the dynamic programming
	for (unsigned tt=1; tt<=sing_num;tt++)  // for each singleton bid
	{
		num_set = singleton[tt-1]->vec->getNumSet();

		for (unsigned t=1;t<=(unsigned)n_subbins;t++)  // for each subbin
		{
			// if the new item has just the right number of units
			if (num_set == t)
			{
				// first time, prev one was unallocated, both allocated but this one is better
				if (tt == 1 || !DP_ARRAY(tt-1,t) || DP_ARRAY(tt-1,t)->amount < singleton[tt-1]->amount)
					DP_ARRAY(tt,t) = singleton[tt-1];
				
				// prev one exists and is better
				else
					DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
			}

			// if the new item has fewer units
			else if (num_set < t)
			{
				// first time -- a special case because there are no previous columns in the table
				// or, neither of the previous entries exist
				if (tt == 1 || (!DP_ARRAY(tt-1,t-num_set) && !DP_ARRAY(tt-1,t))) 
					DP_ARRAY(tt,t) = singleton[tt-1];
				
				// both previous entries exist
				else if (DP_ARRAY(tt-1,t-num_set) && DP_ARRAY(tt-1,t))
				{
					// previous entry is better
					if (DP_ARRAY(tt-1,t)->amount >= singleton[tt-1]->amount + DP_ARRAY(tt-1,t-num_set)->amount)
					{
						DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
					}

					// new entry is better
					else DP_ARRAY(tt,t) = mergeBids(DP_ARRAY(tt-1,t-num_set),singleton[tt-1]);
				}

				// only big previous entry exists
				else if (!DP_ARRAY(tt-1,t-num_set) && DP_ARRAY(tt-1,t))
				{
					// previous entry is better
					if (DP_ARRAY(tt-1,t)->amount >= singleton[tt-1]->amount)
					{
						DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
					}

					// new entry is better
					else DP_ARRAY(tt,t) = singleton[tt-1];
				}

				// only small previous entry exists
				else if (DP_ARRAY(tt-1,t-num_set) && !DP_ARRAY(tt-1,t))
					DP_ARRAY(tt,t) = mergeBids(DP_ARRAY(tt-1,t-num_set),singleton[tt-1]);
				
				// strange error
				else
				{
					printf ("multibin::constructSingletonArray() -- strange error.\n");
					exit(1);
				}
			}

//			// if the new item has fewer units
//			else if (num_set < t)
//			{
//				// first time -- a special case because there are no previous columns in the table
//				if (tt == 1) 
//					DP_ARRAY(tt,t) = singleton[tt-1];
//				
//				// previous entry is better: promote it
//				else if (DP_ARRAY(tt-1,t-num_set) && DP_ARRAY(tt-1,t) && 
//					DP_ARRAY(tt-1,t)->amount >= singleton[tt-1]->amount + DP_ARRAY(tt-1,t-num_set)->amount)
//				{
//					DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
//				}
//				
//				else if (DP_ARRAY(tt-1, t) && DP_ARRAY(tt-1, t-num_set) && DP_ARRAY(tt-1, t)->amount && (singleton[tt-1]->amount + DP_ARRAY(tt-1, t-num_set)->amount))
//				else if (DP_ARRAY(tt-1, t) && DP_ARRAY(tt-1, t-num_set))
//				{
//				    DP_ARRAY(tt, t) = mergeBids(DP_ARRAY(tt-1,t-num_set),singleton[tt-1]);
//				}
//			
//				// big previous entry is better but smaller previous entry doesn't exist: promote big previous entry
//				else if (!DP_ARRAY(tt-1,t-num_set) && DP_ARRAY(tt-1,t) && 
//					DP_ARRAY(tt-1,t)->amount >= singleton[tt-1]->amount)
//				{
//					DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
//				}
//				
//				// merge the two records together, creating a new bid
//				else if (DP_ARRAY(tt-1,t-num_set))
//					DP_ARRAY(tt,t) = mergeBids(DP_ARRAY(tt-1,t-num_set),singleton[tt-1]);
//
//				// there is no previous entry, so there's nothing to merge
//				else
//					DP_ARRAY(tt,t) = singleton[tt-1];
//			}

			// if the new item has more units -- it can't be taken
			else //if (num_set > t)
			{
				// first time
				if (tt == 1) DP_ARRAY(tt,t) = NULL;
				else DP_ARRAY(tt,t) = DP_ARRAY(tt-1,t);
			}
		}
	}

	// mark items not to delete
	for (t=1;t<=(unsigned)n_subbins;t++)
	{
		if (DP_ARRAY(sn,t))
		{
			DP_ARRAY(sn,t)->delete_flag = 1; // mark protected
			if (DP_ARRAY(sn,t)->bid_array)
			{
				for (int tt=0; tt<DP_ARRAY(sn,t)->bid_array->num; tt++)
					DP_ARRAY(sn,t)->bid_array->bid[tt]->delete_flag = 1; // mark protected
			}
		}
	}
	
	// delete unused singletons
	for (t=0;t<(unsigned)sing_num;t++)
	{
		if (singleton[t]->delete_flag == 0)
			delete singleton[t];
	}
	
	// export singletons back to the array
	// memory can be leaked here, but probably not very much
	for (t=1;t<=(unsigned)n_subbins;t++)
	{
		// if the DP_ARRAY entry is not NULL
		if (DP_ARRAY(sn,t))
		{
			// if the entry names the correct number of units
			if ((unsigned)DP_ARRAY(sn,t)->vec->getNumSet() == t)
				singleton[t-1] = DP_ARRAY(sn,t);
			
			// if not, pad it up to the right number of units by adding a zero bid
			else
				singleton[t-1] = mergeBids(DP_ARRAY(sn,t), createDummyBid(t-DP_ARRAY(sn,t)->vec->getNumSet()));
		}

		// if the DP_ARRAY entry is NULL, we have to create a whole new zero bid
		else 
			singleton[t-1] = createDummyBid(t);
	}

	// mark singleton bids as such
	for (t=0;t<(unsigned)n_subbins;t++)
		singleton[t]->delete_flag = DF_IS_SINGLETON;

	// now add the biggest singleton to the list of actual bids.  It must be there, it must be last.
	//this->add(singleton[0],false);
	//no longer! always add in singletons manually now!

	delete[] dp_array;
}

// merge two bids together.  Bid1 may be a composite bid; bid2 may not.
Bid *MultiBin::mergeBids(Bid *bid1, Bid *bid2)
{
	// make the vector
	VECTOR *temp_vec = new Vector(NUM_ITEMS,max_array,1);
	bid2->vec->unionWith(temp_vec,bid1->vec);
	
	// start to build the bid
	Bid *temp_bid = new Bid(temp_vec,bid1->amount + bid2->amount,-2,bin_number,NULL);
//	temp_bid->delete_flag = 2;  // indicate that this bid should be deleted along with the bidhistory
	int temp_hist_num = 2;  // there are at least two bids in this composite bid
	
	// if bid1 is already a composite bid
	if (bid1->bid_array)
		temp_hist_num += bid1->bid_array->num;
	
	// build the history
	BidHistory *temp_hist = new BidHistory(temp_hist_num);
	if (bid1->bid_array)
		bid1->bid_array->copyTo(temp_hist);
	else
		// if bid1 is not a composite bid, it needs to go in the history (if it is, it is just the union of its history)
		temp_hist->append(bid1);
	temp_hist->append(bid2);
	temp_bid->bid_array = temp_hist;
	
	// return the bid we've created
	return temp_bid;
}	 

// create a dummy bid with zero revenue, number -1 and a given number of units
Bid *MultiBin::createDummyBid(int num_units)
{
	Vector *temp_vector = new Vector(NUM_ITEMS,max_array,1);
	temp_vector->set(bin_number,num_units);
	Bid *temp_bid = new Bid(temp_vector,0,-1,bin_number,NULL);
//	temp_bid->delete_flag = 3;
	return temp_bid;
}

// for debugging: print out the singleton arrays generated by dynamic programming
void MultiBin::printSingletonArray()
{
	printf ("bin %d (%d): ",bin_number,num_subbins-1);
	for (int t=0;t<num_subbins-1;t++)
	{
		if (singleton[t]->bid_num != -2)
			printf("%s%d",(t==0?"":", "),singleton[t]->bid_num);
		else
		{
			printf ("%s(",(t==0?"":", "));
			for (int tt=0;tt<singleton[t]->bid_array->num;tt++)
				printf("%s%d",(tt==0?"":", "),singleton[t]->bid_array->bid[tt]->bid_num);
			printf (")");
		}
	}
	printf ("\n");
}

#endif

