// binset.cpp: implementation of the BinSet class.
//
//////////////////////////////////////////////////////////////////////

#include "defines.h"

#ifndef MULTI_UNIT

#include "binset.h"
#include <assert.h>
#include <math.h>
#include <string.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

// constructor
BinSet::BinSet(int num_bins)
{
	this->num_bins = num_bins;
	bin = new Bin*[num_bins];
	for (int t=0; t<num_bins; t++)
		bin[t] = NULL;
	current_bin = 0;
	num_bids = 0;

	// singletons
	singleton = new unsigned[num_bins];
	memset(singleton,0,num_bins * sizeof(singleton[0]));

	// allocate binstack
	binstack = new int[num_bins];
	binstack_pointer = -1;
}

// destructor
BinSet::~BinSet()
{
	for (int t=0; t<num_bins; t++)
		delete bin[t];
	delete[] bin;
	delete[] singleton;
}

// put the bids into bins
void BinSet::populateBins(InputFile &infile)
{
	unsigned i,k;
	int first_good=-1;
	
	// allocate the bins
	for (i=0; i<NUM_ITEMS; i++)
		bin[i] = new Bin(i,infile.getNumBids(i),bin );

	// extract bids from the input parameter and populate the lists
	for (i = 0; i < infile.num_bids; i++) 
	{
		if (infile.bid[i]->deleted) continue;
		num_bids++;

		BitVector *temp_bitvector = new BitVector(NUM_ITEMS,NULL,1);
		first_good = NUM_ITEMS + 10;


		// add each bid to whatever bin it belongs to: the bin corresponding to the first found good
		for (k=0; k < infile.bid[i]->numGoods; k++) 
		{
			int good_num = infile.map(infile.bid[i]->array[k]);
			temp_bitvector->setBit(good_num);
			first_good = min(first_good,good_num);
		}

		// add the new bitvector to the bin
		Bid *temp_bid = new Bid(temp_bitvector,infile.bid[i]->amount,infile.bid[i]->bid_num,first_good,infile.bid[i]);
		assert (temp_bid->vec->readArray(first_good));
		assert (temp_bid->vec->lowOrderPresent(0) == first_good);
		this->bin[first_good]->add(temp_bid);
		
		// update singleton array
		if (temp_bitvector->getNumSet() == 1)
			singleton[first_good] = max(singleton[first_good],infile.bid[i]->amount);
	}
}

// output pruning data
void BinSet::gatherPruningStats()
{
#ifdef GATHER_PRUNING_STATS
	// hits
	FILE *fp = fopen("pruning_hit.csv","at");
	for (i=0;i<NUM_ITEMS;i++)
		fprintf (fp,"%d, ",this->prune_hits[i]);
	fprintf (fp,"\n");
	fclose(fp);
	
	// prunes
	fp = fopen("pruning_prune.csv","at");
	for (i=0;i<NUM_ITEMS;i++)
		fprintf (fp,"%d, ",this->prune_bin[i]);
	fprintf (fp,"\n");
	fclose(fp);
#endif

}

// build the pruning tables
void BinSet::constructPruningTables()
{
	signed l,k;
	
	// construct the pruning table for each good, for each bin
	for (l = num_bins - 1; l >= 0; l--)  // each bin, in descending order
	{
		Bin *this_bin = bin[l];

		// copy the previous bin's lists
		for (k=l+1;k<num_bins;k++)  // this gets skipped the first time (l==num_bins-1)
			bin[l]->getPruneTable(k)->copyFrom(bin[l+1]->getPruneTable(k));

		// add each bid to all the applicable pruning tables
		this_bin->constructNewBin();
		while (this_bin->nextBid())	// each bid in bin l
		{
			k = -1;

			// for each good in the bid
			while ((k = this_bin->currentBid()->vec->lowOrderPresent(k+1)) != -1)
			{
				// add the bid to the pruning table for that good
				this_bin->getPruneTable(k)->addBid(this_bin->currentBid());
			}
		} 
	}
}


// set up the first dynamic bin before running allocate
void BinSet::makeFirstBin(Allocation *allocation,int best_amount)
{
	Bid::firstSubbin();
	bin[0]->constructNewBin(allocation,best_amount);	
}

// sort all the base bins
void BinSet::sortBins()
{
#ifdef SORT_BINS
	for (int t=0;t<num_bins;t++)
		bin[t]->sort();
#endif
	return;
}

#endif // !multi_unit
