// INLINE functions from the BIN class

#ifndef BIN2
#define BIN2

#include "defines.h"

#ifndef MULTI_UNIT

#include <stdio.h>
#include <stdlib.h>

#include "bin.h"
#include "qsort.h"
#include "stats.h"

extern Stats *stats;

/* add */
INLINE void Bin::add(Bid *b)
{
	// error checking
#ifdef _DEBUG // was BIN_ERROR_CHECKING
	if (base_num+1 > maxnum)
	{
		printf ("Error -- too many items in bin!\n");
		exit(1);
	}
#endif

	// allocate another bin item
	base_bid[base_num++] = b;
}

// accessor for private member prune_table -- use with care!
// used by BinSet for the construction of the pruning tables
INLINE LinkedList* Bin::getPruneTable(int item)
{
	return prune_table[item];
}

// find the prune value for all unallocated goods in an allocation
INLINE unsigned Bin::pruneVal(BitVector *vec)
{
	estimate = 0;
//	lom_count = 0;
	t = bin_number;

	// consider the revenue from each unallocated good
	while (t != -1 && t < NUM_ITEMS - NUM_DUMMY_ITEMS)
	{
		estimate += pruneVal(t, vec);
//		lom_count++;
		t=vec->lowOrderMissing(t+1);
	}
	
	return (estimate + ((1<<PRUNE_SHIFT)-1) ) >> PRUNE_SHIFT;
}

// consider the next bid
INLINE bool Bin::nextBid()
{
	// last bid
	if (current_bid == dyn_num - 1)
	{
		current_bid = -1;
		return false;
	}
	
	// next bid
	else current_bid++;

	return true;
}

INLINE Allocation * Bin::tempAllocation()
{
	return temp_allocation;
}

// number of items in the bin
INLINE int Bin::dynNum()
{
	return dyn_num;
}

// return the current bid
INLINE Bid * Bin::currentBid()
{
#ifdef _DEBUG
	if (current_bid == -1 || current_bid >= dyn_num)
	{
		printf ("Bin::currentBid() error!\n");
		exit(1);
	}
#endif
	return dyn_bid[current_bid];
}

// return the percentage through the goods in this bin
INLINE int Bin::percentDone()
{
	if (dyn_num == 0 || current_bid == -1)
		return 0;
	else return max((100 * current_bid) / dyn_num,0);
}

INLINE int Bin::binNumber()
{
	return bin_number;
}

extern bool no_progress_before_abort;

// runs every time a bid has been searched (prune, cache, real search)
// right now just for randomization, but in the future could be used for other things
INLINE void Bin::bidSearched(Bid *this_bid)
{
#ifdef RANDOMIZE
	if (this->bin_number == 0)
	{
		no_progress_before_abort = false;
		#ifdef DELETE_SEARCHED_BIDS
			this_bid->input_bid->deleted = true;
		#endif
	}
#endif
}

// return the pruning overestimate for good t, vector vec
INLINE int Bin::pruneVal(int t, BitVector *vec)
{
	return (int) prune_table[t]->maxAvgRevenue(vec);
	//return (int) prune_table[t]->maxAvgRevenue();
}

// make a new bin, containing only bids disjunct from allocation and
// sorted dynamically, //using the dyn_bid array, and update the pruning
// values etc. for the dyn_bid
// this function pre-calculates all of the values that are important to Cass::allocate()
INLINE void Bin::constructNewBin(Allocation *allocation,unsigned best_amount,BidHistory *history)
{
	unsigned temp_prune;
	int temp_lom;
	int temp_prune_avg;
	max_prune_val = 0;
	dyn_num=0;

	// consider every bid in the base bin
	for (int t=0;t<base_num; t++)
	{
		// test for disjunction
		if (base_bid[t]->vec->disjunctFrom(allocation->vec,bin_number)) 
		{
			// add the bid to the allocation, so we can start examining the vector
		  allocation->vec->unionWith(temp_vec,base_bid[t]->vec);
		  // this is where temp_vec changes.
			
			// determine next bin after adding this bid
			temp_lom = temp_vec->lowOrderMissing(bin_number);

			// construct pruning value
			#ifdef PRUNING_ENABLED
				if (temp_lom != -1)
				{
					temp_prune = bin[temp_lom]->pruneVal(temp_vec);
					if ((unsigned) temp_vec->getNumSet() == NUM_ITEMS) 
						temp_prune_avg = 0;
					else
						temp_prune_avg = (temp_prune << PRUNE_SHIFT) / (NUM_ITEMS - temp_vec->getNumSet());
					temp_prune +=  base_bid[t]->amount; // NB: temp_prune includes the bid amount.
				}
				else 
				{
					temp_prune = base_bid[t]->amount;  // NB: temp_prune includes the bid amount.
					temp_prune_avg = base_bid[t]->average;
				}
			#else
				temp_prune = LARGE_NUMBER;
				temp_prune_avg = 0;
			#endif

			// add the bid if it *can't* be pruned out
			if (temp_prune + allocation->amount > best_amount)
			{
				// add the bid to the new bin
				dyn_bid[dyn_num] = base_bid[t];
				dyn_bid[dyn_num]->setPruneVal(temp_prune);
				dyn_bid[dyn_num]->setNextBin(temp_lom);
				dyn_bid[dyn_num]->setPruneAvg(temp_prune_avg);
				dyn_num++;
			}

			// else, keep track of the amount pruned for caching purposes
			else
			{ 
				stats->incrementPruneHit();
				bidSearched(base_bid[t]);
				#ifdef CACHING_ENABLED
					max_prune_val = max (max_prune_val,temp_prune);
				#endif
			}
		}
	}

	// sort the new bin
	#ifdef DYNAMICALLY_ORDER_BINS
		if (dyn_num) 
		{
//			if (this->bin_number == 0)
//				Qsort((void *)(dyn_bid),dyn_num,sizeof(Bid *),qsortCompareFirst);
//			else
				Qsort((void *)(dyn_bid),dyn_num,sizeof(Bid *),qsortCompare);
		}
	#endif

	// start from the beginning of the new bin
	current_bid = -1;
}

// construct a new bin with no reordering
INLINE void Bin::constructNewBin()
{
	memcpy(dyn_bid,base_bid,sizeof(Bid *)*base_num);
	dyn_num = base_num;
	current_bid = -1;
}

#endif //!multiunit
#endif
