// Param.cpp: implementation of the Param class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#ifndef _WIN32
	#include <unistd.h>
#endif

#define max(a,b) (a>b?a:b)

// this must preceed including Param.h
// #define DNAME_DEF
#include "Param.h"
#include "Legacy.h"

// static members

 const char* Param::CATS_VERSION_STRING = "2.1";

 const int Param::dname_count = 7;
 const char* Param::dname[] = {"arbitrary","matching","multipaths","paths","regions","scheduling", "legacy"};


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

// parse the parameters: done in the constructor
// all distribution-specific parameters are stored in a table as strings, and parsed in the distribution itself
Param::Param(char** argv, int argc) 
{
	// initalize variables to defaults
	distribution = UNASSIGNED;
	num_runs = 1;
	cplex_output = false;
	random_seed = 0; //- make results replicatable
	random_seed2 = 0; //- make results replicatable
	filename = NULL;
	cplex_filename = NULL;
	output_settings_buffer = NULL;
	num_bids = 0;
	num_goods = 0;
	remove_dominated_bids = true;
	output_parameter_settings = true;
	integer_prices = false;
	parse_error = false;
	verbatim = false;
	output_frequency = 20; // no parameter for this one
	random_parameters = false;
	random_goods = false;
	random_bids = false;
	bid_alpha = 1000;
	converting = false;

	// no parameters passed
	if (argc == 1)
	{
		usage();
		parse_error = true;
		return;
	}
	
	// parse parameters
	for (int i = 1; i<argc; i++)
	{
		// now read in values
		if (argv[i][0] == '-' || argv[i][0] == '/')
		{
			char *in = &(argv[i][1]); // skip the first character

			// distribution
			if (!stricmp(in,"d"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }

				for (int t=0;t<dname_count;t++)
					if (stricmp(argv[i+1],dname[t]) == 0)
						distribution = (dist_type)t;

				if (distribution == UNASSIGNED)
				{
					printf ("Unknown distribution: %s\n",argv[i+1]);
					parse_error = true;
					return;
				}
				i++;
			}

			// num_runs
			else if (!stricmp(in,"n"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				num_runs = atoi(argv[i+1]);
				if (!num_runs) {usage(); parse_error = true; return; }
				i++;
			}

			// cplex_output
			else if (!stricmp(in,"cplex"))
			{
				cplex_output = true;
			}

			// bid_alpha
			else if (!stricmp(in,"bid_alpha"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				bid_alpha = atoi(argv[i+1]);
				if (!bid_alpha) {usage(); parse_error = true; return; }
				i++;
			}

			// seed1
			else if (!stricmp(in,"seed"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				random_seed = atoi(argv[i+1]);
				if (!random_seed) {usage(); parse_error = true; return; }
				i++;
			}

			// seed2
			else if (!stricmp(in,"seed2"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				random_seed2 = atoi(argv[i+1]);
				if (!random_seed2) {usage(); parse_error = true; return; }
				i++;
			}

			// bids
			else if (!stricmp(in,"bids"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				num_bids = atoi(argv[i+1]);
				if (!num_bids) {usage(); parse_error = true; return; }
				i++;
			}

			// random bids
			else if (!stricmp(in,"random_bids"))
			{
				if (i+2 >= argc) {usage(); parse_error = true; return; }
				random_bids = true;
				min_bids = atoi(argv[++i]);
				max_bids = atoi(argv[++i]);
				if (!min_bids || !max_bids) {usage(); parse_error = true; return; }
			}

			// goods
			else if (!stricmp(in,"goods"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				num_goods = atoi(argv[i+1]);
				if (!num_goods) {usage(); parse_error = true; return; }
				i++;
			}

			// random goods
			else if (!stricmp(in,"random_goods"))
			{
				if (i+2 >= argc) {usage(); parse_error = true; return; }
				random_goods = true;
				min_goods = atoi(argv[++i]);
				max_goods = atoi(argv[++i]);
				if (!min_goods || !max_goods) {usage(); parse_error = true; return; }
			}

			// no_dom_check
			else if (!stricmp(in,"no_dom_check"))
			{
				remove_dominated_bids = false;
			}

			// random_parameters
			else if (!stricmp(in,"random_parameters"))
			{
				random_parameters = true;
			}

			// filename
			else if (!stricmp(in,"filename"))
			{
				if (i+1 >= argc) {usage(); parse_error = true; return; }
				filename = argv[i+1];
				cplex_filename = filename; // in the future this could be a separate parameter
				i++;
			}

			// verbatim
			else if (!stricmp(in, "verbatim"))
			{
			  verbatim = true;
			}

			// integer_prices
			else if (!stricmp(in,"int_prices"))
			{
				integer_prices = true;
			}

			// output_parameter_settings
			else if (!stricmp(in,"no_output"))
			{
				output_parameter_settings = false;
			}

			// cats2cplex
			else if (!stricmp(in, "cats2cplex"))
			{
				if (i+2 >= argc) {usage(); parse_error = true; return; }
				filename = argv[i+1];
				cplex_filename = argv[i+2];
				converting = true;
				verbatim = true;
				i += 2;
			}

			// help
			else if (!stricmp(in,"h") || !stricmp(in,"help") || !stricmp(in,"?"))
			{
				usage();
				parse_error = true;
				return;
			}
		}
	}

	if (converting) return;

	// check for errors in the input
	if ((!random_bids && num_bids == 0) ||
		 (!random_goods && num_goods == 0) ||
		 (distribution == UNASSIGNED))
	{
		usage();
		parse_error = true;
		return;
	}

	if (num_runs > 1)
		verbatim = false;

	if (!filename) {
	  filename = dname[distribution];
	  cplex_filename = dname[distribution];
	}

	// other initialization

	if (!random_seed)
	  random_seed = (long)time(NULL) + (long)getpid();

	if (!random_seed2) {
	  // wait a bit of time in case we just took seed1 from the clock
	  for (int t=0;t<random_seed%100000;t++);
	  random_seed2 = time(NULL) % 99873 - (long)getpid();
	}

	Seed();
}

Param::~Param()
{
	if (output_settings_buffer) delete[] output_settings_buffer;
}

// allow the use of a different random number generator for different platforms
double Param::DRand()
{
#ifndef _WIN32
	return drand48();
#else
	return (double)rand()/RAND_MAX;
#endif
}

// random number on a range
double Param::DRand(double minval, double maxval)
{
	return Param::DRand() * (maxval - minval) + minval;
}

long Param::LRand()
{
#ifndef _WIN32
	return lrand48();
#else
	return rand();
#endif
}

// random number on a range
long Param::LRand(long minval, long maxval)
{
	return Param::LRand() % (maxval - minval + 1) + minval;
}

void Param::Seed()
{
#ifndef _WIN32
  srand48(random_seed);
#else
	srand(random_seed);
#endif
}

// write out the parameter settings
char *Param::outputSettings(bool tofile)
{
	if (!output_settings_buffer)
	  output_settings_buffer = new char[500];

	if (tofile) sprintf(output_settings_buffer,"%% Goods: %d; Bids: %d; Distribution: %s; Runs: %d; Seed: %d",num_goods, num_bids,
		dname[distribution], num_runs, random_seed);
	else sprintf(output_settings_buffer,"Goods: %d\nBids: %d\nDistribution: %s\nRuns: %d\nSeed: %d\n",num_goods, num_bids,
		dname[distribution], num_runs, random_seed);

	return output_settings_buffer;
}

int Param::Round(double val)
{
	return (int)(val + (double)0.5);
}

// support for ERand under windows compilers that can't handle it
double Param::ERand(unsigned short x[3])
{
#ifndef _WIN32
	return erand48(x);
#else
	return Param::DRand();
#endif
}

void Param::usage()
{
// 80 cols:  12345678901234567890123456789012345678901234567890123456789012345678901234567890
	int count = printf ("CATS v%s (http://robotics.stanford.edu/CATS)\n",CATS_VERSION_STRING);
	int count2 = printf ("Kevin Leyton-Brown, Mark Pearson, Galen Andrew, Yoav Shoham; Stanford University\n");
    int t; // -- NOTE: MSVC uses older syntax, so cannot declare it withn a forloop :(
	for (t=0;t<max(count,count2);t++) printf ("=");
	printf ("\n\n");

	if (distribution == UNASSIGNED)
	{
		printf ("What follows are the general parameters for CATS.\n");
		printf ("To see parameters for a specific distribution, select a distribution with -help\n\n");
	}

	printf ("Required Parameters (no default values):\n");
	printf ("  -d [no default]: selects a distribution.  Valid options (without quotes) are:");
	count = 80;
	for (t=0;t<dname_count;t++)
	{
		if (count >= 70) 
		{
			printf ("\n     ");
			count = 0;
		}
		count += printf ("\"%s\"; ",dname[t]);
	}
	printf ("\b\b \n\n");

	printf ("  -goods [no default]: number of goods.\n");
	printf ("       --OR--\n");
	printf ("  -random_goods [no default] [no default]: min/max number of goods.\n\n");

	printf ("  -bids [no default]: number of bids to generate.\n");
	printf ("       --OR--\n");
	printf ("  -random_bids [no default] [no default]: min/max number of bids.\n");

	printf ("\nOptional Parameters:\n");
	printf ("  -n [1]: number of files that you want CATS to generate.\n");
	printf ("  -seed [taken from clock]: random seed\n");
	printf ("  -seed2 [taken from clock]: random seed 2: used only for normal distributions.\n");
	printf ("  -bid_alpha [1000]: multiply bids by this before rounding, when -int_prices.\n");

	printf ("\nOptional Flags (no value follows the flag):\n");
	printf ("  -cplex: write a CPLEX-compatible .lp file as well as a CATS data file\n");
	printf ("  -no_dom_check: CATS removes dominated bids, unless you include this flag\n");
	printf ("  -int_prices: write integer prices in bids instead of real-valued prices\n");
	printf ("  -filename:  set filename prefix for generated instances\n");
	printf ("  -verbatim:  do not append indices to the filename. works only with -n 1\n");
	printf ("  -no_output: do not write unnecessary output to the screen\n");
	printf ("  -help, -h, -?: print this usage screen\n");

	// more information about supported distributions:
	if (distribution != UNASSIGNED)
	{
		printf ("\n");
		switch (distribution) {
		case LEGACY:
			Legacy::usage();
			break;
		default:
			printf ("No help screen currently exists for the distribution %s\n",dname[distribution]);
		}
	}

	printf("\nCATS also has a utility for converting from CATS output to CPLEX output.\n");
	printf("This is run with the following flag:\n");
	printf("  -cats2cplex [input filename] [output filename]\n");
}
