// Legacy.cpp: implementation of the Legacy class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "Legacy.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


Legacy::Legacy(Param &p, char * argv[], int argc)
{
	norm = new Normal (p);
	output_buffer = NULL;

	// first set default values
	num_goods_type = DECAY;
	binom_cutoff = 0.2;
	q = 5;
	const_goods = 3;
	mu_goods = 4;
	sigma_goods = 1;
	pricing_type = LINEAR_RANDOM;
	high_fixed = 1000;
	low_fixed = 0;
	high_linearly = 1000;
	low_linearly = 1;
	mu_price = 16;
	sigma_price = 3;
	alpha = 0.55;

	// read through the parameters.  Ignore those that don't apply
	for (int i = 1; i<argc; i++)
	{
		// now read in values
		if (argv[i][0] == '-' || argv[i][0] == '/')
		{
			char *in = &(argv[i][1]); // skip the first character

			// num_goods_type
			if (!stricmp(in,"num_goods_type"))
			{
				if (i+1 >= argc) usage();
				num_goods_type = (goods_enum) atoi(argv[i+1]);
				i++;
			}

			// binom_cutoff
			if (!stricmp(in,"binom_cutoff"))
			{
				if (i+1 >= argc) usage();
				binom_cutoff = atof(argv[i+1]);
				i++;
			}

			// q
			else if (!stricmp(in,"q"))
			{
				if (i+1 >= argc) usage();
				q = atof(argv[i+1]);
				i++;
			}

			// const_goods
			else if (!stricmp(in,"const_goods"))
			{
				if (i+1 >= argc) usage();
				const_goods = atoi(argv[i+1]);
				i++;
			}

			// mu_goods
			else if (!stricmp(in,"mu_goods"))
			{
				if (i+1 >= argc) usage();
				mu_goods = atof(argv[i+1]);
				i++;
			}

			// sigma_goods
			else if (!stricmp(in,"sigma_goods"))
			{
				if (i+1 >= argc) usage();
				sigma_goods = atof(argv[i+1]);
				i++;
			}

			// pricing_type
			else if (!stricmp(in,"pricing_type"))
			{
				if (i+1 >= argc) usage();
				pricing_type = (pricing_enum) atoi(argv[i+1]);
				i++;
			}

			// high_fixed
			else if (!stricmp(in,"high_fixed"))
			{
				if (i+1 >= argc) usage();
				high_fixed = atof(argv[i+1]);
				i++;
			}

			// low_fixed
			else if (!stricmp(in,"low_fixed"))
			{
				if (i+1 >= argc) usage();
				low_fixed = atof(argv[i+1]);
				i++;
			}

			// high_linearly
			else if (!stricmp(in,"high_linearly"))
			{
				if (i+1 >= argc) usage();
				high_linearly = atof(argv[i+1]);
				i++;
			}

			// low_linearly
			else if (!stricmp(in,"low_linearly"))
			{
				if (i+1 >= argc) usage();
				low_linearly = atof(argv[i+1]);
				i++;
			}

			// mu_price
			else if (!stricmp(in,"mu_price"))
			{
				if (i+1 >= argc) usage();
				mu_price = atof(argv[i+1]);
				i++;
			}

			// sigma_price
			else if (!stricmp(in,"sigma_price"))
			{
				if (i+1 >= argc) usage();
				sigma_price = atof(argv[i+1]);
				i++;
			}

			// alpha
			else if (!stricmp(in,"alpha"))
			{
				if (i+1 >= argc) usage();
				alpha = atof(argv[i+1]);
				i++;
			}

			// L1
			else if (!stricmp(in,"L1"))
			{
				low_fixed = 0;
				high_fixed = 1;
				num_goods_type = RANDOM;
				pricing_type = FIXED_RANDOM;
			}

			// L1a
			else if (!stricmp(in,"L1a"))
			{
				low_fixed = 0;
				high_fixed = 1000;
				num_goods_type = RANDOM;
				pricing_type = FIXED_RANDOM;
			}

			// L2
			else if (!stricmp(in,"L2"))
			{
				low_linearly = 0;
				high_linearly = 1;
				num_goods_type = RANDOM;
				pricing_type = LINEAR_RANDOM;
			}

			// L2a
			else if (!stricmp(in,"L2a"))
			{
				low_linearly = 500;
				high_linearly = 1500;
				num_goods_type = RANDOM;
				pricing_type = LINEAR_RANDOM;
			}

			// L3
			else if (!stricmp(in,"L3"))
			{
				low_linearly = 0; 
				high_linearly = 1;
				const_goods = 3;
				num_goods_type = CONSTANT;
				pricing_type = FIXED_RANDOM;
			}

			// L4
			else if (!stricmp(in,"L4"))
			{
				low_linearly = 0;
				high_linearly = 1;
				alpha = 0.55;
				num_goods_type = DECAY;
				pricing_type = LINEAR_RANDOM;
			}

			// L4a
			else if (!stricmp(in,"L4a"))
			{
				low_linearly = 1;
				high_linearly = 1000;
				alpha = 0.55;
				num_goods_type = DECAY;
				pricing_type = LINEAR_RANDOM;
			}

			// L5
			else if (!stricmp(in,"L5"))
			{
				mu_goods = 4;
				sigma_goods = 1;
				mu_price = 16;
				sigma_price = 3;
				num_goods_type = NORMAL_GOODS;
				pricing_type = NORMAL_PRICE;
			}

			// L6
			else if (!stricmp(in,"L6"))
			{
				low_linearly = 0.5;
				high_linearly = 1.5;
				num_goods_type = EXPONENTIAL;
				pricing_type = LINEAR_RANDOM;
			}

			// L6a
			else if (!stricmp(in,"L6a"))
			{
				low_linearly = 500;
				high_linearly = 1500;
				num_goods_type = EXPONENTIAL;
				pricing_type = LINEAR_RANDOM;
			}

			// L7
			else if (!stricmp(in,"L7"))
			{
				low_linearly = 0.5;
				high_linearly = 1.5;
				binom_cutoff = 0.2;
				num_goods_type = BINOMIAL;
				pricing_type = LINEAR_RANDOM;
			}

			// L7a
			else if (!stricmp(in,"L7a"))
			{
				low_linearly = 500;
				high_linearly = 1500;
				binom_cutoff = 0.2;
				num_goods_type = BINOMIAL;
				pricing_type = LINEAR_RANDOM;
			}

			// L8
			else if (!stricmp(in,"L8"))
			{
				const_goods = 3;
				num_goods_type = CONSTANT;
				pricing_type = QUADRATIC;
			}
		}
	}
	
	if (p.random_parameters)
	{
		// L3
		const_goods = Param::LRand(3,10);
		
		// L4
		alpha = Param::DRand(0.5,0.95);
		
		// L5
		mu_goods = Param::LRand(2,10);
		sigma_goods = Param::DRand(0.0001,mu_goods);
		sigma_price = Param::DRand(0.0001,2*mu_price);
		
		// L6
		q = 1.0 / Param::DRand(0.5,1);
		
		// L7
		binom_cutoff = Param::DRand(0.005, 0.4);
		
		double dev = Param::DRand(0,1);
		
		// price deviations
		high_linearly = 1 + dev;
		low_linearly = 1 - dev;
		low_fixed = 1.0 - dev;
		high_fixed = 1.0;
	}
	
}

void Legacy::usage()
{
// 80 cols:  12345678901234567890123456789012345678901234567890123456789012345678901234567890
	int count = printf ("Usage for Legacy Distribution:\n");
	for (int t=0;t<count;t++) printf ("=");

	printf ("\n\nNumber of Goods:\n");
	printf ("  -num_goods_type [5]: {1=Binom.;2=Expon.;3=Random;4=Constant;5=Decay;6=Normal}\n");
	printf ("  -binom_cutoff [0.2]: probability of adding each good under binomial\n");
	printf ("  -q [5]: parameter for exponential\n");
	printf ("  -const_goods [3]: number of constant goods\n");
	printf ("  -alpha [0.55]: probability of adding another good for decay\n");
	printf ("  -mu_goods [4]: mean for normally distributed goods\n");
	printf ("  -sigma_goods [1]: stdev for normally distributed goods\n");
	
	printf ("\nPricing:\n");
	printf ("  -pricing_type [2]: {1=FixedRandom; 2=LinearRandom; 3=Normal; 4=Quadratic}\n");
	printf ("  -low_fixed [0]: smallest fixed value\n");
	printf ("  -high_fixed [1000]: largest fixed value\n");
	printf ("  -low_linearly [1]: smallest linear value\n");
	printf ("  -high_linearly [1000]: largest linear value\n");
	printf ("  -mu_price [16]: mean for normally distributed prices\n");
	printf ("  -sigma_price [3]: stdev for normally distributed prices\n");
	
	printf ("\nPredefined Legacy Distributions:\n(These can be used instead of the above legacy-specific parameters.)\n");
	printf ("  -L1:  Random; FixedRandom, low_fixed=0, high_fixed=1\n");
	printf ("  -L1a: Random; FixedRandom, low_fixed=0, high_fixed=1000\n");
	printf ("  -L2:  Random; LinearRandom, low_linearly=0, high_linearly=1\n");
	printf ("  -L2a: Random; LinearRandom, low_linearly=500, high_linearly=1500\n");
	printf ("  -L3:  Constant, const_goods=3; FixedRandom, low_fixed=0, high_fixed=1\n"); 
	printf ("  -L4:  Decay, alpha=0.55; LinearRandom, low_linearly=0, high_linearly=1\n");
	printf ("  -L4a: Decay, alpha=0.55; LinearRandom, low_linearly=1, high_linearly=1000\n");
	printf ("  -L5:  Normal, mu_goods=4, sigma_goods=1; Normal, mu_price=16, sigma_price=3\n");
	printf ("  -L6:  Exponential, q=5; LinearRandom, low_linearly=0.5, high_linearly=1.5\n");
	printf ("  -L6a: Exponential, q=5; LinearRandom, low_linearly=500, high_linearly=1500\n");
	printf ("  -L7:  Binom,binom_cutoff=0.2;LinearRandom,low_linearly=0.5,high_linearly=1.5\n");
	printf ("  -L7a: Binom,binom_cutoff=0.2;LinearRandom,low_linearly=500,high_linearly=1500\n");
	printf ("  -L8:  Constant, const_goods=3; Quadratic [not implemented at this time]\n");
//
//	printed_usage = true;
}

Legacy::~Legacy()
{
  if (norm) delete norm;
  if (output_buffer) delete[] output_buffer;
}

// generate the set of bids
BidSet *Legacy::generate(Param &p)
{
	BidSet *bidset = new BidSet(p.num_goods, p.num_bids);

	// keep adding bids until the right number has been generated
	for (int t=1; bidset->numBids() < p.num_bids; t++)
	{
		int num = generateNumGoods(p);
		double price = generatePrice(num,p);
		Bid *bid = generateBid(num,p.num_goods,price);  // this selects num goods uniformly without replacement
		bidset->add(bid, p.remove_dominated_bids);
		
		if (p.output_parameter_settings && (!(t%p.output_frequency) || bidset->numBids() == p.num_bids))
		{
			int count = printf ("Number of %sbids: %d  Number of tries: %d",
				(p.remove_dominated_bids?"non-dominated ":""),bidset->numBids(),t);
			for (int counter = 0; counter < count; counter++)
				printf ("\b");
		}

#ifdef OUTPUT_DOMINATION_DATA
			if (t >= 35000) break;
#endif
	}

	if (p.output_parameter_settings) printf ("\n");

	return bidset;
}

// Decide how many goods to include in the bid
int Legacy::generateNumGoods(Param &p)
{
	int num_goods = 0;
	
	switch (num_goods_type) {
		
	// binomial
	case BINOMIAL:
	{
		do
		{
			num_goods = 0;
			for (int j = 0; j < p.num_goods; j++) 
			{
				double r = Param::DRand();
				if (r < binom_cutoff) 
					num_goods++;
			}
		} while (num_goods == 0);
		break;
	}
	
	// exponential
	case EXPONENTIAL:
	{
		// a normal exponential distribution is num_goods = - 1/lambda * ln(1-rand()))
		// here we call 1/lambda q, and we make sure that num_goods < p->num_goods
		// note that "log" in c is actually "ln"
		do num_goods = 1 + (int)floor(- q * log(1.0 - (1.0 - exp(- p.num_goods/q))*Param::DRand()));
		while (num_goods == 0);
		break;
	}
	
	// Sandholm Random: pick a random number of unique goods
	case RANDOM:
	{
		do num_goods = (int)(Param::DRand() * p.num_goods) + 1;
		while (num_goods > p.num_goods || num_goods == 0);  // I think this will never happen anyway
		break; 
	}
	
	// Sandholm Uniform: always the same number of goods
	case CONSTANT:
	{	
		num_goods = const_goods;
		break;
	}
		
	// Sandholm Decay: keep adding more goods with probability p
	case DECAY:
	{
		do for (num_goods=0; num_goods < p.num_goods && Param::DRand() <= alpha; num_goods++);
		while (num_goods == 0);
		break;
	}

	// normal
	case NORMAL_GOODS:
	{
		do num_goods = Param::Round(norm->draw(mu_goods,sigma_goods));
		while (num_goods == 0 || num_goods > p.num_goods);
		break;
	}
	}
	
	return num_goods;
}

// generate a price for the bid
double Legacy::generatePrice(int n, const Param &p)
{
	double price=0.0;
	
	switch (pricing_type)
	{
	case FIXED_RANDOM:
	{
		price = Param::DRand() * (high_fixed - low_fixed) + low_fixed;
		break;
	}
	case LINEAR_RANDOM:
	{
		price = Param::DRand() * n * (high_linearly - low_linearly) + low_linearly * n;
		break;
	}
	case NORMAL_PRICE:
	{
		price = norm->draw(mu_price,sigma_price);
		break;
	}
	case QUADRATIC:
	{
		// needs a concept of bidders.  I'm not implementing this one yet
		break;
	}
	}

	return price;
}


// return a bid with n goods and price p
Bid *Legacy::generateBid(int goods_in_bid, int total_goods, double price)
{
	Bid *temp = new Bid(price);
	
	bool *assigned = new bool[total_goods];
	int n;

	for (n=0; n<total_goods; n++) assigned[n] = false;

	for (n = goods_in_bid; n > 0; n--) 
	{
		int g = Param::LRand(0,total_goods-1);
		while (assigned[g])
			g = Param::LRand(0,total_goods-1);

		temp->addGood(g);
		assigned[g] = true;
	}

	delete[] assigned;

	return temp;
}

char *Legacy::outputSettings(bool tofile)
{
	const char *goods[] = {"","Binomial","Exponential","Random","Constant","Decay","Normal"};
	const char *pricing[] = {"","Fixed Random","Linear Random","Normal","Quadratic"};
	char comment[3];
	if (tofile) {comment[0] = '%'; comment[1] = ' '; comment[2] = 0;}
	else comment[0] = 0;
	char buffer1[150],buffer2[100],buffer3[100],buffer4[100];

	sprintf (buffer1,"%sLegacy distribution parameters:\n%sNumber of goods chosen according to %s distribution",
		comment,comment,goods[num_goods_type]);
	switch(num_goods_type)
	{
	case BINOMIAL:
		sprintf (buffer2," (binom_cutoff = %g)",binom_cutoff);
		break;
	case EXPONENTIAL:
		sprintf (buffer2," (q = %g)",q);
		break;
	case CONSTANT:
		sprintf (buffer2," (const_goods = %d)",const_goods);
		break;
	case DECAY:
		sprintf (buffer2," (alpha = %g)",alpha);
		break;
	case NORMAL_GOODS:
		sprintf (buffer2," (mu_goods = %g, sigma_goods = %g)",mu_goods,sigma_goods);
		break;
	default:
		buffer2[0] = 0;
	}
	sprintf (buffer3,".\n%sPrice chosen according to %s distribution", comment, pricing[pricing_type]);
	switch(pricing_type)
	{
	case FIXED_RANDOM:
		sprintf (buffer4," (low_fixed = %g, high_fixed = %g)",low_fixed,high_fixed);
		break;
	case LINEAR_RANDOM:
		sprintf (buffer4," (low_linearly = %g, high_linearly = %g)",low_linearly, high_linearly);
		break;
	case NORMAL_PRICE:
		sprintf (buffer4," (mu_price = %g, sigma_price = %g)",mu_price,sigma_price);
		break;
	default:
		buffer4[0] = 0;
	}
	
	// put it all together

	if (output_buffer)
	  delete[] output_buffer;

	output_buffer = new char[strlen(buffer1)+strlen(buffer2)+strlen(buffer3)+strlen(buffer4)+20];

	sprintf(output_buffer,"%s%s%s%s%s%s.\n",buffer1,(tofile || !buffer2[0] ?"":"\n  "),buffer2,buffer3,
		(tofile || !buffer4[0] ?"":"\n  "),buffer4);

	return output_buffer;
}
