/* Polynomial Optics
 * (C) 2012, Matthias Hullin <hullin@cs.ubc.ca>
 * (University of British Columbia)
 *
 * Feel free to do what you want with this code, as long as this 
 * header remains intact. Needless to say, we appreciate proper 
 * attribution, be it through citation of our EGSR 2012 paper, movie 
 * credits, or otherwise.
 *
 * http://www.cs.ubc.ca/labs/imager/tr/2012/PolynomialOptics/
 * 
 * ==================================
 * OpticalElements/OpticalMaterial.hh
 * ==================================
 * Defines class OpticalMaterial with database lookup by name and ior/Abbe no.
 * Function get_index(float lambda) returns refractive index at 
 * lambda in nm.
 * Data retrieved from Schott, Hoya, Ohara and Newport Glass.
 */


#ifndef OpticalMaterial_hh
#define OpticalMaterial_hh

#include <cmath>
#include <cstring>
#include <stdio.h>

#define __sqr(x) ((x)*(x))

enum DispersionModel {
  mod_sellmeier, 
  mod_reciprocal, // Not implemented
  mod_abbe        // Not implemented
}; 

typedef struct {
  char name[16];
  float nd;
  float vd;
  float B1, B2, B3;
  float C1, C2, C3;
  char mfg[16];
  DispersionModel model;
} GlassDef;



const GlassDef glass_database[] =  
  // Obtained from:
  // http://www.us.schott.com/advanced_optics/english/download/schott_optical_glass_catalogue_excel_december_2011.xls
  // http://www.oharacorp.com/xls/glass-catalog-2011-03-14.xls
  // http://www.hoyaoptics.com/pdf/MasterOpticalGlass.xls (converted to Sellmeier coefficients by looking up Ohara substitute types)
  { 
    {"F2",	1.62004,	36.37,	1.34533359E+00,	2.09073176E-01,	9.37357162E-01,	9.97743871E-03,	4.70450767E-02,	1.11886764E+02,	"Schott",	mod_sellmeier},
    {"F2HT",	1.62004,	36.37,	1.34533359E+00,	2.09073176E-01,	9.37357162E-01,	9.97743871E-03,	4.70450767E-02,	1.11886764E+02,	"Schott",	mod_sellmeier},
    {"F5",	1.60342,	38.03,	1.31044630E+00,	1.96034260E-01,	9.66129770E-01,	9.58633048E-03,	4.57627627E-02,	1.15011883E+02,	"Schott",	mod_sellmeier},
    {"K10",	1.50137,	56.41,	1.15687082E+00,	6.42625444E-02,	8.72376139E-01,	8.09424251E-03,	3.86051284E-02,	1.04747730E+02,	"Schott",	mod_sellmeier},
    {"K7",	1.51112,	60.41,	1.12735550E+00,	1.24412303E-01,	8.27100531E-01,	7.20341707E-03,	2.69835916E-02,	1.00384588E+02,	"Schott",	mod_sellmeier},
    {"KZFS12",	1.69600,	36.29,	1.55624873E+00,	2.39769276E-01,	9.47887658E-01,	1.02012744E-02,	4.69277969E-02,	6.98370722E+01,	"Schott",	mod_sellmeier},
    {"LAFN7",	1.74950,	34.95,	1.66842615E+00,	2.98512803E-01,	1.07743760E+00,	1.03159999E-02,	4.69216348E-02,	8.25078509E+01,	"Schott",	mod_sellmeier},
    {"LF5",	1.58144,	40.85,	1.28035628E+00,	1.63505973E-01,	8.93930112E-01,	9.29854416E-03,	4.49135769E-02,	1.10493685E+02,	"Schott",	mod_sellmeier},
    {"LLF1",	1.54814,	45.75,	1.21640125E+00,	1.33664540E-01,	8.83399468E-01,	8.57807248E-03,	4.20143003E-02,	1.07593060E+02,	"Schott",	mod_sellmeier},
    {"N-BAF10",	1.67003,	47.11,	1.58514950E+00,	1.43559385E-01,	1.08521269E+00,	9.26681282E-03,	4.24489805E-02,	1.05613573E+02,	"Schott",	mod_sellmeier},
    {"N-BAF4",	1.60568,	43.72,	1.42056328E+00,	1.02721269E-01,	1.14380976E+00,	9.42015382E-03,	5.31087291E-02,	1.10278856E+02,	"Schott",	mod_sellmeier},
    {"N-BAF51",	1.65224,	44.96,	1.51503623E+00,	1.53621958E-01,	1.15427909E+00,	9.42734715E-03,	4.30826500E-02,	1.24889868E+02,	"Schott",	mod_sellmeier},
    {"N-BAF52",	1.60863,	46.60,	1.43903433E+00,	9.67046052E-02,	1.09875818E+00,	9.07800128E-03,	5.08212080E-02,	1.05691856E+02,	"Schott",	mod_sellmeier},
    {"N-BAK1",	1.57250,	57.55,	1.12365662E+00,	3.09276848E-01,	8.81511957E-01,	6.44742752E-03,	2.22284402E-02,	1.07297751E+02,	"Schott",	mod_sellmeier},
    {"N-BAK2",	1.53996,	59.71,	1.01662154E+00,	3.19903051E-01,	9.37232995E-01,	5.92383763E-03,	2.03828415E-02,	1.13118417E+02,	"Schott",	mod_sellmeier},
    {"N-BAK4",	1.56883,	55.98,	1.28834642E+00,	1.32817724E-01,	9.45395373E-01,	7.79980626E-03,	3.15631177E-02,	1.05965875E+02,	"Schott",	mod_sellmeier},
    {"N-BALF4",	1.57956,	53.87,	1.31004128E+00,	1.42038259E-01,	9.64929351E-01,	7.96596450E-03,	3.30672072E-02,	1.09197320E+02,	"Schott",	mod_sellmeier},
    {"N-BALF5",	1.54739,	53.63,	1.28385965E+00,	7.19300942E-02,	1.05048927E+00,	8.25815975E-03,	4.41920027E-02,	1.07097324E+02,	"Schott",	mod_sellmeier},
    {"N-BASF2",	1.66446,	36.00,	1.53652081E+00,	1.56971102E-01,	1.30196815E+00,	1.08435729E-02,	5.62278762E-02,	1.31339700E+02,	"Schott",	mod_sellmeier},
    {"N-BASF64",1.70400,	39.38,	1.65554268E+00,	1.71319770E-01,	1.33664448E+00,	1.04485644E-02,	4.99394756E-02,	1.18961472E+02,	"Schott",	mod_sellmeier},
    {"N-BK10",	1.49782,	66.95,	8.88308131E-01,	3.28964475E-01,	9.84610769E-01,	5.16900822E-03,	1.61190045E-02,	9.97575331E+01,	"Schott",	mod_sellmeier},
    {"N-BK7",	1.51680,	64.17,	1.03961212E+00,	2.31792344E-01,	1.01046945E+00,	6.00069867E-03,	2.00179144E-02,	1.03560653E+02,	"Schott",	mod_sellmeier},
    {"N-BK7HT",	1.51680,	64.17,	1.03961212E+00,	2.31792344E-01,	1.01046945E+00,	6.00069867E-03,	2.00179144E-02,	1.03560653E+02,	"Schott",	mod_sellmeier},
    {"N-F2",	1.62005,	36.43,	1.39757037E+00,	1.59201403E-01,	1.26865430E+00,	9.95906143E-03,	5.46931752E-02,	1.19248346E+02,	"Schott",	mod_sellmeier},
    {"N-FK5",	1.48749,	70.41,	8.44309338E-01,	3.44147824E-01,	9.10790213E-01,	4.75111955E-03,	1.49814849E-02,	9.78600293E+01,	"Schott",	mod_sellmeier},
    {"N-FK51A",	1.48656,	84.47,	9.71247817E-01,	2.16901417E-01,	9.04651666E-01,	4.72301995E-03,	1.53575612E-02,	1.68681330E+02,	"Schott",	mod_sellmeier},
    {"N-K5",	1.52249,	59.48,	1.08511833E+00,	1.99562005E-01,	9.30511663E-01,	6.61099503E-03,	2.41108660E-02,	1.11982777E+02,	"Schott",	mod_sellmeier},
    {"N-KF9",	1.52346,	51.54,	1.19286778E+00,	8.93346571E-02,	9.20819805E-01,	8.39154696E-03,	4.04010786E-02,	1.12572446E+02,	"Schott",	mod_sellmeier},
    {"N-KZFS11",1.63775,	42.41,	1.33222450E+00,	2.89241610E-01,	1.15161734E+00,	8.40298480E-03,	3.44239720E-02,	8.84310532E+01,	"Schott",	mod_sellmeier},
    {"N-KZFS2",	1.55836,	54.01,	1.23697554E+00,	1.53569376E-01,	9.03976272E-01,	7.47170505E-03,	3.08053556E-02,	7.01731084E+01,	"Schott",	mod_sellmeier},
    {"N-KZFS4",	1.61336,	44.49,	1.35055424E+00,	1.97575506E-01,	1.09962992E+00,	8.76282070E-03,	3.71767201E-02,	9.03866994E+01,	"Schott",	mod_sellmeier},
    {"N-KZFS5",	1.65412,	39.70,	1.47460789E+00,	1.93584488E-01,	1.26589974E+00,	9.86143816E-03,	4.45477583E-02,	1.06436258E+02,	"Schott",	mod_sellmeier},
    {"N-KZFS8",	1.72047,	34.70,	1.62693651E+00,	2.43698760E-01,	1.62007141E+00,	1.08808630E-02,	4.94207753E-02,	1.31009163E+02,	"Schott",	mod_sellmeier},
    {"N-LAF2",	1.74397,	44.85,	1.80984227E+00,	1.57295550E-01,	1.09300370E+00,	1.01711622E-02,	4.42431765E-02,	1.00687748E+02,	"Schott",	mod_sellmeier},
    {"N-LAF21",	1.78800,	47.49,	1.87134529E+00,	2.50783010E-01,	1.22048639E+00,	9.33322280E-03,	3.45637762E-02,	8.32404866E+01,	"Schott",	mod_sellmeier},
    {"N-LAF33",	1.78582,	44.05,	1.79653417E+00,	3.11577903E-01,	1.15981863E+00,	9.27313493E-03,	3.58201181E-02,	8.73448712E+01,	"Schott",	mod_sellmeier},
    {"N-LAF34",	1.77250,	49.62,	1.75836958E+00,	3.13537785E-01,	1.18925231E+00,	8.72810026E-03,	2.93020832E-02,	8.51780644E+01,	"Schott",	mod_sellmeier},
    {"N-LAF35",	1.74330,	49.40,	1.51697436E+00,	4.55875464E-01,	1.07469242E+00,	7.50943203E-03,	2.60046715E-02,	8.05945159E+01,	"Schott",	mod_sellmeier},
    {"N-LAF36",	1.79952,	42.37,	1.85744228E+00,	2.94098729E-01,	1.16615417E+00,	9.82397191E-03,	3.84309138E-02,	8.93984634E+01,	"Schott",	mod_sellmeier},
    {"N-LAF7",	1.74950,	34.82,	1.74028764E+00,	2.26710554E-01,	1.32525548E+00,	1.07925580E-02,	5.38626639E-02,	1.06268665E+02,	"Schott",	mod_sellmeier},
    {"N-LAK10",	1.72003,	50.62,	1.72878017E+00,	1.69257825E-01,	1.19386956E+00,	8.86014635E-03,	3.63416509E-02,	8.29009069E+01,	"Schott",	mod_sellmeier},
    {"N-LAK12",	1.67790,	55.20,	1.17365704E+00,	5.88992398E-01,	9.78014394E-01,	5.77031797E-03,	2.00401678E-02,	9.54873482E+01,	"Schott",	mod_sellmeier},
    {"N-LAK14",	1.69680,	55.41,	1.50781212E+00,	3.18866829E-01,	1.14287213E+00,	7.46098727E-03,	2.42024834E-02,	8.09565165E+01,	"Schott",	mod_sellmeier},
    {"N-LAK21",	1.64049,	60.10,	1.22718116E+00,	4.20783743E-01,	1.01284843E+00,	6.02075682E-03,	1.96862889E-02,	8.84370099E+01,	"Schott",	mod_sellmeier},
    {"N-LAK22",	1.65113,	55.89,	1.14229781E+00,	5.35138441E-01,	1.04088385E+00,	5.85778594E-03,	1.98546147E-02,	1.00834017E+02,	"Schott",	mod_sellmeier},
    {"N-LAK33A",1.75393,	52.27,	1.44116999E+00,	5.71749501E-01,	1.16605226E+00,	6.80933877E-03,	2.22291824E-02,	8.09379555E+01,	"Schott",	mod_sellmeier},
    {"N-LAK33B",1.75500,	52.30,	1.42288601E+00,	5.93661336E-01,	1.16135260E+00,	6.70283452E-03,	2.19416210E-02,	8.07407701E+01,	"Schott",	mod_sellmeier},
    {"N-LAK34",	1.72916,	54.50,	1.26661442E+00,	6.65919318E-01,	1.12496120E+00,	5.89278062E-03,	1.97509041E-02,	7.88894174E+01,	"Schott",	mod_sellmeier},
    {"N-LAK7",	1.65160,	58.52,	1.23679889E+00,	4.45051837E-01,	1.01745888E+00,	6.10105538E-03,	2.01388334E-02,	9.06380380E+01,	"Schott",	mod_sellmeier},
    {"N-LAK8",	1.71300,	53.83,	1.33183167E+00,	5.46623206E-01,	1.19084015E+00,	6.20023871E-03,	2.16465439E-02,	8.25827736E+01,	"Schott",	mod_sellmeier},
    {"N-LAK9",	1.69100,	54.71,	1.46231905E+00,	3.44399589E-01,	1.15508372E+00,	7.24270156E-03,	2.43353131E-02,	8.54686868E+01,	"Schott",	mod_sellmeier},
    {"N-LASF31A",1.88300,	40.76,	1.96485075E+00,	4.75231259E-01,	1.48360109E+00,	9.82060155E-03,	3.44713438E-02,	1.10739863E+02,	"Schott",	mod_sellmeier},
    {"N-LASF40",1.83404,	37.30,	1.98550331E+00,	2.74057042E-01,	1.28945661E+00,	1.09583310E-02,	4.74551603E-02,	9.69085286E+01,	"Schott",	mod_sellmeier},
    {"N-LASF41",1.83501,	43.13,	1.86348331E+00,	4.13307255E-01,	1.35784815E+00,	9.10368219E-03,	3.39247268E-02,	9.33580595E+01,	"Schott",	mod_sellmeier},
    {"N-LASF43",1.80610,	40.61,	1.93502827E+00,	2.36629350E-01,	1.26291344E+00,	1.04001413E-02,	4.47505292E-02,	8.74375690E+01,	"Schott",	mod_sellmeier},
    {"N-LASF44",1.80420,	46.50,	1.78897105E+00,	3.86758670E-01,	1.30506243E+00,	8.72506277E-03,	3.08085023E-02,	9.27743824E+01,	"Schott",	mod_sellmeier},
    {"N-LASF45",1.80107,	34.97,	1.87140198E+00,	2.67777879E-01,	1.73030008E+00,	1.12171920E-02,	5.05134972E-02,	1.47106505E+02,	"Schott",	mod_sellmeier},
    {"N-LASF45HT",1.80107,	34.97,	1.87140198E+00,	2.67777879E-01,	1.73030008E+00,	1.12171920E-02,	5.05134972E-02,	1.47106505E+02,	"Schott",	mod_sellmeier},
    {"N-LASF46A",1.90366,	31.32,	2.16701566E+00,	3.19812761E-01,	1.66004486E+00,	1.23595524E-02,	5.60610282E-02,	1.07047718E+02,	"Schott",	mod_sellmeier},
    {"N-LASF46B",1.90366,	31.32,	2.17988922E+00,	3.06495184E-01,	1.56882437E+00,	1.25805384E-02,	5.67191367E-02,	1.05316538E+02,	"Schott",	mod_sellmeier},
    {"N-LASF9",	1.85025,	32.17,	2.00029547E+00,	2.98926886E-01,	1.80691843E+00,	1.21426017E-02,	5.38736236E-02,	1.56530829E+02,	"Schott",	mod_sellmeier},
    {"N-LASF9HT",1.85025,	32.17,	2.00029547E+00,	2.98926886E-01,	1.80691843E+00,	1.21426017E-02,	5.38736236E-02,	1.56530829E+02,	"Schott",	mod_sellmeier},
    {"N-PK51",	1.52855,	76.98,	1.15610775E+00,	1.53229344E-01,	7.85618966E-01,	5.85597402E-03,	1.94072416E-02,	1.40537046E+02,	"Schott",	mod_sellmeier},
    {"N-PK52A",	1.49700,	81.61,	1.02960700E+00,	1.88050600E-01,	7.36488165E-01,	5.16800155E-03,	1.66658798E-02,	1.38964129E+02,	"Schott",	mod_sellmeier},
    {"N-PSK3",	1.55232,	63.46,	8.87272110E-01,	4.89592425E-01,	1.04865296E+00,	4.69824067E-03,	1.61818463E-02,	1.04374975E+02,	"Schott",	mod_sellmeier},
    {"N-PSK53A",1.61800,	63.39,	1.38121836E+00,	1.96745645E-01,	8.86089205E-01,	7.06416337E-03,	2.33251345E-02,	9.74847345E+01,	"Schott",	mod_sellmeier},
    {"N-SF1",	1.71736,	29.62,	1.60865158E+00,	2.37725916E-01,	1.51530653E+00,	1.19654879E-02,	5.90589722E-02,	1.35521676E+02,	"Schott",	mod_sellmeier},
    {"N-SF10",	1.72828,	28.53,	1.62153902E+00,	2.56287842E-01,	1.64447552E+00,	1.22241457E-02,	5.95736775E-02,	1.47468793E+02,	"Schott",	mod_sellmeier},
    {"N-SF11",	1.78472,	25.68,	1.73759695E+00,	3.13747346E-01,	1.89878101E+00,	1.31887070E-02,	6.23068142E-02,	1.55236290E+02,	"Schott",	mod_sellmeier},
    {"N-SF14",	1.76182,	26.53,	1.69022361E+00,	2.88870052E-01,	1.70451870E+00,	1.30512113E-02,	6.13691880E-02,	1.49517689E+02,	"Schott",	mod_sellmeier},
    {"N-SF15",	1.69892,	30.20,	1.57055634E+00,	2.18987094E-01,	1.50824017E+00,	1.16507014E-02,	5.97856897E-02,	1.32709339E+02,	"Schott",	mod_sellmeier},
    {"N-SF2",	1.64769,	33.82,	1.47343127E+00,	1.63681849E-01,	1.36920899E+00,	1.09019098E-02,	5.85683687E-02,	1.27404933E+02,	"Schott",	mod_sellmeier},
    {"N-SF4",	1.75513,	27.38,	1.67780282E+00,	2.82849893E-01,	1.63539276E+00,	1.26793450E-02,	6.02038419E-02,	1.45760496E+02,	"Schott",	mod_sellmeier},
    {"N-SF5",	1.67271,	32.25,	1.52481889E+00,	1.87085527E-01,	1.42729015E+00,	1.12547560E-02,	5.88995392E-02,	1.29141675E+02,	"Schott",	mod_sellmeier},
    {"N-SF57",	1.84666,	23.78,	1.87543831E+00,	3.73757490E-01,	2.30001797E+00,	1.41749518E-02,	6.40509927E-02,	1.77389795E+02,	"Schott",	mod_sellmeier},
    {"N-SF57HT",1.84666,	23.78,	1.87543831E+00,	3.73757490E-01,	2.30001797E+00,	1.41749518E-02,	6.40509927E-02,	1.77389795E+02,	"Schott",	mod_sellmeier},
    {"N-SF57HTULTRA",1.84666,	23.78,	1.87543831E+00,	3.73757490E-01,	2.30001797E+00,	1.41749518E-02,	6.40509927E-02,	1.77389795E+02,	"Schott",	mod_sellmeier},
    {"N-SF6",	1.80518,	25.36,	1.77931763E+00,	3.38149866E-01,	2.08734474E+00,	1.33714182E-02,	6.17533621E-02,	1.74017590E+02,	"Schott",	mod_sellmeier},
    {"N-SF66",	1.92286,	20.88,	2.02459760E+00,	4.70187196E-01,	2.59970433E+00,	1.47053225E-02,	6.92998276E-02,	1.61817601E+02,	"Schott",	mod_sellmeier},
    {"N-SF6HT",	1.80518,	25.36,	1.77931763E+00,	3.38149866E-01,	2.08734474E+00,	1.33714182E-02,	6.17533621E-02,	1.74017590E+02,	"Schott",	mod_sellmeier},
    {"N-SF6HTULTRA",1.80518,	25.36,	1.77931763E+00,	3.38149866E-01,	2.08734474E+00,	1.33714182E-02,	6.17533621E-02,	1.74017590E+02,	"Schott",	mod_sellmeier},
    {"N-SF8",	1.68894,	31.31,	1.55075812E+00,	2.09816918E-01,	1.46205491E+00,	1.14338344E-02,	5.82725652E-02,	1.33241650E+02,	"Schott",	mod_sellmeier},
    {"N-SK11",	1.56384,	60.80,	1.17963631E+00,	2.29817295E-01,	9.35789652E-01,	6.80282081E-03,	2.19737205E-02,	1.01513232E+02,	"Schott",	mod_sellmeier},
    {"N-SK14",	1.60311,	60.60,	9.36155374E-01,	5.94052018E-01,	1.04374583E+00,	4.61716525E-03,	1.68859270E-02,	1.03736265E+02,	"Schott",	mod_sellmeier},
    {"N-SK16",	1.62041,	60.32,	1.34317774E+00,	2.41144399E-01,	9.94317969E-01,	7.04687339E-03,	2.29005000E-02,	9.27508526E+01,	"Schott",	mod_sellmeier},
    {"N-SK2",	1.60738,	56.65,	1.28189012E+00,	2.57738258E-01,	9.68186040E-01,	7.27191640E-03,	2.42823527E-02,	1.10377773E+02,	"Schott",	mod_sellmeier},
    {"N-SK2HT",	1.60738,	56.65,	1.28189012E+00,	2.57738258E-01,	9.68186040E-01,	7.27191640E-03,	2.42823527E-02,	1.10377773E+02,	"Schott",	mod_sellmeier},
    {"N-SK4",	1.61272,	58.63,	1.32993741E+00,	2.28542996E-01,	9.88465211E-01,	7.16874107E-03,	2.46455892E-02,	1.00886364E+02,	"Schott",	mod_sellmeier},
    {"N-SK5",	1.58913,	61.27,	9.91463823E-01,	4.95982121E-01,	9.87393925E-01,	5.22730467E-03,	1.72733646E-02,	9.83594579E+01,	"Schott",	mod_sellmeier},
    {"N-SSK2",	1.62229,	53.27,	1.43060270E+00,	1.53150554E-01,	1.01390904E+00,	8.23982975E-03,	3.33736841E-02,	1.06870822E+02,	"Schott",	mod_sellmeier},
    {"N-SSK5",	1.65844,	50.88,	1.59222659E+00,	1.03520774E-01,	1.05174016E+00,	9.20284626E-03,	4.23530072E-02,	1.06927374E+02,	"Schott",	mod_sellmeier},
    {"N-SSK8",	1.61773,	49.83,	1.44857867E+00,	1.17965926E-01,	1.06937528E+00,	8.69310149E-03,	4.21566593E-02,	1.11300666E+02,	"Schott",	mod_sellmeier},
    {"N-ZK7",	1.50847,	61.19,	1.07715032E+00,	1.68079109E-01,	8.51889892E-01,	6.76601657E-03,	2.30642817E-02,	8.90498778E+01,	"Schott",	mod_sellmeier},
    {"P-LAF37",	1.75550,	45.66,	1.76003244E+00,	2.48286745E-01,	1.15935122E+00,	9.38006396E-03,	3.60537464E-02,	8.64324693E+01,	"Schott",	mod_sellmeier},
    {"P-LAK35",	1.69350,	53.20,	1.39324260E+00,	4.18882766E-01,	1.04380700E+00,	7.15959695E-03,	2.33637446E-02,	8.83284426E+01,	"Schott",	mod_sellmeier},
    {"P-LASF47",1.80610,	40.90,	1.85543101E+00,	3.15854649E-01,	1.28561839E+00,	1.00328203E-02,	3.87095168E-02,	9.45421507E+01,	"Schott",	mod_sellmeier},
    {"P-LASF50",1.80860,	40.46,	1.84910553E+00,	3.29828674E-01,	1.30400901E+00,	9.99234757E-03,	3.87437988E-02,	9.58967681E+01,	"Schott",	mod_sellmeier},
    {"P-LASF51",1.81000,	40.93,	1.84568806E+00,	3.39001600E-01,	1.32418921E+00,	9.88495571E-03,	3.78097402E-02,	9.78415430E+01,	"Schott",	mod_sellmeier},
    {"P-PK53",	1.52690,	66.22,	9.60316767E-01,	3.40437227E-01,	7.77865595E-01,	5.31032986E-03,	1.75073434E-02,	1.06875330E+02,	"Schott",	mod_sellmeier},
    {"P-SF67",	1.90680,	21.40,	1.97464225E+00,	4.67095921E-01,	2.43154209E+00,	1.45772324E-02,	6.69790359E-02,	1.57444895E+02,	"Schott",	mod_sellmeier},
    {"P-SF68",	2.00520,	21.00,	2.33300670E+00,	4.52961396E-01,	1.25172339E+00,	1.68838419E-02,	7.16086325E-02,	1.18707479E+02,	"Schott",	mod_sellmeier},
    {"P-SF69",	1.72250,	29.23,	1.62594647E+00,	2.35927609E-01,	1.67434623E+00,	1.21696677E-02,	6.00710405E-02,	1.45651908E+02,	"Schott",	mod_sellmeier},
    {"P-SF8",	1.68893,	31.25,	1.55370411E+00,	2.06332561E-01,	1.39708831E+00,	1.16582670E-02,	5.82087757E-02,	1.30748028E+02,	"Schott",	mod_sellmeier},
    {"P-SK57",	1.58700,	59.60,	1.31053414E+00,	1.69376189E-01,	1.10987714E+00,	7.40877235E-03,	2.54563489E-02,	1.07751087E+02,	"Schott",	mod_sellmeier},
    {"P-SK58A",	1.58913,	61.15,	1.31678410E+00,	1.71154756E-01,	1.12501473E+00,	7.20717498E-03,	2.45659595E-02,	1.02739728E+02,	"Schott",	mod_sellmeier},
    {"P-SK60",	1.61035,	57.90,	1.40790442E+00,	1.43381417E-01,	1.16513947E+00,	7.84382378E-03,	2.87769365E-02,	1.05373397E+02,	"Schott",	mod_sellmeier},
    {"SF1",	1.71736,	29.51,	1.55912923E+00,	2.84246288E-01,	9.68842926E-01,	1.21481001E-02,	5.34549042E-02,	1.12174809E+02,	"Schott",	mod_sellmeier},
    {"SF10",	1.72825,	28.41,	1.61625977E+00,	2.59229334E-01,	1.07762317E+00,	1.27534559E-02,	5.81983954E-02,	1.16607680E+02,	"Schott",	mod_sellmeier},
    {"SF2",	1.64769,	33.85,	1.40301821E+00,	2.31767504E-01,	9.39056586E-01,	1.05795466E-02,	4.93226978E-02,	1.12405955E+02,	"Schott",	mod_sellmeier},
    {"SF4",	1.75520,	27.58,	1.61957826E+00,	3.39493189E-01,	1.02566931E+00,	1.25502104E-02,	5.44559822E-02,	1.17652222E+02,	"Schott",	mod_sellmeier},
    {"SF5",	1.67270,	32.21,	1.46141885E+00,	2.47713019E-01,	9.49995832E-01,	1.11826126E-02,	5.08594669E-02,	1.12041888E+02,	"Schott",	mod_sellmeier},
    {"SF56A",	1.78470,	26.08,	1.70579259E+00,	3.44223052E-01,	1.09601828E+00,	1.33874699E-02,	5.79561608E-02,	1.21616024E+02,	"Schott",	mod_sellmeier},
    {"SF57",	1.84666,	23.83,	1.81651371E+00,	4.28893641E-01,	1.07186278E+00,	1.43704198E-02,	5.92801172E-02,	1.21419942E+02,	"Schott",	mod_sellmeier},
    {"SF6",	1.80518,	25.43,	1.72448482E+00,	3.90104889E-01,	1.04572858E+00,	1.34871947E-02,	5.69318095E-02,	1.18557185E+02,	"Schott",	mod_sellmeier},
    {"SF6HT",	1.80518,	25.43,	1.72448482E+00,	3.90104889E-01,	1.04572858E+00,	1.34871947E-02,	5.69318095E-02,	1.18557185E+02,	"Schott",	mod_sellmeier},
    {"SF57HTULTRA",1.84666,	23.83,	1.81651371E+00,	4.28893641E-01,	1.07186278E+00,	1.43704198E-02,	5.92801172E-02,	1.21419942E+02,	"Schott",	mod_sellmeier},
    {"P-BK7",	1.51640,	64.06,	1.18318503E+00,	8.71756426E-02,	1.03133701E+00,	7.22141956E-03,	2.68216805E-02,	1.01702362E+02,	"Schott",	mod_sellmeier},
    {"P-SK57Q1",1.58600,	59.50,	1.30536483E+00,	1.71434328E-01,	1.10117219E+00,	7.36408831E-03,	2.55786047E-02,	1.06726060E+02,	"Schott",	mod_sellmeier},
    {"SF11",	1.78472,	25.76,	1.73848403E+00,	3.11168974E-01,	1.17490871E+00,	1.36068604E-02,	6.15960463E-02,	1.21922711E+02,	"Schott",	mod_sellmeier},
    {"S-FPL53",	1.438750,	94.93,	9.83532327E-01,	6.95688140E-02,	1.11409238E+00,	4.92234955E-03,	1.93581091E-02,	2.64275294E+02,	"Ohara",	mod_sellmeier},
    {"S-FPL52",	1.455999,	90.28,	1.06785857E+00,	3.35857718E-02,	1.10219763E+00,	6.99227302E-03,	-2.0760893E-02,	2.26496541E+02,	"Ohara",	mod_sellmeier},
    {"S-FSL5",	1.487490,	70.23,	1.17447043E+00,	1.40056154E-02,	1.19272435E+00,	8.41855181E-03,	-5.8179077E-02,	1.29599726E+02,	"Ohara",	mod_sellmeier},
    {"S-FPL51",	1.496999,	81.54,	1.17010505E+00,	4.75710783E-02,	7.63832445E-01,	6.16203924E-03,	2.63372876E-02,	1.41882642E+02,	"Ohara",	mod_sellmeier},
    {"S-FTL10",	1.501372,	56.42,	1.10542993E+00,	1.16184325E-01,	1.15396111E+00,	6.69004168E-03,	3.60840691E-02,	1.37546485E+02,	"Ohara",	mod_sellmeier},
    {"S-BSL7",	1.516330,	64.14,	1.15150190E+00,	1.18583612E-01,	1.26301359E+00,	1.05984130E-02,	-1.1822519E-02,	1.29617662E+02,	"Ohara",	mod_sellmeier},
    {"S-NSL36",	1.517417,	52.43,	1.09666153E+00,	1.68990073E-01,	1.20580827E+00,	6.67491123E-03,	3.36095450E-02,	1.41668738E+02,	"Ohara",	mod_sellmeier},
    {"S-NSL3",	1.518229,	58.90,	8.82514764E-01,	3.89271907E-01,	1.10693448E+00,	4.64504582E-03,	2.00551397E-02,	1.36234339E+02,	"Ohara",	mod_sellmeier},
    {"S-NSL5",	1.522494,	59.84,	1.04574577E+00,	2.39613026E-01,	1.15906850E+00,	5.85232280E-03,	2.36858752E-02,	1.31329061E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL6",	1.531717,	48.84,	1.17701777E+00,	1.27958030E-01,	1.34740124E+00,	7.71087686E-03,	4.11325328E-02,	1.54531692E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL12",	1.539956,	59.46,	7.14605258E-01,	6.21993289E-01,	1.22537681E+00,	3.01763913E-03,	1.66505450E-02,	1.43506314E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL2",	1.540720,	47.23,	1.23401499E+00,	9.59796833E-02,	1.20503991E+00,	8.69507801E-03,	4.65611429E-02,	1.37953301E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL1",	1.548141,	45.79,	1.25088944E+00,	9.97973327E-02,	1.20583504E+00,	8.83921279E-03,	4.82685052E-02,	1.37414953E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL41",	1.563839,	60.67,	1.24344200E+00,	1.66301104E-01,	1.10586114E+00,	1.16396708E-02,	-8.9046494E-03,	1.14111220E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL26",	1.567322,	42.82,	1.31066488E+00,	9.41903094E-02,	1.23292644E+00,	9.68897812E-03,	5.27763106E-02,	1.33296422E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL14",	1.568832,	56.36,	1.27553696E+00,	1.46083393E-01,	1.16754699E+00,	7.49692359E-03,	3.10421530E-02,	1.28947092E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL2",	1.570989,	50.80,	1.30923813E+00,	1.14137353E-01,	1.17882259E+00,	8.38873953E-03,	3.99436485E-02,	1.40257892E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL3",	1.571351,	52.95,	1.29366890E+00,	1.32440252E-01,	1.10197293E+00,	8.00367962E-03,	3.54711196E-02,	1.34517431E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL11",	1.572501,	57.74,	8.21314256E-01,	6.12586478E-01,	1.24859637E+00,	3.51436131E-03,	1.79762375E-02,	1.33456670E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL27",	1.575006,	41.50,	1.31433154E+00,	1.12300168E-01,	1.41390100E+00,	9.50404477E-03,	5.24112772E-02,	1.48429972E+02,	"Ohara",	mod_sellmeier},
    {"S-TIL25",	1.581439,	40.75,	1.32122534E+00,	1.23824976E-01,	1.43685254E+00,	9.52091436E-03,	5.16062665E-02,	1.49064883E+02,	"Ohara",	mod_sellmeier},
    {"S-BAM3",	1.582673,	46.42,	1.36955358E+00,	8.53825867E-02,	1.16159771E+00,	9.41331434E-03,	5.04359027E-02,	1.30548899E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL42",	1.583126,	59.38,	1.39570615E+00,	7.18505070E-02,	1.27129267E+00,	1.12218843E-02,	-2.5211742E-02,	1.34497860E+02,	"Ohara",	mod_sellmeier},
    {"S-BAL35",	1.589130,	61.14,	9.41357273E-01,	5.46174895E-01,	1.16168917E+00,	1.40333996E-02,	9.06635683E-04,	1.14163758E+02,	"Ohara",	mod_sellmeier},
    {"S-FTM16",	1.592701,	35.31,	1.32940907E+00,	1.41512125E-01,	1.44299068E+00,	1.02377287E-02,	5.78081956E-02,	1.50597139E+02,	"Ohara",	mod_sellmeier},
    {"S-FPM2",	1.595220,	67.74,	7.61242785E-01,	7.47033375E-01,	9.38928947E-01,	3.21174095E-03,	1.40234423E-02,	1.39523530E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM8",	1.595509,	39.24,	1.37262713E+00,	1.12636276E-01,	1.39786421E+00,	1.03220068E-02,	5.50195044E-02,	1.47735609E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM5",	1.603420,	38.03,	1.38531342E+00,	1.22372945E-01,	1.40508326E+00,	1.04074567E-02,	5.57440088E-02,	1.44878733E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM14",	1.603112,	60.64,	1.28286270E+00,	2.47647429E-01,	1.10383999E+00,	1.22902399E-02,	-6.1314236E-03,	1.06883378E+02,	"Ohara",	mod_sellmeier},
    {"S-PHM53",	1.603001,	65.44,	1.09775423E+00,	4.34816432E-01,	1.13894976E+00,	1.23369400E-02,	-3.7252290E-04,	1.24276984E+02,	"Ohara",	mod_sellmeier},
    {"S-BAM4",	1.605620,	43.70,	1.41059317E+00,	1.11201306E-01,	1.34148939E+00,	9.63312192E-03,	4.98778210E-02,	1.52237696E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM2",	1.607379,	56.81,	8.67168676E-01,	6.72848343E-01,	1.18456107E+00,	3.69311003E-03,	1.81652804E-02,	1.32376147E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM3",	1.612929,	37.00,	1.40691144E+00,	1.28369745E-01,	1.51826191E+00,	1.05633641E-02,	5.68483105E-02,	1.52107924E+02,	"Ohara",	mod_sellmeier},
    {"S-NBM51",	1.613397,	44.27,	1.37023101E+00,	1.77665568E-01,	1.30515471E+00,	8.71920342E-03,	4.05725552E-02,	1.12703058E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM4",	1.612716,	58.72,	9.62443080E-01,	5.95939234E-01,	1.10558352E+00,	4.68062141E-03,	1.78772082E-02,	1.15896432E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM9",	1.614047,	54.99,	1.37020077E+00,	1.89397267E-01,	1.24202324E+00,	7.57631457E-03,	3.00787515E-02,	1.31350111E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM28",	1.617722,	49.81,	1.43822841E+00,	1.28100017E-01,	1.34355530E+00,	8.59779750E-03,	4.08617854E-02,	1.43709890E+02,	"Ohara",	mod_sellmeier},
    {"S-PHM52",	1.618000,	63.33,	1.09966550E+00,	4.78125422E-01,	1.13214074E+00,	1.32718559E-02,	-6.0164969E-04,	1.30595472E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM2",	1.620041,	36.26,	1.42193846E+00,	1.33827968E-01,	1.45060574E+00,	1.07291511E-02,	5.72587546E-02,	1.45381805E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM16",	1.620411,	60.29,	1.14490383E+00,	4.39563911E-01,	1.27688079E+00,	1.37034916E-02,	-1.8651421E-03,	1.19535585E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM22",	1.622296,	53.17,	1.44305741E+00,	1.40786358E-01,	1.26093951E+00,	8.19208910E-03,	3.56911455E-02,	1.31959337E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM10",	1.622799,	57.05,	9.45443081E-01,	6.43237376E-01,	1.17752968E+00,	1.57263798E-02,	1.61924066E-03,	1.21361748E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM15",	1.622992,	58.16,	9.53128328E-01,	6.37613977E-01,	1.65245647E+00,	3.87638985E-03,	1.85094632E-02,	1.59442367E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM1",	1.625882,	35.70,	1.44963830E+00,	1.22986408E-01,	1.38066723E+00,	1.12094282E-02,	5.96265770E-02,	1.38178326E+02,	"Ohara",	mod_sellmeier},
    {"S-BAM12",	1.639300,	44.87,	1.50161605E+00,	1.26987445E-01,	1.43544052E+00,	9.40761826E-03,	4.72602195E-02,	1.41666499E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM18",	1.638539,	55.38,	9.27886025E-01,	7.08858526E-01,	1.18610897E+00,	4.17549199E-03,	1.84691838E-02,	1.22210416E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM27",	1.639799,	34.46,	1.41680470E+00,	1.96785057E-01,	1.68001322E+00,	1.00732158E-02,	5.37616908E-02,	1.64672436E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM81",	1.639999,	60.08,	9.96356844E-01,	6.51392837E-01,	1.22432622E+00,	1.44821587E-02,	1.54826389E-03,	8.99818604E+01,	"Ohara",	mod_sellmeier},
    {"S-TIM22",	1.647689,	33.79,	1.44222294E+00,	1.94432265E-01,	1.74092482E+00,	1.04249404E-02,	5.50235257E-02,	1.69710769E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM71",	1.648498,	53.02,	1.50847885E+00,	1.58099826E-01,	1.36815368E+00,	8.12769076E-03,	3.54200898E-02,	1.36110038E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL54",	1.650996,	56.16,	1.41910189E+00,	2.58416881E-01,	1.07385537E+00,	7.26647428E-03,	2.63842499E-02,	1.02555463E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL7",	1.651597,	58.55,	9.16121247E-01,	7.65948319E-01,	1.27745023E+00,	3.95889743E-03,	1.67547425E-02,	1.10762706E+02,	"Ohara",	mod_sellmeier},
    {"S-NBH5",	1.654115,	39.68,	1.47544521E+00,	1.93060095E-01,	1.50939010E+00,	9.55836740E-03,	4.60430483E-02,	1.26422746E+02,	"Ohara",	mod_sellmeier},
    {"S-BSM25",	1.658441,	50.88,	1.34814257E+00,	3.47530319E-01,	1.38798368E+00,	6.95364366E-03,	2.77863478E-02,	1.42138122E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM39",	1.666800,	33.05,	1.47008105E+00,	2.24752746E-01,	2.44968592E+00,	1.02900432E-02,	5.41276904E-02,	2.37434940E+02,	"Ohara",	mod_sellmeier},
    {"S-BAH11",	1.666718,	48.32,	1.57138860E+00,	1.47869313E-01,	1.28092846E+00,	9.10807936E-03,	4.02401684E-02,	1.30399367E+02,	"Ohara",	mod_sellmeier},
    {"S-BAH32",	1.669979,	39.27,	1.58023630E+00,	1.37504632E-01,	1.60603298E+00,	1.03578062E-02,	5.48393088E-02,	1.47982885E+02,	"Ohara",	mod_sellmeier},
    {"S-BAH10",	1.670029,	47.23,	1.59034337E+00,	1.38464579E-01,	1.21988043E+00,	9.32734340E-03,	4.27498654E-02,	1.19251777E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM25",	1.672700,	32.10,	1.50659233E+00,	2.04786135E-01,	1.92036668E+00,	1.09501562E-02,	5.74980285E-02,	1.78128535E+02,	"Ohara",	mod_sellmeier},
    {"S-NBH52",	1.673000,	38.15,	1.50305799E+00,	2.21715926E-01,	1.84496391E+00,	9.99021738E-03,	4.50327698E-02,	1.63722302E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL56",	1.677898,	50.72,	1.54052000E+00,	2.17748704E-01,	1.30456122E+00,	8.26765101E-03,	3.28533726E-02,	1.24527479E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL12",	1.677900,	55.34,	9.92053895E-01,	7.71377731E-01,	1.18296264E+00,	1.67095063E-02,	2.36750156E-03,	1.05901080E+02,	"Ohara",	mod_sellmeier},
    {"S-TIM28",	1.688931,	31.07,	1.54270810E+00,	2.17113891E-01,	1.81904459E+00,	1.13925005E-02,	5.79224572E-02,	1.67697189E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL9",	1.691002,	54.82,	1.16195687E+00,	6.44860099E-01,	1.25062221E+00,	1.59659509E-02,	5.05502467E-04,	9.38284169E+01,	"Ohara",	mod_sellmeier},
    {"S-LAL58",	1.693495,	50.81,	1.06368789E+00,	7.44939067E-01,	1.59178942E+00,	1.85199640E-02,	1.16295862E-03,	1.56636025E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL13",	1.693501,	53.21,	9.80071267E-01,	8.32904776E-01,	1.28111995E+00,	3.89123698E-03,	1.89164592E-02,	9.89052676E+01,	"Ohara",	mod_sellmeier},
    {"S-LAM59",	1.697002,	48.52,	1.63056133E+00,	1.86994897E-01,	1.30014289E+00,	8.99690705E-03,	3.68011993E-02,	1.22239544E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL14",	1.696797,	55.53,	1.23720970E+00,	5.89722623E-01,	1.31921880E+00,	1.53551320E-02,	-3.0789625E-04,	9.37202947E+01,	"Ohara",	mod_sellmeier},
    {"S-TIM35",	1.698947,	30.13,	1.55849775E+00,	2.30767007E-01,	1.84436099E+00,	1.15367235E-02,	5.86095947E-02,	1.62981888E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM51",	1.699998,	48.08,	1.63847200E+00,	1.88330533E-01,	1.47502357E+00,	9.04853452E-03,	3.72740173E-02,	1.37770050E+02,	"Ohara",	mod_sellmeier},
    {"S-BAH27",	1.701536,	41.24,	1.68939052E+00,	1.33081013E-01,	1.41165515E+00,	1.03598193E-02,	5.33982239E-02,	1.26515503E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL8",	1.712995,	53.87,	1.30663291E+00,	5.71377253E-01,	1.24303605E+00,	6.11862448E-03,	2.12721470E-02,	9.06285686E+01,	"Ohara",	mod_sellmeier},
    {"S-TIH1",	1.717362,	29.52,	1.60326759E+00,	2.42980935E-01,	1.81313592E+00,	1.18019139E-02,	5.91363658E-02,	1.61218747E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM3",	1.717004,	47.92,	1.64258713E+00,	2.39634610E-01,	1.22483026E+00,	8.68246020E-03,	3.51226242E-02,	1.16604369E+02,	"Ohara",	mod_sellmeier},
    {"S-NBH8",	1.720467,	34.71,	1.61344136E+00,	2.57295888E-01,	1.98364455E+00,	1.06386752E-02,	4.87071624E-02,	1.59784404E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM58",	1.720000,	41.98,	1.70984856E+00,	1.73342897E-01,	1.64833565E+00,	1.00852127E-02,	4.70890831E-02,	1.57468520E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM52",	1.720000,	43.69,	1.73442942E+00,	1.51553910E-01,	1.46225433E+00,	1.00690928E-02,	4.70634701E-02,	1.40084396E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM61",	1.720002,	46.02,	1.73883330E+00,	1.50937430E-01,	1.12118445E+00,	9.80244105E-03,	4.33179685E-02,	1.01214625E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL10",	1.719995,	50.23,	1.52812575E+00,	3.67965267E-01,	1.11751784E+00,	7.76817644E-03,	2.72026548E-02,	8.88697400E+01,	"Ohara",	mod_sellmeier},
    {"S-TIH18",	1.721507,	29.23,	1.59921608E+00,	2.59532164E-01,	2.12454543E+00,	1.16469304E-02,	5.84824883E-02,	1.86927779E+02,	"Ohara",	mod_sellmeier},
    {"S-BAH28",	1.723420,	37.95,	1.69493484E+00,	1.92890298E-01,	1.56385948E+00,	1.02723190E-02,	5.21187640E-02,	1.37818035E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH10",	1.728250,	28.46,	1.61549392E+00,	2.62433239E-01,	2.09426189E+00,	1.19830897E-02,	5.96510240E-02,	1.81657554E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL18",	1.729157,	54.68,	1.50276318E+00,	4.30224497E-01,	1.34726060E+00,	1.45462356E-02,	-3.3278415E-03,	9.33508342E+01,	"Ohara",	mod_sellmeier},
    {"S-LAL59",	1.733997,	51.47,	1.13962742E+00,	8.05227838E-01,	1.29488061E+00,	4.93294862E-03,	2.02479960E-02,	9.34746507E+01,	"Ohara",	mod_sellmeier},
    {"S-NBH53",	1.738000,	32.26,	1.65828340E+00,	2.63275666E-01,	2.10142759E+00,	1.13872516E-02,	5.22108137E-02,	1.65523649E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH3",	1.739998,	28.30,	1.64797648E+00,	2.67261917E-01,	2.19772845E+00,	1.21917693E-02,	5.97893039E-02,	1.92158340E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH13",	1.740769,	27.79,	1.62224674E+00,	2.93844589E-01,	1.99225164E+00,	1.18368386E-02,	5.90208025E-02,	1.71959976E+02,	"Ohara",	mod_sellmeier},
    {"S-LAL61",	1.740999,	52.64,	1.11073292E+00,	8.59347773E-01,	1.26707433E+00,	4.64181248E-03,	1.92989261E-02,	8.73917698E+01,	"Ohara",	mod_sellmeier},
    {"S-LAM60",	1.743198,	49.34,	1.60673056E+00,	3.66415640E-01,	1.31761804E+00,	7.75046140E-03,	2.89967611E-02,	9.30720709E+01,	"Ohara",	mod_sellmeier},
    {"S-LAM2",	1.743997,	44.78,	1.77130000E+00,	1.95814230E-01,	1.19487834E+00,	9.76652444E-03,	4.12718628E-02,	1.10458122E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM7",	1.749497,	35.28,	1.71014712E+00,	2.56943292E-01,	1.63986271E+00,	1.05161080E-02,	5.02809636E-02,	1.46181217E+02,	"Ohara",	mod_sellmeier},
    {"S-NBH51",	1.749505,	35.33,	1.71203689E+00,	2.55989588E-01,	1.81456998E+00,	1.07724134E-02,	4.88593504E-02,	1.36359013E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH4",	1.755199,	27.51,	1.66755531E+00,	2.94411865E-01,	2.49422119E+00,	1.22052137E-02,	5.97775329E-02,	2.14869618E+02,	"Ohara",	mod_sellmeier},
    {"S-YGH51",	1.754999,	52.32,	1.08280170E+00,	9.33988681E-01,	1.32367286E+00,	1.81156360E-02,	3.04157575E-03,	9.10353195E+01,	"Ohara",	mod_sellmeier},
    {"S-LAM54",	1.756998,	47.82,	1.84213306E+00,	1.75468631E-01,	1.25750878E+00,	9.43993220E-03,	3.95281122E-02,	8.65463013E+01,	"Ohara",	mod_sellmeier},
    {"S-TIH14",	1.761821,	26.52,	1.68915108E+00,	2.90462024E-01,	2.37971516E+00,	1.28202514E-02,	6.18090841E-02,	2.01094352E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM55",	1.762001,	40.10,	1.85412979E+00,	1.65450323E-01,	1.27255422E+00,	1.08438152E-02,	5.14050980E-02,	1.09986837E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH66",	1.772499,	49.60,	1.39280586E+00,	6.79577094E-01,	1.38702069E+00,	6.08475118E-03,	2.33925351E-02,	9.58354094E+01,	"Ohara",	mod_sellmeier},
    {"S-TIH11",	1.784723,	25.68,	1.72677471E+00,	3.24568628E-01,	2.65816809E+00,	1.29369958E-02,	6.18255245E-02,	2.21904637E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH23",	1.784696,	26.29,	1.73986485E+00,	3.13894918E-01,	2.31093206E+00,	1.29441300E-02,	6.12116868E-02,	1.97420482E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH51",	1.785896,	44.20,	1.82586991E+00,	2.83023349E-01,	1.35964319E+00,	9.35297152E-03,	3.73803057E-02,	1.00655798E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH64",	1.788001,	47.37,	1.83021453E+00,	2.91563590E-01,	1.28544024E+00,	9.04823290E-03,	3.30756689E-02,	8.93675501E+01,	"Ohara",	mod_sellmeier},
    {"S-NBH55",	1.800000,	29.84,	1.83145156E+00,	2.87818024E-01,	2.15208300E+00,	1.22443139E-02,	5.73877310E-02,	1.86099124E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH52",	1.799516,	42.22,	1.85390925E+00,	2.97925555E-01,	1.39382086E+00,	9.55320687E-03,	3.93816850E-02,	1.02706848E+02,	"Ohara",	mod_sellmeier},
    {"S-LAM66",	1.800999,	34.97,	1.92094221E+00,	2.19901208E-01,	1.72705231E+00,	1.15075241E-02,	5.47993543E-02,	1.20133674E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH63",	1.804398,	39.59,	1.89458276E+00,	2.68702978E-01,	1.45705526E+00,	1.02277048E-02,	4.42801243E-02,	1.04874927E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH65",	1.804000,	46.57,	1.68191258E+00,	4.93779818E-01,	1.45682822E+00,	7.76684250E-03,	2.88916181E-02,	9.92574356E+01,	"Ohara",	mod_sellmeier},
    {"S-TIH6",	1.805181,	25.42,	1.77227611E+00,	3.45691250E-01,	2.40788501E+00,	1.31182633E-02,	6.14479619E-02,	2.00753254E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH53",	1.806098,	40.92,	1.91811619E+00,	2.53724399E-01,	1.39473885E+00,	1.02147684E-02,	4.33176011E-02,	1.01938021E+02,	"Ohara",	mod_sellmeier},
    {"S-NPH1",	1.808095,	22.76,	1.75156623E+00,	3.64006304E-01,	2.47874141E+00,	1.35004681E-02,	6.68245147E-02,	1.70756006E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH59",	1.816000,	46.62,	1.51372967E+00,	7.02462343E-01,	1.33600982E+00,	7.05246901E-03,	2.49488689E-02,	1.00085908E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH60",	1.834000,	37.16,	1.95243469E+00,	3.07100210E-01,	1.56578094E+00,	1.06442437E-02,	4.56735302E-02,	1.10281410E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH55",	1.834807,	42.71,	1.95615766E+00,	3.19216215E-01,	1.39173189E+00,	9.79338965E-03,	3.76836296E-02,	9.48775271E+01,	"Ohara",	mod_sellmeier},
    {"PBH56",	1.841390,	24.56,	1.78552677E+00,	4.46684871E-01,	1.21749317E+00,	1.36046011E-02,	5.70875152E-02,	1.29967536E+02,	"Ohara",	mod_sellmeier},
    {"S-TIH53",	1.846660,	23.78,	1.87904886E+00,	3.69719775E-01,	2.33730863E+00,	1.44121770E-02,	6.38817990E-02,	1.82668180E+02,	"Ohara",	mod_sellmeier},
    {"S-NPH53",	1.846660,	23.88,	1.85484904E+00,	3.96194484E-01,	2.43512461E+00,	1.34621486E-02,	6.31945361E-02,	1.70864886E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH71",	1.850260,	32.27,	1.98280031E+00,	3.16758450E-01,	2.44472646E+00,	1.18987456E-02,	5.27156001E-02,	2.13220697E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH58",	1.882997,	40.76,	1.78764964E+00,	6.52635600E-01,	1.79914564E+00,	8.47378536E-03,	3.13126408E-02,	1.32788001E+02,	"Ohara",	mod_sellmeier},
    {"S-NPH2",	1.922860,	18.90,	2.03869510E+00,	4.37269641E-01,	2.96711461E+00,	1.70796224E-02,	7.49254813E-02,	1.74155354E+02,	"Ohara",	mod_sellmeier},
    {"S-NPH3",	1.959060,	17.47,	2.09834903E+00,	4.89088388E-01,	2.94009268E+00,	1.79123869E-02,	7.76653353E-02,	1.60930428E+02,	"Ohara",	mod_sellmeier},
    {"S-LAH79",	2.003300,	28.27,	2.32557148E+00,	5.07967133E-01,	2.43087198E+00,	1.32895208E-02,	5.28335449E-02,	1.61122408E+02,	"Ohara",	mod_sellmeier},
    {"FC5",	1.4874899,	70.23,	1.17447043,	0.0140056154,	1.19272435,	0.0084185518,	-0.0581790767,	129.599726,	"Hoya",	mod_sellmeier},
    {"FCD1",	1.4969993,	81.54,	1.17010505,	0.0475710783,	0.763832445,	0.0061620392,	0.0263372876,	141.882642,	"Hoya",	mod_sellmeier},
    {"FCD10",	1.4559987,	90.28,	1.06785857,	0.0335857718,	1.10219763,	0.006992273,	-0.0207608925,	226.496541,	"Hoya",	mod_sellmeier},
    {"PCD4",	1.618000,	63.33,	1.0996655,	0.478125422,	1.13214074,	0.0132718559,	-0.0006016497,	130.595472,	"Hoya",	mod_sellmeier},
    {"BSC7",	1.5163299,	64.14,	1.1515019,	0.118583612,	1.26301359,	0.010598413,	-0.011822519,	129.617662,	"Hoya",	mod_sellmeier},
    {"E-C3",	1.5182289,	58.9,	0.882514764,	0.389271907,	1.10693448,	0.0046450458,	0.0200551397,	136.234339,	"Hoya",	mod_sellmeier},
    {"BAC4",	1.568832,	56.36,	1.27553696,	0.146083393,	1.16754699,	0.0074969236,	0.031042153,	128.947092,	"Hoya",	mod_sellmeier},
    {"BACD2",	1.6073789,	56.81,	0.867168676,	0.672848343,	1.18456107,	0.00369311,	0.0181652804,	132.376147,	"Hoya",	mod_sellmeier},
    {"BACD4",	1.6127159,	58.72,	0.96244308,	0.595939234,	1.10558352,	0.0046806214,	0.0178772082,	115.896432,	"Hoya",	mod_sellmeier},
    {"BACD5",	1.5891299,	61.14,	0.941357273,	0.546174895,	1.16168917,	0.0140333996,	0.0009066357,	114.163758,	"Hoya",	mod_sellmeier},
    {"E-BACD10",1.622799,	57.05,	0.945443081,	0.643237376,	1.17752968,	0.0157263798,	0.0016192407,	121.361748,	"Hoya",	mod_sellmeier},
    {"BACD11",	1.563839,	60.67,	1.243442,	0.166301104,	1.10586114,	0.0116396708,	-0.0089046494,	114.11122,	"Hoya",	mod_sellmeier},
    {"BACD14",	1.6031119,	60.64,	1.2828627,	0.247647429,	1.10383999,	0.0122902399,	-0.0061314236,	106.883378,	"Hoya",	mod_sellmeier},
    {"BACD15",	1.6229919,	58.16,	0.953128328,	0.637613977,	1.65245647,	0.0038763899,	0.0185094632,	159.442367,	"Hoya",	mod_sellmeier},
    {"BACD16",	1.620411,	60.29,	1.14490383,	0.439563911,	1.27688079,	0.0137034916,	-0.0018651421,	119.535585,	"Hoya",	mod_sellmeier},
    {"BACD18",	1.6385393,	55.38,	0.927886025,	0.708858526,	1.18610897,	0.004175492,	0.0184691838,	122.210416,	"Hoya",	mod_sellmeier},
    {"BACED5",	1.6584409,	50.88,	1.34814257,	0.347530319,	1.38798368,	0.0069536437,	0.0277863478,	142.138122,	"Hoya",	mod_sellmeier},
    {"E-BACED20",1.648498,	53.02,	1.50847885,	0.158099826,	1.36815368,	0.0081276908,	0.0354200898,	136.110038,	"Hoya",	mod_sellmeier},
    {"LACL60",	1.6399987,	60.08,	0.996356844,	0.651392837,	1.22432622,	0.0144821587,	0.0015482639,	89.9818604,	"Hoya",	mod_sellmeier},
    {"LAC7",	1.6515969,	58.55,	0.916121247,	0.765948319,	1.27745023,	0.0039588974,	0.0167547425,	110.762706,	"Hoya",	mod_sellmeier},
    {"LAC8",	1.712995,	53.87,	1.30663291,	0.571377253,	1.24303605,	0.0061186245,	0.021272147,	90.6285686,	"Hoya",	mod_sellmeier},
    {"LAC9",	1.6910019,	54.82,	1.16195687,	0.644860099,	1.25062221,	0.0159659509,	0.0005055025,	93.8284169,	"Hoya",	mod_sellmeier},
    {"LAC10",	1.7199954,	50.23,	1.52812575,	0.367965267,	1.11751784,	0.0077681764,	0.0272026548,	88.86974,	"Hoya",	mod_sellmeier},
    {"LAC12",	1.6779000,	55.34,	0.992053895,	0.771377731,	1.18296264,	0.0167095063,	0.0023675016,	105.90108,	"Hoya",	mod_sellmeier},
    {"LAC13",	1.6935009,	53.21,	0.980071267,	0.832904776,	1.28111995,	0.003891237,	0.0189164592,	98.9052676,	"Hoya",	mod_sellmeier},
    {"LAC14",	1.6967969,	55.53,	1.2372097,	0.589722623,	1.3192188,	0.015355132,	-0.0003078963,	93.7202947,	"Hoya",	mod_sellmeier},
    {"TAC2",	1.740999,	52.64,	1.11073292,	0.859347773,	1.26707433,	0.0046418125,	0.0192989261,	87.3917698,	"Hoya",	mod_sellmeier},
    {"TAC4",	1.733997,	51.47,	1.13962742,	0.805227838,	1.29488061,	0.0049329486,	0.020247996,	93.4746507,	"Hoya",	mod_sellmeier},
    {"TAC6",	1.7549985,	52.32,	1.0828017,	0.933988681,	1.32367286,	0.018115636,	0.0030415758,	91.0353195,	"Hoya",	mod_sellmeier},
    {"TAC8",	1.729157,	54.68,	1.50276318,	0.430224497,	1.3472606,	0.0145462356,	-0.0033278415,	93.3508342,	"Hoya",	mod_sellmeier},
    {"E-CF6",	1.517417,	52.43,	1.09666153,	0.168990073,	1.20580827,	0.0066749112,	0.033609545,	141.668738,	"Hoya",	mod_sellmeier},
    {"E-FEL1",	1.548141,	45.79,	1.25088944,	0.0997973327,	1.20583504,	0.0088392128,	0.0482685052,	137.414953,	"Hoya",	mod_sellmeier},
    {"E-FEL2",	1.54072,	47.23,	1.23401499,	0.0959796833,	1.20503991,	0.008695078,	0.0465611429,	137.953301,	"Hoya",	mod_sellmeier},
    {"E-FEL6",	1.531717,	48.84,	1.17701777,	0.12795803,	1.34740124,	0.0077108769,	0.0411325328,	154.531692,	"Hoya",	mod_sellmeier},
    {"BAF10",	1.6700289,	47.23,	1.59034337,	0.138464579,	1.21988043,	0.0093273434,	0.0427498654,	119.251777,	"Hoya",	mod_sellmeier},
    {"BAF11",	1.6667179,	48.32,	1.5713886,	0.147869313,	1.28092846,	0.0091080794,	0.0402401684,	130.399367,	"Hoya",	mod_sellmeier},
    {"E-FL5",	1.581439,	40.75,	1.32122534,	0.123824976,	1.43685254,	0.0095209144,	0.0516062665,	149.064883,	"Hoya",	mod_sellmeier},
    {"E-FL6",	1.5673219,	42.82,	1.31066488,	0.0941903094,	1.23292644,	0.0096889781,	0.0527763106,	133.296422,	"Hoya",	mod_sellmeier},
    {"E-F1",	1.6258819,	35.7,	1.4496383,	0.122986408,	1.38066723,	0.0112094282,	0.059626577,	138.178326,	"Hoya",	mod_sellmeier},
    {"E-F2",	1.620041,	36.26,	1.42193846,	0.133827968,	1.45060574,	0.0107291511,	0.0572587546,	145.381805,	"Hoya",	mod_sellmeier},
    {"E-F3",	1.6129293,	37,	1.40691144,	0.128369745,	1.51826191,	0.0105633641,	0.0568483105,	152.107924,	"Hoya",	mod_sellmeier},
    {"E-F5",	1.6034199,	38.03,	1.38531342,	0.122372945,	1.40508326,	0.0104074567,	0.0557440088,	144.878733,	"Hoya",	mod_sellmeier},
    {"E-F8",	1.595509,	39.24,	1.37262713,	0.112636276,	1.39786421,	0.0103220068,	0.0550195044,	147.735609,	"Hoya",	mod_sellmeier},
    {"BAFD7",	1.7015359,	41.24,	1.68939052,	0.133081013,	1.41165515,	0.0103598193,	0.0533982239,	126.515503,	"Hoya",	mod_sellmeier},
    {"BAFD8",	1.7234199,	37.95,	1.69493484,	0.192890298,	1.56385948,	0.010272319,	0.052118764,	137.818035,	"Hoya",	mod_sellmeier},
    {"E-FD1",	1.717362,	29.52,	1.60326759,	0.242980935,	1.81313592,	0.0118019139,	0.0591363658,	161.218747,	"Hoya",	mod_sellmeier},
    {"E-FD2",	1.6476889,	33.79,	1.44222294,	0.194432265,	1.74092482,	0.0104249404,	0.0550235257,	169.710769,	"Hoya",	mod_sellmeier},
    {"E-FD4",	1.755199,	27.51,	1.66755531,	0.294411865,	2.49422119,	0.0122052137,	0.0597775329,	214.869618,	"Hoya",	mod_sellmeier},
    {"E-FD5",	1.6727000,	32.1,	1.50659233,	0.204786135,	1.92036668,	0.0109501562,	0.0574980285,	178.128535,	"Hoya",	mod_sellmeier},
    {"E-FD7",	1.639799,	34.46,	1.4168047,	0.196785057,	1.68001322,	0.0100732158,	0.0537616908,	164.672436,	"Hoya",	mod_sellmeier},
    {"E-FD8",	1.6889309,	31.07,	1.5427081,	0.217113891,	1.81904459,	0.0113925005,	0.0579224572,	167.697189,	"Hoya",	mod_sellmeier},
    {"E-FD10",	1.72825,	28.46,	1.61549392,	0.262433239,	2.09426189,	0.0119830897,	0.059651024,	181.657554,	"Hoya",	mod_sellmeier},
    {"E-FD13",	1.7407689,	27.79,	1.62224674,	0.293844589,	1.99225164,	0.0118368386,	0.0590208025,	171.959976,	"Hoya",	mod_sellmeier},
    {"E-FD15",	1.698947,	30.13,	1.55849775,	0.230767007,	1.84436099,	0.0115367235,	0.0586095947,	162.981888,	"Hoya",	mod_sellmeier},
    {"FD60",	1.8051809,	25.42,	1.77227611,	0.34569125,	2.40788501,	0.0131182633,	0.0614479619,	200.753254,	"Hoya",	mod_sellmeier},
    {"FD110",	1.784723,	25.68,	1.72677471,	0.324568628,	2.65816809,	0.0129369958,	0.0618255245,	221.904637,	"Hoya",	mod_sellmeier},
    {"FD140",	1.761821,	26.52,	1.68915108,	0.290462024,	2.37971516,	0.0128202514,	0.0618090841,	201.094352,	"Hoya",	mod_sellmeier},
    {"E-FDS1",	1.92286,	18.9,	2.0386951,	0.437269641,	2.96711461,	0.0170796224,	0.0749254813,	174.155354,	"Hoya",	mod_sellmeier},
    {"FDS30",	1.784696,	26.29,	1.73986485,	0.313894918,	2.31093206,	0.01294413,	0.0612116868,	197.420482,	"Hoya",	mod_sellmeier},
    {"FDS90",	1.8466599,	23.78,	1.87904886,	0.369719775,	2.33730863,	0.014412177,	0.063881799,	182.66818,	"Hoya",	mod_sellmeier},
    {"FF5",	1.5927009,	35.31,	1.32940907,	0.141512125,	1.44299068,	0.0102377287,	0.0578081956,	150.597139,	"Hoya",	mod_sellmeier},
    {"LAFL2",	1.6970023,	48.52,	1.63056133,	0.186994897,	1.30014289,	0.0089969071,	0.0368011993,	122.239544,	"Hoya",	mod_sellmeier},
    {"LAF2",	1.743997,	44.78,	1.77130000,	0.19581423,	1.19487834,	0.0097665244,	0.0412718628,	110.458122,	"Hoya",	mod_sellmeier},
    {"LAF3",	1.7170039,	47.92,	1.64258713,	0.23963461,	1.22483026,	0.0086824602,	0.0351226242,	116.604369,	"Hoya",	mod_sellmeier},
    {"E-LAF7",	1.7495045,	35.33,	1.71203689,	0.255989588,	1.81456998,	0.0107724134,	0.0488593504,	136.359013,	"Hoya",	mod_sellmeier},
    {"NBF1",	1.7431979,	49.34,	1.60673056,	0.36641564,	1.31761804,	0.0077504614,	0.0289967611,	93.0720709,	"Hoya",	mod_sellmeier},
    {"NBF2",	1.756998,	47.82,	1.84213306,	0.175468631,	1.25750878,	0.0094399322,	0.0395281122,	86.5463013,	"Hoya",	mod_sellmeier},
    {"TAF1",	1.7724989,	49.6,	1.39280586,	0.679577094,	1.38702069,	0.0060847512,	0.0233925351,	95.8354094,	"Hoya",	mod_sellmeier},
    {"TAF3",	1.804000,	46.57,	1.68191258,	0.493779818,	1.45682822,	0.0077668425,	0.0288916181,	99.2574356,	"Hoya",	mod_sellmeier},
    {"TAF4",	1.788001,	47.37,	1.83021453,	0.29156359,	1.28544024,	0.0090482329,	0.0330756689,	89.3675501,	"Hoya",	mod_sellmeier},
    {"TAF5",	1.816000,	46.62,	1.51372967,	0.702462343,	1.33600982,	0.007052469,	0.0249488689,	100.085908,	"Hoya",	mod_sellmeier},
    {"NBFD3",	1.8043979,	39.59,	1.89458276,	0.268702978,	1.45705526,	0.0102277048,	0.0442801243,	104.874927,	"Hoya",	mod_sellmeier},
    {"NBFD10",	1.8339999,	37.16,	1.95243469,	0.30710021,	1.56578094,	0.0106442437,	0.0456735302,	110.28141,	"Hoya",	mod_sellmeier},
    {"NBFD11",	1.7858959,	44.2,	1.82586991,	0.283023349,	1.35964319,	0.0093529715,	0.0373803057,	100.655798,	"Hoya",	mod_sellmeier},
    {"NBFD12",	1.7995159,	42.22,	1.85390925,	0.297925555,	1.39382086,	0.0095532069,	0.039381685,	102.706848,	"Hoya",	mod_sellmeier},
    {"NBFD13",	1.8060979,	40.92,	1.91811619,	0.253724399,	1.39473885,	0.0102147684,	0.0433176011,	101.938021,	"Hoya",	mod_sellmeier},
    {"TAFD5",	1.8348072,	42.71,	1.95615766,	0.319216215,	1.39173189,	0.0097933897,	0.0376836296,	94.8775271,	"Hoya",	mod_sellmeier},
    {"TAFD30",	1.8829971,	40.76,	1.78764964,	0.6526356,	1.79914564,	0.0084737854,	0.0313126408,	132.788001,	"Hoya",	mod_sellmeier},
    {"E-ADF10",	1.6133973,	44.27,	1.37023101,	0.177665568,	1.30515471,	0.0087192034,	0.0405725552,	112.703058,	"Hoya",	mod_sellmeier},
    {"E-ADF50",	1.6541153,	39.68,	1.47544521,	0.193060095,	1.5093901,	0.0095583674,	0.0460430483,	126.422746,	"Hoya",	mod_sellmeier},
    {"AIR", 1, 89.3, 1, 0, 0, 0, 0, 0,	"", mod_sellmeier}, 
    {"IRIS", 1, 89.3, 1, 0, 0, 0, 0, 0,	"", mod_sellmeier} 
};

const int NUM_MATERIALS = sizeof(glass_database)/sizeof(GlassDef);


typedef struct {
  char mfg[32];
  char name[32];
  float nd;
  float vd;
} GlassAlias;

// Historic glass names with refractive index and Abbe number. 
// Retrieved from various sources, the single most important database being:
// http://www.newportglass.com/optcat.htm
const GlassAlias alias_database[] =  {
{"SCHOTT","NLAK10",1.72,50.6},
{"SCHOTT","BAF12",1.639,45.2},
{"SCHOTT","BAF13",1.669,45},
{"SCHOTT","BAF3",1.583,46.5},
{"SCHOTT","BAF4",1.606,43.9},
{"SCHOTT","BAF5",1.607,49.4},
{"SCHOTT","BAF50",1.683,44.5},
{"SCHOTT","BAF51",1.652,44.9},
{"SCHOTT","BAF52",1.609,46.4},
{"SCHOTT","BAF53",1.67,47.1},
{"SCHOTT","BAF54",1.667,48.2},
{"SCHOTT","BAF8",1.624,47},
{"SCHOTT","BAF9",1.643,48},
{"SCHOTT","BAFN10",1.67,47.1},
{"SCHOTT","BAFN10",1.67,47.1},
{"SCHOTT","BAFN11",1.667,48.4},
{"SCHOTT","BAFN6",1.589,48.5},
{"SCHOTT","BAK1",1.573,57.5},
{"SCHOTT","BAK2",1.54,59.7},
{"SCHOTT","BAK4",1.569,56.1},
{"SCHOTT","BAK5",1.557,58.7},
{"SCHOTT","BAK50",1.568,58},
{"SCHOTT","BAK6",1.574,56.4},
{"SCHOTT","BALF3",1.571,52.9},
{"SCHOTT","BALF4",1.58,53.7},
{"SCHOTT","BALF5",1.547,53.6},
{"SCHOTT","BALF50",1.589,51.4},
{"SCHOTT","BALF51",1.574,52.1},
{"SCHOTT","BALF6",1.589,53},
{"SCHOTT","BALF8",1.554,51.2},
{"SCHOTT","BALK1",1.526,60},
{"SCHOTT","BALKN3",1.518,60.2},
{"SCHOTT","BASF1",1.626,39},
{"SCHOTT","BASF10",1.65,39.2},
{"SCHOTT","BASF12",1.67,39.2},
{"SCHOTT","BASF13",1.698,38.6},
{"SCHOTT","BASF14",1.7,35},
{"SCHOTT","BASF2",1.664,35.8},
{"SCHOTT","BASF5",1.603,42.5},
{"SCHOTT","BASF50",1.71,36.6},
{"SCHOTT","BASF51",1.724,38.1},
{"SCHOTT","BASF52",1.702,41},
{"SCHOTT","BASF54",1.736,32.2},
{"SCHOTT","BASF55",1.7,34.7},
{"SCHOTT","BASF56",1.657,36.7},
{"SCHOTT","BASF57",1.651,41.9},
{"SCHOTT","BASF6",1.668,41.9},
{"SCHOTT","BASF64",1.701,39.6},
{"SCHOTT","BASF64A",1.704,39.4},
{"SCHOTT","BK1",1.51,63.5},
{"SCHOTT","BK10",1.498,67},
{"SCHOTT","BK3",1.498,65.1},
{"SCHOTT","BK6",1.531,62.2},
{"SCHOTT","BK7",1.517,64.2},
{"SCHOTT","BK8",1.52,63.7},
{"SCHOTT","F1",1.626,35.7},
{"SCHOTT","F13",1.622,36},
{"SCHOTT","F14",1.601,38.2},
{"SCHOTT","F15",1.606,37.8},
{"SCHOTT","F2",1.62,36.4},
{"SCHOTT","F3",1.613,37},
{"SCHOTT","F4",1.617,36.6},
{"SCHOTT","F5",1.603,38},
{"SCHOTT","F5",1.603,38},
{"SCHOTT","F6",1.636,35.3},
{"SCHOTT","F7",1.625,35.6},
{"SCHOTT","F8",1.596,39.2},
{"SCHOTT","F9",1.62,38.1},
{"SCHOTT","FK1",1.471,67.3},
{"SCHOTT","FK3",1.465,65.8},
{"SCHOTT","FK5",1.487,70.4},
{"SCHOTT","FK51",1.487,84.5},
{"SCHOTT","FK52",1.486,81.8},
{"SCHOTT","FK54",1.437,90.7},
{"SCHOTT","FN11",1.621,36.2},
{"SCHOTT","K10",1.501,56.4},
{"SCHOTT","K10",1.501,56.4},
{"SCHOTT","K11",1.5,61.4},
{"SCHOTT","K3",1.518,59},
{"SCHOTT","K4",1.519,57.4},
{"SCHOTT","K5",1.522,59.5},
{"SCHOTT","K50",1.523,60.2},
{"SCHOTT","K51",1.505,59.6},
{"SCHOTT","K7",1.511,60.4},
{"SCHOTT","KF1",1.54,51.1},
{"SCHOTT","KF3",1.515,54.7},
{"SCHOTT","KF50",1.531,51.1},
{"SCHOTT","KF6",1.517,52.2},
{"SCHOTT","KF9",1.523,51.5},
{"SCHOTT","KZF6",1.527,51.1},
{"SCHOTT","KZFN1",1.551,49.6},
{"SCHOTT","KZFN2",1.529,51.6},
{"SCHOTT","KZFS1",1.613,44.3},
{"SCHOTT","KZFS6",1.592,48.5},
{"SCHOTT","KZFS7A",1.681,73.2},
{"SCHOTT","KZFS8",1.72,34.6},
{"SCHOTT","KZFSN2",1.558,54.2},
{"SCHOTT","KZFSN4",1.613,44.3},
{"SCHOTT","KZFSN5",1.654,39.6},
{"SCHOTT","KZFSN7",1.681,37.2},
{"SCHOTT","KZFSN9",1.599,46.9},
{"SCHOTT","LAF11A",1.757,31.7},
{"SCHOTT","LAF13",1.776,37.8},
{"SCHOTT","LAF2",1.744,44.7},
{"SCHOTT","LAF20",1.682,48.2},
{"SCHOTT","LAF21",1.788,47.4},
{"SCHOTT","LAF22A",1.782,37.2},
{"SCHOTT","LAF25",1.784,41.3},
{"SCHOTT","LAF26",1.746,40},
{"SCHOTT","LAF3",1.717,48},
{"SCHOTT","LAF9",1.795,28.4},
{"SCHOTT","LAFN10",1.784,43.9},
{"SCHOTT","LAFN21",1.788,47.5},
{"SCHOTT","LAFN23",1.689,49.7},
{"SCHOTT","LAFN24",1.757,47.8},
{"SCHOTT","LAFN28",1.773,49.6},
{"SCHOTT","LAFN7",1.75,35},
{"SCHOTT","LAFN8",1.735,41.6},
{"SCHOTT","LAK10",1.72,50.4},
{"SCHOTT","LAK11",1.658,57.3},
{"SCHOTT","LAK16A",1.734,51.8},
{"SCHOTT","LAK20",1.693,51.6},
{"SCHOTT","LAK21",1.641,60.1},
{"SCHOTT","LAK23",1.669,57.4},
{"SCHOTT","LAK28",1.744,50.8},
{"SCHOTT","LAK31",1.697,56.4},
{"SCHOTT","LAK33",1.754,52.4},
{"SCHOTT","LAK8",1.713,53.8},
{"SCHOTT","LAK9",1.691,54.7},
{"SCHOTT","LAKL12",1.678,54.9},
{"SCHOTT","LAKL21",1.64,59.7},
{"SCHOTT","LAKN12",1.678,55.2},
{"SCHOTT","LAKN13",1.694,53.3},
{"SCHOTT","LAKN13",1.694,53.3},
{"SCHOTT","LAKN14",1.697,55.4},
{"SCHOTT","LAKN17",1.787,50},
{"SCHOTT","LAKN22",1.651,55.9},
{"SCHOTT","LAKN6",1.642,58},
{"SCHOTT","LAKN6",1.643,58},
{"SCHOTT","LAKN7",1.652,58.5},
{"SCHOTT","LASF11",1.802,44.3},
{"SCHOTT","LASF13",1.855,36.6},
{"SCHOTT","LASF14A",1.822,37.5},
{"SCHOTT","LASF18A",1.913,32.4},
{"SCHOTT","LASF3",1.808,40.6},
{"SCHOTT","LASF32",1.803,30.4},
{"SCHOTT","LASF33",1.806,34.2},
{"SCHOTT","LASF35",2.022,29.1},
{"SCHOTT","LASF36A",1.797,35.1},
{"SCHOTT","LASF8",1.807,31.6},
{"SCHOTT","LASFN15",1.878,38.1},
{"SCHOTT","LASFN18",1.913,32.4},
{"SCHOTT","LASFN30",1.803,46.4},
{"SCHOTT","LASFN31",1.881,41},
{"SCHOTT","LASFN9",1.85,32.2},
{"SCHOTT","LF1",1.573,42.6},
{"SCHOTT","LF2",1.589,40.9},
{"SCHOTT","LF3",1.582,42.1},
{"SCHOTT","LF4",1.578,41.6},
{"SCHOTT","LF5",1.581,40.9},
{"SCHOTT","LF6",1.567,42.8},
{"SCHOTT","LF7",1.575,41.5},
{"SCHOTT","LF8",1.564,43.8},
{"SCHOTT","LGSK2",1.586,61},
{"SCHOTT","LLF1",1.548,45.8},
{"SCHOTT","LLF2",1.541,47.2},
{"SCHOTT","LLF3",1.56,47.2},
{"SCHOTT","LLF4",1.561,45.2},
{"SCHOTT","LLF6",1.532,48.8},
{"SCHOTT","LLF7",1.549,45.4},
{"SCHOTT","N-BAF4",1.606,43.9},
{"SCHOTT","NBAF10",1.67,47.1},
{"SCHOTT","NBAF3",1.583,46.6},
{"SCHOTT","NBAF4",1.609,46.6},
{"SCHOTT","NBAF51",1.652,45},
{"SCHOTT","NBAF52",1.609,46.6},
{"SCHOTT","NBAK1",1.573,57.6},
{"SCHOTT","NBAK2",1.54,59.7},
{"SCHOTT","NBAK4",1.569,56},
{"SCHOTT","NBALF4",1.58,53.9},
{"SCHOTT","NBALF5",1.547,53.6},
{"SCHOTT","NBASF2",1.644,36},
{"SCHOTT","NBASF64",1.704,39.4},
{"SCHOTT","NBK10",1.498,67},
{"SCHOTT","NBK7",1.517,64.2},
{"SCHOTT","NF2",1.62,36.4},
{"SCHOTT","NFK5",1.487,70.4},
{"SCHOTT","NFK51",1.487,84.5},
{"SCHOTT","NFK56",1.434,95},
{"SCHOTT","NK5",1.522,59.5},
{"SCHOTT","NKF9",1.523,51.5},
{"SCHOTT","NKZFS2",1.558,54},
{"SCHOTT","NKZFS4",1.613,44.5},
{"SCHOTT","NLAF2",1.744,44.9},
{"SCHOTT","NLAF21",1.788,47.5},
{"SCHOTT","NLAF3",1.717,48},
{"SCHOTT","NLAF32",1.795,45.5},
{"SCHOTT","NLAF33",1.786,44.1},
{"SCHOTT","NLAF34",1.773,49.6},
{"SCHOTT","NLAF7",1.749,34.8},
{"SCHOTT","NLAK12",1.678,55.2},
{"SCHOTT","NLAK14",1.697,55.4},
{"SCHOTT","NLAK21",1.64,60.1},
{"SCHOTT","NLAK22",1.651,55.9},
{"SCHOTT","NLAK33",1.754,52.4},
{"SCHOTT","NLAK7",1.652,58.5},
{"SCHOTT","NLAK8",1.713,53.8},
{"SCHOTT","NLAK9",1.691,54.7},
{"SCHOTT","NLASF31",1.881,41},
{"SCHOTT","NLASF40",1.834,37.3},
{"SCHOTT","NLASF41",1.835,43.1},
{"SCHOTT","NLASF43",1.806,40.6},
{"SCHOTT","NLASF44",1.804,46.5},
{"SCHOTT","NLASF45",1.801,35.1},
{"SCHOTT","NLLF1",1.548,45.9},
{"SCHOTT","NLLF6",1.532,48.9},
{"SCHOTT","NPK51",1.529,77},
{"SCHOTT","NPK52",1.497,81.6},
{"SCHOTT","NPSK3",1.552,63.5},
{"SCHOTT","NPSK3",1.522,63.5},
{"SCHOTT","NPSK53",1.62,63.5},
{"SCHOTT","NPSK57",1.592,68.4},
{"SCHOTT","NPSK58",1.569,71.2},
{"SCHOTT","NSF1",1.717,29.6},
{"SCHOTT","NSF10",1.728,28.5},
{"SCHOTT","NSF15",1.699,30.2},
{"SCHOTT","NSF4",1.755,27.4},
{"SCHOTT","NSF5",1.673,32.3},
{"SCHOTT","NSF56",1.785,26.1},
{"SCHOTT","NSF6",1.805,25.4},
{"SCHOTT","NSF64",1.706,30.2},
{"SCHOTT","NSF8",1.689,31.3},
{"SCHOTT","NSK10",1.623,57},
{"SCHOTT","NSK11",1.564,60.8},
{"SCHOTT","NSK14",1.603,60.6},
{"SCHOTT","NSK15",1.623,58},
{"SCHOTT","NSK16",1.62,60.3},
{"SCHOTT","NSK18",1.639,55.4},
{"SCHOTT","NSK2",1.607,56.7},
{"SCHOTT","NSK4",1.613,58.6},
{"SCHOTT","NSK5",1.589,61.3},
{"SCHOTT","NSSK2",1.622,53.3},
{"SCHOTT","NSSK5",1.658,50.9},
{"SCHOTT","NSSK8",1.618,49.8},
{"SCHOTT","NZK7",1.508,61.2},
{"SCHOTT","PK1",1.504,66.9},
{"SCHOTT","PK2",1.518,65.1},
{"SCHOTT","PK3",1.525,64.7},
{"SCHOTT","PK50",1.521,69.7},
{"SCHOTT","PK51A",1.529,77},
{"SCHOTT","PSK2",1.569,63.1},
{"SCHOTT","PSK3",1.552,63.5},
{"SCHOTT","PSK50",1.558,67.3},
{"SCHOTT","PSK52",1.603,65.4},
{"SCHOTT","PSK53A",1.62,63.5},
{"SCHOTT","PSK54",1.586,64.6},
{"SCHOTT","SF1",1.717,29.5},
{"SCHOTT","SF10",1.728,28.4},
{"SCHOTT","SF11",1.785,25.8},
{"SCHOTT","SF12",1.648,33.8},
{"SCHOTT","SF13",1.741,27.6},
{"SCHOTT","SF14",1.762,26.5},
{"SCHOTT","SF15",1.699,30.1},
{"SCHOTT","SF16",1.646,34.1},
{"SCHOTT","SF17",1.65,33.7},
{"SCHOTT","SF18",1.722,29.3},
{"SCHOTT","SF19",1.667,33},
{"SCHOTT","SF2",1.648,33.9},
{"SCHOTT","SF3",1.74,28.2},
{"SCHOTT","SF4",1.755,27.6},
{"SCHOTT","SF5",1.673,32.2},
{"SCHOTT","SF50",1.655,32.9},
{"SCHOTT","SF51",1.66,32.9},
{"SCHOTT","SF52",1.689,30.6},
{"SCHOTT","SF53",1.728,28.7},
{"SCHOTT","SF54",1.741,28.1},
{"SCHOTT","SF55",1.762,27},
{"SCHOTT","SF56A",1.785,26.1},
{"SCHOTT","SF57",1.847,23.8},
{"SCHOTT","SF58",1.918,21.5},
{"SCHOTT","SF59",1.953,20.4},
{"SCHOTT","SF6",1.805,25.4},
{"SCHOTT","SF61",1.751,27.5},
{"SCHOTT","SF62",1.681,31.9},
{"SCHOTT","SF63",1.748,27.7},
{"SCHOTT","SF64A",1.706,30.3},
{"SCHOTT","SF66",1.923,20.9},
{"SCHOTT","SF7",1.64,34.6},
{"SCHOTT","SF8",1.689,31.2},
{"SCHOTT","SF9",1.654,33.7},
{"SCHOTT","SFL4",1.755,27.2},
{"SCHOTT","SFL57",1.847,23.6},
{"SCHOTT","SFL6",1.805,25.4},
{"SCHOTT","SFN64",1.706,30.3},
{"SCHOTT","SK1",1.61,56.7},
{"SCHOTT","SK10",1.623,56.9},
{"SCHOTT","SK11",1.564,60.8},
{"SCHOTT","SK12",1.583,59.5},
{"SCHOTT","SK13",1.592,58.3},
{"SCHOTT","SK14",1.603,60.6},
{"SCHOTT","SK15",1.623,58.1},
{"SCHOTT","SK16",1.62,60.3},
{"SCHOTT","SK18A",1.639,55.4},
{"SCHOTT","SK19",1.613,57.4},
{"SCHOTT","SK2",1.607,56.7},
{"SCHOTT","SK20",1.56,61.2},
{"SCHOTT","SK3",1.609,58.9},
{"SCHOTT","SK4",1.613,58.6},
{"SCHOTT","SK5",1.589,61.3},
{"SCHOTT","SK51",1.621,60.3},
{"SCHOTT","SK52",1.639,55.5},
{"SCHOTT","SK55",1.62,60.1},
{"SCHOTT","SK6",1.614,56.4},
{"SCHOTT","SK7",1.607,59.5},
{"SCHOTT","SK8",1.611,55.9},
{"SCHOTT","SK9",1.614,55.2},
{"SCHOTT","SSK1",1.617,53.9},
{"SCHOTT","SSK2",1.622,53.2},
{"SCHOTT","SSK3",1.615,51.2},
{"SCHOTT","SSK4",1.618,55.1},
{"SCHOTT","SSK50",1.618,52.6},
{"SCHOTT","SSK51",1.604,53.6},
{"SCHOTT","SSK52",1.658,50.9},
{"SCHOTT","SSKN5",1.658,50.9},
{"SCHOTT","SSKN8",1.618,49.8},
{"SCHOTT","TIF1",1.511,51},
{"SCHOTT","TIF2",1.533,46},
{"SCHOTT","TIF3",1.548,42.2},
{"SCHOTT","TIF4",1.584,37},
{"SCHOTT","TIF6",1.617,31},
{"SCHOTT","TIFN5",1.594,35.5},
{"SCHOTT","TIK1",1.479,58.7},
{"SCHOTT","TISF1",1.673,28.9},
{"SCHOTT","UBK7",1.517,64.3},
{"SCHOTT","UK50",1.523,60.4},
{"SCHOTT","ULTRAN20",1.487,84.5},
{"SCHOTT","ULTRAN30",1.548,74.2},
{"SCHOTT","ZK1",1.533,58},
{"SCHOTT","ZK5",1.534,55.3},
{"SCHOTT","ZKN7",1.508,61.2},
{"PILKINGTON","BF606439",1.606,43.9},
{"PILKINGTON","BF700412",1.7,41.2},
{"PILKINGTON","BF702410",1.702,41},
{"PILKINGTON","BLF574521",1.574,52.1},
{"PILKINGTON","BORF613443",1.613,44.3},
{"PILKINGTON","BORF614439",1.614,43.9},
{"PILKINGTON","BSC510644",1.51,64.4},
{"PILKINGTON","BSC517642",1.517,64.2},
{"PILKINGTON","BSC517642B",1.517,64.2},
{"PILKINGTON","DBC564608",1.564,60.8},
{"PILKINGTON","DBC569631",1.569,63.1},
{"PILKINGTON","DBC589613",1.589,61.3},
{"PILKINGTON","DBC589613A",1.589,61.3},
{"PILKINGTON","DBC603606",1.603,60.6},
{"PILKINGTON","DBC603606A",1.603,60.6},
{"PILKINGTON","DBC607567",1.607,56.7},
{"PILKINGTON","DBC607567A",1.607,56.7},
{"PILKINGTON","DBC607595",1.607,59.5},
{"PILKINGTON","DBC610567",1.61,56.7},
{"PILKINGTON","DBC610573",1.61,57.3},
{"PILKINGTON","DBC613586",1.613,58.6},
{"PILKINGTON","DBC613586A",1.613,58.6},
{"PILKINGTON","DBC620603",1.62,60.3},
{"PILKINGTON","DBC620603A",1.62,60.3},
{"PILKINGTON","DBC620603B",1.62,60.3},
{"PILKINGTON","DBC623569",1.623,56.9},
{"PILKINGTON","DBC623569A",1.623,56.9},
{"PILKINGTON","DBC623581",1.623,58.1},
{"PILKINGTON","DBC623581A",1.623,58.1},
{"PILKINGTON","DBC639554",1.639,55.4},
{"PILKINGTON","DBC639554A",1.639,55.4},
{"PILKINGTON","DBC658509",1.658,50.9},
{"PILKINGTON","DBC658509A",1.658,50.9},
{"PILKINGTON","DEDF717295",1.717,29.5},
{"PILKINGTON","DEDF717295A",1.717,29.5},
{"PILKINGTON","DEDF728284",1.728,28.4},
{"PILKINGTON","DEDF728284A",1.728,28.4},
{"PILKINGTON","DEDF741278",1.741,27.8},
{"PILKINGTON","DEDF741278A",1.741,27.8},
{"PILKINGTON","DEDF741281",1.741,28.1},
{"PILKINGTON","DEDF748278",1.748,27.8},
{"PILKINGTON","DEDF755276",1.755,27.6},
{"PILKINGTON","DEDF755276A",1.755,27.6},
{"PILKINGTON","DEDF762265",1.762,26.5},
{"PILKINGTON","DEDF762265A",1.762,26.5},
{"PILKINGTON","DEDF785258",1.785,25.8},
{"PILKINGTON","DEDF785258A",1.785,25.8},
{"PILKINGTON","DEDF805254",1.805,25.4},
{"PILKINGTON","DEDF805254A",1.805,25.4},
{"PILKINGTON","DEDF847238A",1.847,23.8},
{"PILKINGTON","DF620364",1.62,36.4},
{"PILKINGTON","DF620364A",1.62,36.4},
{"PILKINGTON","DF626357",1.626,35.7},
{"PILKINGTON","EDF648339",1.648,33.9},
{"PILKINGTON","EDF648339A",1.648,33.9},
{"PILKINGTON","EDF653335",1.653,33.5},
{"PILKINGTON","EDF673322",1.673,32.2},
{"PILKINGTON","EDF673322A",1.673,32.2},
{"PILKINGTON","EDF689312",1.689,31.2},
{"PILKINGTON","EDF689312A",1.689,31.2},
{"PILKINGTON","EDF699301",1.699,30.1},
{"PILKINGTON","EDF699301A",1.699,30.1},
{"PILKINGTON","EDF706300",1.706,30},
{"PILKINGTON","ELF541472",1.541,47.2},
{"PILKINGTON","ELF541472A",1.541,47.2},
{"PILKINGTON","ELF548456",1.548,45.6},
{"PILKINGTON","HC518603",1.518,60.3},
{"PILKINGTON","HC519604",1.519,60.4},
{"PILKINGTON","HC522595A",1.522,59.5},
{"PILKINGTON","HC523586",1.523,58.6},
{"PILKINGTON","HC524592A",1.524,59.2},
{"PILKINGTON","HC525579A",1.525,57.9},
{"PILKINGTON","LAC641601B",1.641,60.1},
{"PILKINGTON","LAC643580",1.643,58},
{"PILKINGTON","LAC652585",1.652,58.5},
{"PILKINGTON","LAC658573",1.658,57.3},
{"PILKINGTON","LAC669574",1.669,57.4},
{"PILKINGTON","LAC678552B",1.678,55.2},
{"PILKINGTON","LAC694533",1.694,53.3},
{"PILKINGTON","LAC697554B",1.697,55.4},
{"PILKINGTON","LAC697562",1.697,56.2},
{"PILKINGTON","LAC697564",1.697,56.4},
{"PILKINGTON","LAC713538B",1.713,53.8},
{"PILKINGTON","LAC720504B",1.72,50.4},
{"PILKINGTON","LAC734517",1.734,51.7},
{"PILKINGTON","LACA691547B",1.691,54.7},
{"PILKINGTON","LACB691548",1.691,54.8},
{"PILKINGTON","LAF689497B",1.689,49.7},
{"PILKINGTON","LAF717479",1.717,47.9},
{"PILKINGTON","LAF717480B",1.717,48},
{"PILKINGTON","LAF744447B",1.744,44.7},
{"PILKINGTON","LAF744448",1.744,44.8},
{"PILKINGTON","LAF788475",1.788,47.5},
{"PILKINGTON","LAF788475B",1.788,47.5},
{"PILKINGTON","LAF850322",1.85,32.2},
{"PILKINGTON","LAFA689495",1.689,49.5},
{"PILKINGTON","LDF673321A",1.673,32.1},
{"PILKINGTON","LDF706303",1.706,30.3},
{"PILKINGTON","LDF728285A",1.728,28.5},
{"PILKINGTON","LDF785258A",1.785,25.8},
{"PILKINGTON","LDF805254",1.805,25.4},
{"PILKINGTON","LDF805254A",1.805,25.4},
{"PILKINGTON","LDF847238A",1.847,23.8},
{"PILKINGTON","LF567428",1.567,42.8},
{"PILKINGTON","LF573426",1.573,42.6},
{"PILKINGTON","LF575415",1.575,41.5},
{"PILKINGTON","LF579411",1.579,41.1},
{"PILKINGTON","LF581409",1.581,40.9},
{"PILKINGTON","LF581409A",1.581,40.9},
{"PILKINGTON","LF582421",1.582,42.1},
{"PILKINGTON","LLAC701515",1.701,51.5},
{"PILKINGTON","MBC540597",1.54,59.7},
{"PILKINGTON","MBC557587",1.557,58.7},
{"PILKINGTON","MBC569561",1.569,56.1},
{"PILKINGTON","MBC569561A",1.569,56.1},
{"PILKINGTON","MBC572577",1.572,57.7},
{"PILKINGTON","MBC573575",1.573,57.5},
{"PILKINGTON","MBC573576A",1.573,57.6},
{"PILKINGTON","TF529516",1.529,51.6},
{"PILKINGTON","TF530512",1.53,51.2},
{"PILKINGTON","ZC508612A",1.508,61.2},
{"OHARA","APL1",1.517,69.6},
{"OHARA","BAH10",1.67,47.3},
{"OHARA","BAH13",1.669,45},
{"OHARA","BAH22",1.664,35.8},
{"OHARA","BAH24",1.651,38.3},
{"OHARA","BAH26",1.668,41.9},
{"OHARA","BAH27",1.702,41.2},
{"OHARA","BAH28",1.723,38},
{"OHARA","BAH30",1.65,39.4},
{"OHARA","BAH32",1.67,39.3},
{"OHARA","BAH51",1.683,44.7},
{"OHARA","BAH53",1.686,44},
{"OHARA","BAH54",1.695,42.2},
{"OHARA","BAH71",1.702,40.1},
{"OHARA","BAH77",1.702,41.2},
{"OHARA","BAH78",1.723,38},
{"OHARA","BAL11",1.573,57.8},
{"OHARA","BAL12",1.54,59.5},
{"OHARA","BAL15",1.557,58.7},
{"OHARA","BAL16",1.574,56.5},
{"OHARA","BAL2",1.571,50.8},
{"OHARA","BAL21",1.548,62.8},
{"OHARA","BAL22",1.569,63.2},
{"OHARA","BAL23",1.552,63.8},
{"OHARA","BAL3",1.571,53},
{"OHARA","BAL35",1.589,61.2},
{"OHARA","BAL4",1.58,53.7},
{"OHARA","BAL41",1.564,60.7},
{"OHARA","BAL42",1.583,59.4},
{"OHARA","BAL5",1.547,53.6},
{"OHARA","BAL50",1.56,61.2},
{"OHARA","BAL6",1.589,53.2},
{"OHARA","BAL7",1.589,51.2},
{"OHARA","BAM1",1.557,48.6},
{"OHARA","BAM12",1.639,44.9},
{"OHARA","BAM2",1.57,49.3},
{"OHARA","BAM21",1.626,39.2},
{"OHARA","BAM23",1.607,40.3},
{"OHARA","BAM25",1.603,42.3},
{"OHARA","BAM3",1.583,46.4},
{"OHARA","BAM4",1.603,43.7},
{"OHARA","BAM5",1.607,49.2},
{"OHARA","BAM6",1.589,48.6},
{"OHARA","BAM8",1.624,47.1},
{"OHARA","BAM9",1.643,47.9},
{"OHARA","BPH35",1.645,40.8},
{"OHARA","BPH40",1.677,37.5},
{"OHARA","BPH45",1.719,33.5},
{"OHARA","BPH5",1.654,39.7},
{"OHARA","BPH50",1.74,31.7},
{"OHARA","BPH8",1.72,34.7},
{"OHARA","BPM4",1.613,43.8},
{"OHARA","BPM51",1.613,44.2},
{"OHARA","BPM51",1.613,44.3},
{"OHARA","BSL1",1.51,63.6},
{"OHARA","BSL10",1.498,66.8},
{"OHARA","BSL12",1.519,64.5},
{"OHARA","BSL21",1.504,66.8},
{"OHARA","BSL22",1.518,65},
{"OHARA","BSL23",1.525,64.6},
{"OHARA","BSL3",1.498,65},
{"OHARA","BSL4",1.5,66},
{"OHARA","BSL6",1.531,62.5},
{"OHARA","BSL7",1.516,64.2},
{"OHARA","BSL7W",1.516,64.2},
{"OHARA","BSM1",1.61,56.5},
{"OHARA","BSM10",1.623,57.1},
{"OHARA","BSM14",1.603,60.7},
{"OHARA","BSM15",1.623,58.2},
{"OHARA","BSM16",1.62,60.3},
{"OHARA","BSM16C",1.62,60.3},
{"OHARA","BSM18",1.639,55.4},
{"OHARA","BSM19",1.613,57.4},
{"OHARA","BSM2",1.607,56.8},
{"OHARA","BSM21",1.617,54},
{"OHARA","BSM22",1.622,53.2},
{"OHARA","BSM23",1.615,51.2},
{"OHARA","BSM24",1.618,55.1},
{"OHARA","BSM25",1.658,50.9},
{"OHARA","BSM28",1.618,49.8},
{"OHARA","BSM29",1.62,49.6},
{"OHARA","BSM3",1.609,59},
{"OHARA","BSM36",1.643,58.4},
{"OHARA","BSM4",1.613,58.8},
{"OHARA","BSM6",1.614,56.4},
{"OHARA","BSM7",1.607,59.4},
{"OHARA","BSM71",1.649,53},
{"OHARA","BSM8",1.611,55.9},
{"OHARA","BSM81",1.64,60.1},
{"OHARA","BSM9",1.614,55},
{"OHARA","BSM93",1.641,56.9},
{"OHARA","FPL51",1.497,81.6},
{"OHARA","FPL53",1.438,95},
{"OHARA","FSL1",1.471,67.4},
{"OHARA","FSL3",1.465,65.9},
{"OHARA","FSL5",1.487,70.2},
{"OHARA","FTL10",1.501,56.4},
{"OHARA","FTL8",1.511,51},
{"OHARA","FTM16",1.593,35.3},
{"OHARA","FTM8",1.533,45.9},
{"OHARA","L-BAL35",1.589,61.2},
{"OHARA","L-BAL42",1.583,59.4},
{"OHARA","L-BSM15",1.623,58},
{"OHARA","L-LAL12",1.678,54.9},
{"OHARA","L-LAM69",1.731,40.5},
{"OHARA","L-TIM28",1.689,31.1},
{"OHARA","LAH51",1.786,44.2},
{"OHARA","LAH52",1.8,42.2},
{"OHARA","LAH53",1.806,41},
{"OHARA","LAH54",1.816,44.4},
{"OHARA","LAH55",1.835,42.7},
{"OHARA","LAH57",1.863,41.5},
{"OHARA","LAH58",1.883,40.8},
{"OHARA","LAH59",1.816,46.6},
{"OHARA","LAH60",1.834,37.2},
{"OHARA","LAH62",1.803,46.7},
{"OHARA","LAH63",1.804,39.6},
{"OHARA","LAH64",1.788,47.4},
{"OHARA","LAH65",1.804,46.6},
{"OHARA","LAH66",1.773,49.6},
{"OHARA","LAH71",1.85,32.3},
{"OHARA","LAH75",1.874,35.3},
{"OHARA","LAH78",1.901,31.5},
{"OHARA","LAL10",1.72,50.3},
{"OHARA","LAL11",1.658,57.3},
{"OHARA","LAL12",1.678,55.3},
{"OHARA","LAL13",1.694,53.2},
{"OHARA","LAL14",1.697,55.5},
{"OHARA","LAL18",1.729,54.7},
{"OHARA","LAL52",1.67,57.3},
{"OHARA","LAL53",1.67,51.6},
{"OHARA","LAL54",1.651,56.2},
{"OHARA","LAL55",1.651,56.2},
{"OHARA","LAL56",1.678,50.7},
{"OHARA","LAL57",1.678,53.4},
{"OHARA","LAL58",1.694,50.8},
{"OHARA","LAL59",1.734,51.5},
{"OHARA","LAL60",1.726,53.6},
{"OHARA","LAL61",1.741,52.7},
{"OHARA","LAL64",1.697,56.5},
{"OHARA","LAL7",1.652,58.5},
{"OHARA","LAL8",1.713,53.8},
{"OHARA","LAL9",1.691,54.8},
{"OHARA","LAM2",1.744,44.8},
{"OHARA","LAM3",1.717,47.9},
{"OHARA","LAM51",1.7,48.1},
{"OHARA","LAM52",1.72,43.7},
{"OHARA","LAM53",1.735,49.8},
{"OHARA","LAM54",1.757,47.8},
{"OHARA","LAM55",1.762,40.1},
{"OHARA","LAM56",1.686,49.2},
{"OHARA","LAM57",1.713,43.2},
{"OHARA","LAM58",1.72,42},
{"OHARA","LAM59",1.697,48.5},
{"OHARA","LAM60",1.743,49.3},
{"OHARA","LAM61",1.72,46},
{"OHARA","LAM62",1.783,36.2},
{"OHARA","LAM65",1.75,35},
{"OHARA","LAM66",1.801,35},
{"OHARA","LAM7",1.75,35.3},
{"OHARA","LAM8",1.735,41.1},
{"OHARA","NSL1",1.51,62.1},
{"OHARA","NSL2",1.516,56.8},
{"OHARA","NSL21",1.526,60},
{"OHARA","NSL23",1.518,60.3},
{"OHARA","NSL3",1.518,59},
{"OHARA","NSL31",1.54,51},
{"OHARA","NSL32",1.526,51.2},
{"OHARA","NSL33",1.515,54.7},
{"OHARA","NSL34",1.534,51.6},
{"OHARA","NSL35",1.523,50.8},
{"OHARA","NSL36",1.517,52.4},
{"OHARA","NSL5",1.522,59.8},
{"OHARA","NSL50",1.523,60},
{"OHARA","NSL51",1.523,58.5},
{"OHARA","NSL7",1.511,60.5},
{"OHARA","PBH1",1.717,29.5},
{"OHARA","PBH10",1.728,28.5},
{"OHARA","PBH11",1.785,25.7},
{"OHARA","PBH11W",1.785,25.7},
{"OHARA","PBH13",1.741,27.8},
{"OHARA","PBH13W",1.741,27.8},
{"OHARA","PBH14",1.762,26.6},
{"OHARA","PBH14W",1.762,26.6},
{"OHARA","PBH18",1.722,29.2},
{"OHARA","PBH21",1.923,20.9},
{"OHARA","PBH23",1.785,26.2},
{"OHARA","PBH23W",1.785,26.2},
{"OHARA","PBH25",1.762,27.1},
{"OHARA","PBH25W",1.762,27.1},
{"OHARA","PBH3",1.74,28.3},
{"OHARA","PBH3W",1.74,28.3},
{"OHARA","PBH4",1.755,27.5},
{"OHARA","PBH4W",1.755,27.5},
{"OHARA","PBH51",1.72,29.3},
{"OHARA","PBH53",1.847,23.9},
{"OHARA","PBH53W",1.847,23.9},
{"OHARA","PBH54",1.751,27.7},
{"OHARA","PBH6",1.805,25.4},
{"OHARA","PBH6W",1.805,25.4},
{"OHARA","PBH71",1.923,21.3},
{"OHARA","PBH72",1.915,21.2},
{"OHARA","PBL1",1.548,45.8},
{"OHARA","PBL2",1.541,47.2},
{"OHARA","PBL21",1.573,42.6},
{"OHARA","PBL22",1.589,41.1},
{"OHARA","PBL23",1.582,42.1},
{"OHARA","PBL24",1.578,41.5},
{"OHARA","PBL25",1.581,40.8},
{"OHARA","PBL26",1.567,42.8},
{"OHARA","PBL27",1.575,41.5},
{"OHARA","PBL28",1.564,43.8},
{"OHARA","PBL3",1.56,47},
{"OHARA","PBL4",1.561,45.2},
{"OHARA","PBL6",1.532,48.9},
{"OHARA","PBL7",1.549,45.6},
{"OHARA","PBM1",1.626,35.7},
{"OHARA","PBM10",1.624,36.5},
{"OHARA","PBM11",1.621,35.9},
{"OHARA","PBM2",1.62,36.3},
{"OHARA","PBM22",1.648,33.8},
{"OHARA","PBM25",1.673,32.1},
{"OHARA","PBM27",1.64,34.5},
{"OHARA","PBM28",1.689,31.1},
{"OHARA","PBM28W",1.689,31.1},
{"OHARA","PBM29",1.654,33.6},
{"OHARA","PBM3",1.613,37},
{"OHARA","PBM32",1.648,33.8},
{"OHARA","PBM35",1.699,30.1},
{"OHARA","PBM39",1.667,33.1},
{"OHARA","PBM4",1.617,36.6},
{"OHARA","PBM5",1.603,38},
{"OHARA","PBM6",1.636,35.4},
{"OHARA","PBM8",1.596,39.2},
{"OHARA","PBM9",1.62,38.1},
{"OHARA","PHM51",1.617,62.8},
{"OHARA","PHM52",1.618,63.4},
{"OHARA","PHM53",1.603,65.5},
{"OHARA","S-APL1",1.517,69.6},
{"OHARA","S-BAH10",1.67,47.3},
{"OHARA","S-BAH11",1.667,48.3},
{"OHARA","S-BAH27",1.702,41.2},
{"OHARA","S-BAH28",1.723,38},
{"OHARA","S-BAH32",1.67,39.3},
{"OHARA","S-BAH54",1.695,42.2},
{"OHARA","S-BAL11",1.573,57.8},
{"OHARA","S-BAL12",1.54,59.5},
{"OHARA","S-BAL14",1.569,56.3},
{"OHARA","S-BAL2",1.571,50.8},
{"OHARA","S-BAL22",1.569,63.2},
{"OHARA","S-BAL3",1.571,53},
{"OHARA","S-BAL35",1.589,61.2},
{"OHARA","S-BAL41",1.564,60.7},
{"OHARA","S-BAL42",1.583,59.4},
{"OHARA","S-BAL50",1.56,61.2},
{"OHARA","S-BAM12",1.639,44.9},
{"OHARA","S-BAM3",1.583,46.4},
{"OHARA","S-BAM4",1.606,43.7},
{"OHARA","S-BSL7",1.516,64.1},
{"OHARA","S-BSM10",1.623,57},
{"OHARA","S-BSM14",1.603,60.7},
{"OHARA","S-BSM15",1.623,58.2},
{"OHARA","S-BSM16",1.62,60.3},
{"OHARA","S-BSM18",1.639,55.4},
{"OHARA","S-BSM2",1.607,56.8},
{"OHARA","S-BSM22",1.622,53.2},
{"OHARA","S-BSM25",1.658,50.9},
{"OHARA","S-BSM28",1.618,49.8},
{"OHARA","S-BSM36",1.643,58.4},
{"OHARA","S-BSM4",1.613,58.7},
{"OHARA","S-BSM71",1.649,53},
{"OHARA","S-BSM81",1.64,60.1},
{"OHARA","S-BSM9",1.614,55},
{"OHARA","S-BSM93",1.641,56.9},
{"OHARA","S-FPL51",1.497,81.6},
{"OHARA","S-FPL52",1.456,90.3},
{"OHARA","S-FPL53",1.439,95},
{"OHARA","S-FSL5",1.487,70.2},
{"OHARA","S-FTL10",1.501,56.4},
{"OHARA","S-FTM16",1.593,35.3},
{"OHARA","S-LAH51",1.786,44.2},
{"OHARA","S-LAH52",1.8,42.2},
{"OHARA","S-LAH53",1.806,40.9},
{"OHARA","S-LAH54",1.816,44.4},
{"OHARA","S-LAH55",1.835,42.7},
{"OHARA","S-LAH58",1.883,40.8},
{"OHARA","S-LAH59",1.816,46.6},
{"OHARA","S-LAH60",1.834,37.2},
{"OHARA","S-LAH63",1.804,39.6},
{"OHARA","S-LAH64",1.788,47.4},
{"OHARA","S-LAH65",1.804,46.6},
{"OHARA","S-LAH66",1.773,49.6},
{"OHARA","S-LAH67",1.795,45.3},
{"OHARA","S-LAH75",1.874,35.3},
{"OHARA","S-LAH79",2.003,28.3},
{"OHARA","S-LAL10",1.72,50.2},
{"OHARA","S-LAL11",1.658,57.3},
{"OHARA","S-LAL12",1.678,55.3},
{"OHARA","S-LAL13",1.694,53.2},
{"OHARA","S-LAL14",1.697,55.5},
{"OHARA","S-LAL18",1.729,54.7},
{"OHARA","S-LAL52",1.67,57.3},
{"OHARA","S-LAL54",1.651,56.2},
{"OHARA","S-LAL56",1.678,50.7},
{"OHARA","S-LAL58",1.694,50.8},
{"OHARA","S-LAL59",1.734,51.5},
{"OHARA","S-LAL60",1.726,53.6},
{"OHARA","S-LAL61",1.741,52.7},
{"OHARA","S-LAL7",1.652,58.5},
{"OHARA","S-LAL8",1.713,53.9},
{"OHARA","S-LAL9",1.691,54.8},
{"OHARA","S-LAM2",1.744,44.8},
{"OHARA","S-LAM3",1.717,47.9},
{"OHARA","S-LAM51",1.7,48.1},
{"OHARA","S-LAM52",1.72,43.7},
{"OHARA","S-LAM54",1.757,47.8},
{"OHARA","S-LAM55",1.762,40.1},
{"OHARA","S-LAM58",1.72,42},
{"OHARA","S-LAM59",1.697,48.5},
{"OHARA","S-LAM60",1.743,49.3},
{"OHARA","S-LAM61",1.72,46},
{"OHARA","S-LAM66",1.801,35},
{"OHARA","S-LAM7",1.75,35.3},
{"OHARA","S-NBH5",1.654,39.7},
{"OHARA","S-NBH8",1.72,34.7},
{"OHARA","S-NBM51",1.613,44.3},
{"OHARA","S-NPH1",1.808,22.8},
{"OHARA","S-NPH2",1.923,18.9},
{"OHARA","S-NSL2",1.516,56.8},
{"OHARA","S-NSL3",1.518,59},
{"OHARA","S-NSL36",1.517,52.4},
{"OHARA","S-NSL5",1.522,59.8},
{"OHARA","S-PHM51",1.617,62.8},
{"OHARA","S-PHM52",1.618,63.4},
{"OHARA","S-PHM53",1.603,65.5},
{"OHARA","S-TIH1",1.717,29.5},
{"OHARA","S-TIH10",1.728,28.5},
{"OHARA","S-TIH11",1.785,25.7},
{"OHARA","S-TIH13",1.741,27.8},
{"OHARA","S-TIH14",1.762,26.5},
{"OHARA","S-TIH18",1.722,29.2},
{"OHARA","S-TIH23",1.785,26.3},
{"OHARA","S-TIH3",1.74,28.3},
{"OHARA","S-TIH4",1.755,27.5},
{"OHARA","S-TIH53",1.847,23.8},
{"OHARA","S-TIH6",1.805,25.4},
{"OHARA","S-TIL1",1.548,45.8},
{"OHARA","S-TIL2",1.541,47.2},
{"OHARA","S-TIL25",1.581,40.7},
{"OHARA","S-TIL26",1.567,42.8},
{"OHARA","S-TIL27",1.575,41.5},
{"OHARA","S-TIL6",1.532,48.9},
{"OHARA","S-TIM1",1.626,35.7},
{"OHARA","S-TIM2",1.62,36.3},
{"OHARA","S-TIM22",1.648,33.8},
{"OHARA","S-TIM25",1.673,32.1},
{"OHARA","S-TIM27",1.64,34.5},
{"OHARA","S-TIM28",1.689,31.1},
{"OHARA","S-TIM3",1.613,37},
{"OHARA","S-TIM35",1.699,30.1},
{"OHARA","S-TIM39",1.667,33},
{"OHARA","S-TIM5",1.603,38},
{"OHARA","S-TIM6",1.636,35.4},
{"OHARA","S-TIM8",1.596,39.2},
{"OHARA","S-YGH51",1.755,52.3},
{"OHARA","S-YGH52",1.787,50},
{"OHARA","SSL2",1.529,51.7},
{"OHARA","SSL4",1.57,48.1},
{"OHARA","SSL5",1.521,52.6},
{"OHARA","SSL6",1.527,51.1},
{"OHARA","TIH11",1.785,25.7},
{"OHARA","TIH14",1.762,26.5},
{"OHARA","TIH23",1.785,26.3},
{"OHARA","TIH53",1.847,23.8},
{"OHARA","TIH6",1.805,25.4},
{"OHARA","TIM11",1.621,35.9},
{"OHARA","TPH55",1.755,25.1},
{"OHARA","YGH51",1.755,52.3},
{"OHARA","YGH52",1.787,50},
{"OHARA","ZSL1",1.533,58},
{"OHARA","ZSL4",1.512,58.1},
{"OHARA","ZSL5",1.534,55.5},
{"OHARA","ZSL7",1.508,60.8},
{"HOYA","ADC1",1.62,62.2},
{"HOYA","ADC1",1.62,62.2},
{"HOYA","ADC2",1.597,55.3},
{"HOYA","ADF1",1.565,53},
{"HOYA","ADF10",1.613,44.4},
{"HOYA","ADF10",1.613,44.4},
{"HOYA","ADF355",1.645,40.8},
{"HOYA","ADF4",1.613,44.9},
{"HOYA","ADF40",1.613,44.3},
{"HOYA","ADF405",1.677,37.5},
{"HOYA","ADF455",1.719,33.5},
{"HOYA","ADF50",1.654,39.6},
{"HOYA","ADF50",1.654,39.6},
{"HOYA","ADF505",1.74,31.7},
{"HOYA","ADF8",1.687,42.9},
{"HOYA","ATC1",1.62,62.2},
{"HOYA","ATF2",1.651,30.8},
{"HOYA","ATF4",1.654,44.7},
{"HOYA","BAC1",1.573,57.5},
{"HOYA","BAC2",1.54,59.7},
{"HOYA","BAC4",1.569,56},
{"HOYA","BAC5",1.557,58.6},
{"HOYA","BAC6",1.574,56.4},
{"HOYA","BACD1",1.61,56.7},
{"HOYA","BACD10",1.623,56.9},
{"HOYA","BACD10",1.623,56.9},
{"HOYA","BACD11",1.564,60.8},
{"HOYA","BACD12",1.583,59.5},
{"HOYA","BACD13",1.592,58.3},
{"HOYA","BACD14",1.603,60.7},
{"HOYA","BACD14",1.603,60.7},
{"HOYA","BACD15",1.623,58.1},
{"HOYA","BACD15 ",1.623,58.1},
{"HOYA","BACD16",1.62,60.3},
{"HOYA","BACD16",1.62,60.3},
{"HOYA","BACD165",1.62,59.8},
{"HOYA","BACD18",1.639,55.5},
{"HOYA","BACD18",1.639,55.5},
{"HOYA","BACD2",1.607,56.7},
{"HOYA","BACD2",1.607,56.7},
{"HOYA","BACD3",1.609,58.9},
{"HOYA","BACD4",1.613,58.6},
{"HOYA","BACD4",1.613,58.6},
{"HOYA","BACD5",1.589,61.3},
{"HOYA","BACD50",1.607,55},
{"HOYA","BACD6",1.614,56.4},
{"HOYA","BACD7",1.607,59.5},
{"HOYA","BACD8",1.611,55.8},
{"HOYA","BACD9",1.614,55.1},
{"HOYA","BACED1",1.617,53.9},
{"HOYA","BACED2",1.622,53.1},
{"HOYA","BACED20",1.649,53},
{"HOYA","BACED20",1.649,53},
{"HOYA","BACED3",1.615,51.2},
{"HOYA","BACED4",1.618,55.2},
{"HOYA","BACED5",1.658,50.9},
{"HOYA","BACED5",1.658,50.9},
{"HOYA","BACED9",1.62,49.8},
{"HOYA","BACL1",1.526,60.1},
{"HOYA","BACL3",1.518,60.4},
{"HOYA","BAF10",1.67,47.2},
{"HOYA","BAF10",1.67,47.2},
{"HOYA","BAF11",1.667,48.3},
{"HOYA","BAF11",1.667,48.3},
{"HOYA","BAF12",1.639,45.1},
{"HOYA","BAF13",1.669,44.9},
{"HOYA","BAF2",1.57,49.4},
{"HOYA","BAF20",1.686,43.9},
{"HOYA","BAF21",1.664,48.9},
{"HOYA","BAF22",1.683,44.7},
{"HOYA","BAF23",1.652,44.9},
{"HOYA","BAF3",1.583,46.5},
{"HOYA","BAF4",1.606,43.9},
{"HOYA","BAF5",1.607,49.3},
{"HOYA","BAF6",1.589,48.5},
{"HOYA","BAF7",1.608,46.2},
{"HOYA","BAF8",1.624,47.1},
{"HOYA","BAF8",1.624,47.1},
{"HOYA","BAF9",1.643,47.9},
{"HOYA","BAFD1",1.626,39.1},
{"HOYA","BAFD10",1.65,39.3},
{"HOYA","BAFD14",1.7,34.9},
{"HOYA","BAFD15",1.702,40.2},
{"HOYA","BAFD15",1.702,40.2},
{"HOYA","BAFD16",1.6,42.5},
{"HOYA","BAFD2",1.664,35.9},
{"HOYA","BAFD3",1.607,40.4},
{"HOYA","BAFD4",1.651,38.3},
{"HOYA","BAFD5",1.603,42.4},
{"HOYA","BAFD6",1.668,41.9},
{"HOYA","BAFD7",1.702,41.2},
{"HOYA","BAFD7",1.702,41.2},
{"HOYA","BAFD8",1.723,38},
{"HOYA","BAFD8",1.723,38},
{"HOYA","BAFL2",1.571,50.9},
{"HOYA","BAFL3",1.571,53},
{"HOYA","BAFL4",1.58,53.7},
{"HOYA","BAFL6",1.589,52.9},
{"HOYA","BSC1",1.51,63.4},
{"HOYA","BSC3",1.498,65.1},
{"HOYA","BSC4",1.5,66},
{"HOYA","BSC6",1.531,62.1},
{"HOYA","BSC7",1.517,64.2},
{"HOYA","BSC7",1.517,64.2},
{"HOYA","C10",1.501,56.3},
{"HOYA","C12",1.523,58.6},
{"HOYA","C2",1.516,56.8},
{"HOYA","C3",1.518,59},
{"HOYA","C3",1.518,59},
{"HOYA","C40",1.523,60.4},
{"HOYA","C5",1.522,59.5},
{"HOYA","C7",1.511,60.5},
{"HOYA","CF1",1.54,51},
{"HOYA","CF2",1.526,51.1},
{"HOYA","CF3",1.515,54.6},
{"HOYA","CF4",1.534,51.5},
{"HOYA","CF5",1.523,51},
{"HOYA","CF6",1.517,52.2},
{"HOYA","CF6",1.517,52.2},
{"HOYA","E-BACD10",1.623,56.9},
{"HOYA","E-BACD10",1.623,56.9},
{"HOYA","E-BACED20",1.649,53},
{"HOYA","E-BACED20",1.649,53},
{"HOYA","E-CF6",1.517,52.2},
{"HOYA","E-CF6",1.517,52.2},
{"HOYA","EADF10",1.613,44.4},
{"HOYA","EADF10",1.613,44.4},
{"HOYA","EADF50",1.654,39.6},
{"HOYA","EADF50",1.654,39.6},
{"HOYA","EBAF8",1.624,47.1},
{"HOYA","EBAF8",1.624,47.1},
{"HOYA","EC3",1.518,59},
{"HOYA","EC3",1.518,59},
{"HOYA","EDF7",1.64,34.6},
{"HOYA","EF1",1.626,35.7},
{"HOYA","EF1",1.626,35.7},
{"HOYA","EF2",1.62,36.3},
{"HOYA","EF2",1.62,36.3},
{"HOYA","EF3",1.613,37},
{"HOYA","EF3",1.613,37},
{"HOYA","EF5",1.603,38},
{"HOYA","EF5",1.603,38},
{"HOYA","EF8",1.596,39.2},
{"HOYA","EF8",1.596,39.2},
{"HOYA","EFD1",1.717,29.5},
{"HOYA","EFD1",1.717,29.5},
{"HOYA","EFD10",1.728,28.3},
{"HOYA","EFD10",1.728,28.3},
{"HOYA","EFD13",1.741,27.8},
{"HOYA","EFD13",1.741,27.8},
{"HOYA","EFD15",1.699,30.1},
{"HOYA","EFD15",1.699,30.1},
{"HOYA","EFD2",1.648,33.8},
{"HOYA","EFD2",1.678,33.8},
{"HOYA","EFD4",1.755,27.5},
{"HOYA","EFD4",1.755,27.5},
{"HOYA","EFD5",1.673,32.2},
{"HOYA","EFD5",1.673,32.2},
{"HOYA","EFD7",1.64,34.6},
{"HOYA","EFD8",1.689,31.2},
{"HOYA","EFD8",1.689,31.2},
{"HOYA","EFDS1",1.923,20.9},
{"HOYA","EFDS1",1.923,20.9},
{"HOYA","EFEL1",1.548,45.8},
{"HOYA","EFEL1",1.548,45.8},
{"HOYA","EFEL2",1.541,47.2},
{"HOYA","EFEL2",1.541,47.2},
{"HOYA","EFEL6",1.532,48.8},
{"HOYA","EFEL6",1.532,48.8},
{"HOYA","EFL5",1.581,40.9},
{"HOYA","EFL6",1.567,42.8},
{"HOYA","ELAF7",1.75,35},
{"HOYA","ELAF7",1.75,35},
{"HOYA","F1",1.626,35.7},
{"HOYA","F1",1.626,35.7},
{"HOYA","F1",1.626,35.7},
{"HOYA","F11",1.621,36},
{"HOYA","F15",1.606,37.9},
{"HOYA","F2",1.62,36.3},
{"HOYA","F2",1.62,36.3},
{"HOYA","F3",1.613,37},
{"HOYA","F3",1.613,37},
{"HOYA","F4",1.617,36.6},
{"HOYA","F5",1.603,38},
{"HOYA","F5",1.603,38},
{"HOYA","F6",1.636,35.3},
{"HOYA","F7",1.625,35.6},
{"HOYA","F8",1.596,39.2},
{"HOYA","F8",1.596,39.2},
{"HOYA","F9",1.62,38.1},
{"HOYA","FC1",1.471,67.3},
{"HOYA","FC3",1.465,65.8},
{"HOYA","FC5",1.487,70.4},
{"HOYA","FCD1",1.497,81.6},
{"HOYA","FCD10",1.457,90.8},
{"HOYA","FD1",1.717,29.5},
{"HOYA","FD1",1.717,29.5},
{"HOYA","FD10",1.728,28.3},
{"HOYA","FD10",1.728,28.3},
{"HOYA","FD11",1.785,25.7},
{"HOYA","FD11",1.785,25.7},
{"HOYA","FD110",1.785,25.7},
{"HOYA","FD110",1.785,25.7},
{"HOYA","FD12",1.648,33.8},
{"HOYA","FD13",1.741,27.8},
{"HOYA","FD13",1.741,27.8},
{"HOYA","FD14",1.762,26.6},
{"HOYA","FD14",1.762,26.6},
{"HOYA","FD140",1.762,26.6},
{"HOYA","FD140",1.762,26.6},
{"HOYA","FD15",1.699,30.1},
{"HOYA","FD15",1.699,30.1},
{"HOYA","FD18",1.722,29.2},
{"HOYA","FD19",1.667,33.1},
{"HOYA","FD2",1.648,33.8},
{"HOYA","FD2",1.648,33.8},
{"HOYA","FD20",1.72,29.3},
{"HOYA","FD3",1.74,28.2},
{"HOYA","FD4",1.755,27.5},
{"HOYA","FD4",1.755,27.5},
{"HOYA","FD41",1.751,27.7},
{"HOYA","FD5",1.673,32.2},
{"HOYA","FD5",1.673,32.2},
{"HOYA","FD6",1.805,25.5},
{"HOYA","FD60",1.805,25.5},
{"HOYA","FD7",1.64,34.6},
{"HOYA","FD7",1.64,34.6},
{"HOYA","FD8",1.689,31.2},
{"HOYA","FD8",1.689,31.2},
{"HOYA","FD9",1.654,33.7},
{"HOYA","FDS1",1.923,20.9},
{"HOYA","FDS1",1.923,20.9},
{"HOYA","FDS10",1.981,20.5},
{"HOYA","FDS2",1.728,28.7},
{"HOYA","FDS3",1.785,26.1},
{"HOYA","FDS30",1.785,26.1},
{"HOYA","FDS30",1.785,26.1},
{"HOYA","FDS5",1.762,26.9},
{"HOYA","FDS9",1.847,23.8},
{"HOYA","FDS90",1.847,23.8},
{"HOYA","FEL1",1.548,45.8},
{"HOYA","FEL1",1.548,45.8},
{"HOYA","FEL2",1.541,47.2},
{"HOYA","FEL2",1.541,47.2},
{"HOYA","FEL3",1.56,47.1},
{"HOYA","FEL4",1.561,45.2},
{"HOYA","FEL6",1.532,48.8},
{"HOYA","FEL6",1.532,48.8},
{"HOYA","FEL7",1.549,45.4},
{"HOYA","FF1",1.511,51},
{"HOYA","FF2",1.533,45.9},
{"HOYA","FF3",1.593,35.5},
{"HOYA","FF5",1.593,35.5},
{"HOYA","FF5",1.593,35.5},
{"HOYA","FF8",1.752,25.1},
{"HOYA","FF8",1.752,25.1},
{"HOYA","FL1",1.573,42.6},
{"HOYA","FL2",1.589,41},
{"HOYA","FL3",1.582,42},
{"HOYA","FL4",1.578,41.7},
{"HOYA","FL5",1.581,40.9},
{"HOYA","FL57",1.576,41.4},
{"HOYA","FL6",1.567,42.8},
{"HOYA","FL7",1.575,41.5},
{"HOYA","LAC10",1.72,50.3},
{"HOYA","LAC10",1.72,50.3},
{"HOYA","LAC11",1.658,57.3},
{"HOYA","LAC12",1.678,55.5},
{"HOYA","LAC12",1.678,55.5},
{"HOYA","LAC13",1.694,53.3},
{"HOYA","LAC13",1.694,53.3},
{"HOYA","LAC14",1.697,55.5},
{"HOYA","LAC14",1.697,55.5},
{"HOYA","LAC15",1.697,56.1},
{"HOYA","LAC6",1.643,58},
{"HOYA","LAC7",1.652,58.4},
{"HOYA","LAC7",1.652,58.4},
{"HOYA","LAC8",1.713,53.9},
{"HOYA","LAC8",1.713,53.9},
{"HOYA","LAC9",1.691,54.7},
{"HOYA","LAC9",1.691,54.7},
{"HOYA","LACL1",1.641,56.8},
{"HOYA","LACL2",1.65,39.3},
{"HOYA","LACL2",1.65,55.7},
{"HOYA","LACL3",1.665,53.4},
{"HOYA","LACL30",1.658,53.4},
{"HOYA","LACL4",1.67,51.7},
{"HOYA","LACL5",1.694,50.8},
{"HOYA","LACL5",1.694,50.8},
{"HOYA","LACL6",1.64,60.2},
{"HOYA","LACL60",1.64,60.2},
{"HOYA","LACL60",1.64,60.2},
{"HOYA","LACL7",1.67,57.3},
{"HOYA","LACL8",1.678,53.4},
{"HOYA","LACL9",1.678,50.6},
{"HOYA","LACL9",1.678,50.6},
{"HOYA","LAF2",1.744,44.9},
{"HOYA","LAF2",1.744,44.9},
{"HOYA","LAF20",1.744,44.9},
{"HOYA","LAF3",1.717,48},
{"HOYA","LAF3",1.717,48},
{"HOYA","LAF30",1.717,48},
{"HOYA","LAF7",1.75,35},
{"HOYA","LAF7",1.75,35},
{"HOYA","LAFL1",1.686,49.4},
{"HOYA","LAFL2",1.697,48.5},
{"HOYA","LAFL2",1.697,48.5},
{"HOYA","LAFL20",1.682,48.2},
{"HOYA","LAFL23",1.689,49.5},
{"HOYA","LAFL3",1.7,47.8},
{"HOYA","LAFL4",1.713,43.3},
{"HOYA","LAFL5",1.72,43.9},
{"HOYA","LAFL6",1.72,42},
{"HOYA","MBACD12",1.583,59.5},
{"HOYA","MBACD5N",1.589,61.2},
{"HOYA","MBAF41",1.606,44},
{"HOYA","MBAF41",1.606,44},
{"HOYA","MLAC130",1.694,53.2},
{"HOYA","MLAC130",1.694,53.2},
{"HOYA","MLAF81",1.731,40.5},
{"HOYA","MLAF81",1.731,40.5},
{"HOYA","MNBF1",1.743,49.3},
{"HOYA","MNBF1",1.743,49.3},
{"HOYA","MNBFD13",1.806,40.7},
{"HOYA","MNBFD82",1.815,37},
{"HOYA","MNBFD83",1.82,29.7},
{"HOYA","NBF1",1.743,49.2},
{"HOYA","NBF1",1.743,49.2},
{"HOYA","NBF2",1.757,47.7},
{"HOYA","NBF2",1.757,47.7},
{"HOYA","NBF3",1.735,49.8},
{"HOYA","NBFD1",1.789,42.8},
{"HOYA","NBFD10",1.834,37.3},
{"HOYA","NBFD11",1.786,43.9},
{"HOYA","NBFD11",1.786,43.9},
{"HOYA","NBFD12",1.8,42.3},
{"HOYA","NBFD13",1.806,40.7},
{"HOYA","NBFD14",1.802,44.3},
{"HOYA","NBFD15",1.806,33.3},
{"HOYA","NBFD2",1.797,41.1},
{"HOYA","NBFD3",1.805,39.6},
{"HOYA","NBFD4",1.831,36.5},
{"HOYA","NBFD40",1.829,36.5},
{"HOYA","NBFD5",1.762,40.3},
{"HOYA","NBFD6",1.757,36.3},
{"HOYA","NBFD7",1.783,36.1},
{"HOYA","NBFD8",1.807,35.5},
{"HOYA","NBFD9",1.757,31.8},
{"HOYA","PC1",1.504,66.9},
{"HOYA","PC2",1.518,65.2},
{"HOYA","PC3",1.525,64.6},
{"HOYA","PCD2",1.569,63.1},
{"HOYA","PCD3",1.552,63.4},
{"HOYA","PCD4",1.618,63.4},
{"HOYA","PCD4",1.618,63.4},
{"HOYA","PCD5",1.617,62.8},
{"HOYA","PCD53",1.62,63.5},
{"HOYA","PCS1",1.517,69.7},
{"HOYA","SBF1",1.551,49.5},
{"HOYA","SBF2",1.529,51.6},
{"HOYA","SBF5",1.521,52.8},
{"HOYA","SBF6",1.527,51.1},
{"HOYA","TAC1",1.726,53.4},
{"HOYA","TAC2",1.741,52.6},
{"HOYA","TAC2",1.741,52.6},
{"HOYA","TAC4",1.734,51.1},
{"HOYA","TAC4",1.734,51.1},
{"HOYA","TAC40",1.73,51.4},
{"HOYA","TAC6",1.755,52.3},
{"HOYA","TAC6",1.755,52.3},
{"HOYA","TAC8",1.729,54.7},
{"HOYA","TAC8",1.729,54.7},
{"HOYA","TAF1",1.773,49.6},
{"HOYA","TAF1",1.773,49.6},
{"HOYA","TAF2",1.795,45.4},
{"HOYA","TAF2",1.795,45.4},
{"HOYA","TAF3",1.804,46.5},
{"HOYA","TAF3",1.804,46.5},
{"HOYA","TAF4",1.788,47.5},
{"HOYA","TAF4",1.788,47.5},
{"HOYA","TAF5",1.816,46.6},
{"HOYA","TAFD1",1.815,42.2},
{"HOYA","TAFD10",1.816,44.5},
{"HOYA","TAFD13",1.855,36.6},
{"HOYA","TAFD17",1.878,38.2},
{"HOYA","TAFD2",1.838,42},
{"HOYA","TAFD20",1.903,29.8},
{"HOYA","TAFD21",1.933,27.9},
{"HOYA","TAFD30",1.883,40.8},
{"HOYA","TAFD31",1.922,36},
{"HOYA","TAFD5",1.835,43},
{"HOYA","TAFD6",1.848,43},
{"HOYA","TAFD9",1.85,32.2},
{"HOYA","ZNC1",1.533,58.1},
{"HOYA","ZNC5",1.534,55.4},
{"HOYA","ZNC7",1.508,61.3},
{"HIKARI","E-BAK2",1.54,59.4},
{"HIKARI","E-BK7",1.517,64.1},
{"HIKARI","E-FK01",1.497,81.6},
{"HIKARI","E-FK5",1.487,70.4},
{"HIKARI","E-FKH1",1.498,82.5},
{"HIKARI","E-K3",1.518,58.9},
{"HIKARI","E-K5",1.522,59.7},
{"HIKARI","E-KF6",1.517,52.3},
{"HIKARI","E-LLF1",1.548,45.8},
{"HIKARI","E-LLF2",1.541,47.2},
{"HIKARI","E-LLF6",1.532,48.9},
{"HIKARI","E-PKH1",1.519,70},
{"HIKARI","E-BAF10",1.67,47.2},
{"HIKARI","E-BAF11",1.667,48.3},
{"HIKARI","E-BAF12",1.639,44.9},
{"HIKARI","E-BAF3",1.583,46.5},
{"HIKARI","E-BAF4",1.606,43.8},
{"HIKARI","E-BAF8",1.624,47},
{"HIKARI","E-BAK1",1.572,57.7},
{"HIKARI","E-BAK4",1.569,56.3},
{"HIKARI","E-BALF4",1.58,53.7},
{"HIKARI","E-BASF2",1.664,35.9},
{"HIKARI","E-BASF6",1.668,42},
{"HIKARI","E-BASF7",1.702,41.2},
{"HIKARI","E-BASF8",1.723,38},
{"HIKARI","E-F1",1.626,35.8},
{"HIKARI","E-F2",1.62,36.3},
{"HIKARI","E-F3",1.613,37},
{"HIKARI","E-F5",1.603,38},
{"HIKARI","E-F8",1.596,39.2},
{"HIKARI","E-LAF01",1.7,48.1},
{"HIKARI","E-LAF010",1.743,49.3},
{"HIKARI","E-LAF016",1.801,35},
{"HIKARI","E-LAF02",1.72,43.7},
{"HIKARI","E-LAF04",1.757,47.8},
{"HIKARI","E-LAF05",1.762,40.1},
{"HIKARI","E-LAF09",1.697,48.5},
{"HIKARI","E-LAF11",1.757,31.8},
{"HIKARI","E-LAF2",1.744,44.8},
{"HIKARI","E-LAF3",1.717,47.9},
{"HIKARI","E-LAF7",1.749,35.3},
{"HIKARI","E-LAF9",1.795,28.5},
{"HIKARI","E-LAFH2",1.804,33.9},
{"HIKARI","E-LAK01",1.64,60.1},
{"HIKARI","E-LAK011",1.741,52.7},
{"HIKARI","E-LAK02",1.67,57.3},
{"HIKARI","E-LAK04",1.651,56.2},
{"HIKARI","E-LAK06",1.678,50.7},
{"HIKARI","E-LAK09",1.734,51.5},
{"HIKARI","E-LAK10",1.72,50.2},
{"HIKARI","E-LAK12",1.678,55.4},
{"HIKARI","E-LAK13",1.694,53.2},
{"HIKARI","E-LAK14",1.697,55.5},
{"HIKARI","E-LAK18",1.729,54.7},
{"HIKARI","E-LAK7",1.652,58.5},
{"HIKARI","E-LAK8",1.713,53.9},
{"HIKARI","E-LAK9",1.691,54.8},
{"HIKARI","E-LAKH1",1.748,52.3},
{"HIKARI","E-LASF01",1.786,44.2},
{"HIKARI","E-LASF010",1.834,37.2},
{"HIKARI","E-LASF013",1.804,39.6},
{"HIKARI","E-LASF014",1.788,47.4},
{"HIKARI","E-LASF015",1.804,46.6},
{"HIKARI","E-LASF016",1.772,49.6},
{"HIKARI","E-LASF017",1.795,45.3},
{"HIKARI","E-LASF02",1.8,42.2},
{"HIKARI","E-LASF03",1.806,40.9},
{"HIKARI","E-LASF04",1.816,44.3},
{"HIKARI","E-LASF05",1.835,42.7},
{"HIKARI","E-LASF08",1.883,40.8},
{"HIKARI","E-LASF09",1.816,46.6},
{"HIKARI","E-LASFH2",1.767,46.8},
{"HIKARI","E-LASFH6",1.806,33.3},
{"HIKARI","E-LASFH9",1.903,35.7},
{"HIKARI","E-LASKH2",1.755,52.3},
{"HIKARI","E-LF5",1.581,40.7},
{"HIKARI","E-LF6",1.567,42.7},
{"HIKARI","E-LF7",1.575,41.5},
{"HIKARI","E-PSK02",1.618,63.4},
{"HIKARI","E-PSK03",1.603,65.5},
{"HIKARI","E-SF03",1.847,23.8},
{"HIKARI","E-SF1",1.717,29.5},
{"HIKARI","E-SF10",1.728,28.5},
{"HIKARI","E-SF11",1.785,25.7},
{"HIKARI","E-SF13",1.741,27.8},
{"HIKARI","E-SF14",1.762,26.6},
{"HIKARI","E-SF15",1.699,30.1},
{"HIKARI","E-SF2",1.648,33.8},
{"HIKARI","E-SF4",1.755,27.5},
{"HIKARI","E-SF5",1.673,32.1},
{"HIKARI","E-SF6",1.805,25.4},
{"HIKARI","E-SF7",1.64,34.6},
{"HIKARI","E-SF8",1.689,31.1},
{"HIKARI","E-SFS3",1.785,26.3},
{"HIKARI","E-SK10",1.632,57},
{"HIKARI","E-SK11",1.564,60.7},
{"HIKARI","E-SK14",1.603,60.7},
{"HIKARI","E-SK15",1.623,58.2},
{"HIKARI","E-SK16",1.62,60.3},
{"HIKARI","E-SK2",1.607,56.8},
{"HIKARI","E-SK4",1.613,58.7},
{"HIKARI","E-SK5",1.589,61.2},
{"HIKARI","E-SSK2",1.622,53.1},
{"HIKARI","E-SSK5",1.658,50.9},
{"HIKARI","E-SSK8",1.618,49.8},
{"HIKARI","F2",1.62,36.3},
{"HIKARI","F5",1.603,38.1},
{"HIKARI","LAF11",1.757,31.7},
{"HIKARI","LAF7",1.75,35.2},
{"HIKARI","LAF9",1.795,28.6},
{"HIKARI","S-SK18",1.639,55.5},
{"HIKARI","SF03",1.847,23.9},
{"HIKARI","SF13",1.741,27.6},
{"HIKARI","SF5",1.673,32.2},
{"HIKARI","SF8",1.689,31.2},
{"HAYWARD","BF1",1.617,38.5},
{"HAYWARD","BF2",1.605,43.4},
{"HAYWARD","BF3",1.649,35.8},
{"HAYWARD","BF4",1.65,38.4},
{"HAYWARD","BF5",1.623,38.2},
{"HAYWARD","BF6",1.612,45},
{"HAYWARD","BSC1",1.511,63.5},
{"HAYWARD","BSC2",1.517,64.5},
{"HAYWARD","BSC5",1.505,66.5},
{"HAYWARD","C1",1.523,58.5},
{"HAYWARD","C2",1.513,60.5},
{"HAYWARD","C3",1.508,61},
{"HAYWARD","CF1",1.53,51.8},
{"HAYWARD","CF2",1.525,51.2},
{"HAYWARD","CF3",1.514,54.8},
{"HAYWARD","CF4",1.502,56.7},
{"HAYWARD","DBC1",1.611,58.8},
{"HAYWARD","DBC10",1.613,59.5},
{"HAYWARD","DBC2",1.617,55},
{"HAYWARD","DBC3",1.611,57.2},
{"HAYWARD","DBC4",1.589,61},
{"HAYWARD","DBC5",1.637,55.5},
{"HAYWARD","DBC6",1.623,57.2},
{"HAYWARD","DBC7",1.62,60},
{"HAYWARD","DBC8",1.607,59.5},
{"HAYWARD","DBC9",1.651,55.8},
{"HAYWARD","DBF2",1.657,51.2},
{"HAYWARD","DBF3",1.67,47.4},
{"HAYWARD","DBF4",1.645,48},
{"HAYWARD","DBF5",1.7,41},
{"HAYWARD","DBF6",1.671,52},
{"HAYWARD","DF1",1.605,38},
{"HAYWARD","DF2",1.617,36.6},
{"HAYWARD","DF3",1.621,36.2},
{"HAYWARD","DF4",1.649,33.9},
{"HAYWARD","DF5",1.61,37.5},
{"HAYWARD","DF6",1.636,35},
{"HAYWARD","DF7",1.626,35.7},
{"HAYWARD","DF8",1.654,34},
{"HAYWARD","EDF2",1.673,32.2},
{"HAYWARD","EDF3",1.72,29.2},
{"HAYWARD","EDF4",1.689,30.9},
{"HAYWARD","EDF5",1.786,25.6},
{"HAYWARD","EDF7",1.667,32.3},
{"HAYWARD","EDF8",1.728,28.3},
{"HAYWARD","LBC1",1.541,59.8},
{"HAYWARD","LBC2",1.572,57.5},
{"HAYWARD","LBC3",1.574,56.8},
{"HAYWARD","LBC4",1.588,53.3},
{"HAYWARD","LBC5",1.569,56},
{"HAYWARD","LF1",1.573,42.5},
{"HAYWARD","LF2",1.58,41},
{"HAYWARD","LF3",1.576,41.3},
{"HAYWARD","LF4",1.583,41.8},
{"HAYWARD","LF5",1.549,45.7},
{"CORNING","A63-65",1.465,65.7},
{"CORNING","A86-82",1.486,81.8},
{"CORNING","A87-70",1.487,70.4},
{"CORNING","A87-84",1.487,84.4},
{"CORNING","A88-66",1.488,65.8},
{"CORNING","B10-63",1.51,63.5},
{"CORNING","B11-60",1.511,60.4},
{"CORNING","B16-64",1.517,64.2},
{"CORNING","B18-60",1.518,60.3},
{"CORNING","B18-65",1.518,65.1},
{"CORNING","B19-61",1.519,61.4},
{"CORNING","B23-59",1.523,59.4},
{"CORNING","B29-52",1.529,51.8},
{"CORNING","B29-77",1.529,76.9},
{"CORNING","B39-59",1.54,59.7},
{"CORNING","B41-48",1.541,47.6},
{"CORNING","B42-73",1.542,73.4},
{"CORNING","B48-46",1.548,45.7},
{"CORNING","B48-53",1.548,53.5},
{"CORNING","B52-67",1.552,67},
{"CORNING","B58-53",1.558,52.5},
{"CORNING","B58-72",1.558,72},
{"CORNING","B64-61",1.564,60.9},
{"CORNING","B69-56",1.569,56},
{"CORNING","B72-57",1.573,57.5},
{"CORNING","B81-41",1.581,40.8},
{"CORNING","B89-61",1.589,61.2},
{"CORNING","B96-67",1.596,67.4},
{"CORNING","C04-38",1.604,38.1},
{"CORNING","C04-64",1.604,64},
{"CORNING","C06-44",1.606,43.9},
{"CORNING","C07-57",1.607,56.6},
{"CORNING","C09-59",1.609,59},
{"CORNING","C13-44",1.613,44.4},
{"CORNING","C13-56",1.614,56.4},
{"CORNING","C13-58",1.613,58.5},
{"CORNING","C16-44",1.616,44.4},
{"CORNING","C17-55",1.618,55.1},
{"CORNING","C20-36",1.62,36.3},
{"CORNING","C20-60",1.62,60.3},
{"CORNING","C23-53",1.623,53.1},
{"CORNING","C23-57",1.623,56.9},
{"CORNING","C23-58",1.623,58.1},
{"CORNING","C24-47",1.624,46.9},
{"CORNING","C26-36",1.626,35.6},
{"CORNING","C37-35",1.637,35.3},
{"CORNING","C38-42",1.638,42.4},
{"CORNING","C39-56",1.639,55.5},
{"CORNING","C41-60",1.641,60.1},
{"CORNING","C42-58",1.642,58.1},
{"CORNING","C48-34",1.648,33.9},
{"CORNING","C48-46",1.648,46.2},
{"CORNING","C51-39",1.65,39.2},
{"CORNING","C51-56",1.651,55.9},
{"CORNING","C52-58",1.652,58.5},
{"CORNING","C55-40",1.655,40.1},
{"CORNING","C57-57",1.658,57.2},
{"CORNING","C58-51",1.659,51},
{"CORNING","C67-33",1.667,33.1},
{"CORNING","C67-48",1.667,48.4},
{"CORNING","C70-47",1.67,47.1},
{"CORNING","C72-32",1.673,32.3},
{"CORNING","C78-56",1.678,55.5},
{"CORNING","C89-31",1.689,31.2},
{"CORNING","C90-50",1.689,49.6},
{"CORNING","C90-55",1.691,54.8},
{"CORNING","C97-36",1.696,36.5},
{"CORNING","C97-55",1.697,55.4},
{"CORNING","C97-56",1.697,56.2},
{"CORNING","C99-30",1.699,30.1},
{"CORNING","D01-41",1.702,41.1},
{"CORNING","D13-54",1.713,53.8},
{"CORNING","D17-29",1.717,29.5},
{"CORNING","D17-48",1.717,48},
{"CORNING","D20-50",1.72,50.3},
{"CORNING","D23-38",1.724,38},
{"CORNING","D28-28",1.728,28.4},
{"CORNING","D31-28",1.731,28.4},
{"CORNING","D34-26",1.734,26.5},
{"CORNING","D34-51",1.734,51.4},
{"CORNING","D40-28",1.74,28.1},
{"CORNING","D44-45",1.744,44.8},
{"CORNING","D56-27",1.755,27.6},
{"CORNING","D62-27",1.762,26.9},
{"CORNING","D85-25",1.785,25.8},
{"CORNING","D85-26",1.785,25.9},
{"CORNING","D88-47",1.788,47.4},
{"CORNING","D96-43",1.796,43.2},
{"CORNING","E00-46",1.8,45.6},
{"CORNING","E03-47",1.803,46.4},
{"CORNING","E05-25",1.805,25.5},
{"CORNING","E37-30",1.836,30.2},
{"CORNING","E65-40",1.865,40.1},
{"CORNING","E78-38",1.878,38.5},
{"(unknown) RUSSIA","C001",1.44,68.8},
{"(unknown) RUSSIA","C002",1.487,70},
{"(unknown) RUSSIA","C004",1.49,65.1},
{"(unknown) RUSSIA","C005",1.478,65.6},
{"(unknown) RUSSIA","C007",1.47,66.8},
{"(unknown) RUSSIA","C009",1.483,66.3},
{"(unknown) RUSSIA","C011",1.471,68.5},
{"(unknown) RUSSIA","C021",1.52,69.1},
{"(unknown) RUSSIA","C022",1.547,67.6},
{"(unknown) RUSSIA","C024",1.58,65.1},
{"(unknown) RUSSIA","C040",1.608,65.2},
{"(unknown) RUSSIA","C050",1.522,59.6},
{"(unknown) RUSSIA","C052",1.5,66},
{"(unknown) RUSSIA","C054",1.51,63.4},
{"(unknown) RUSSIA","C056",1.516,64.1},
{"(unknown) RUSSIA","C059",1.515,60.6},
{"(unknown) RUSSIA","C061",1.534,55.5},
{"(unknown) RUSSIA","C063",1.519,60.4},
{"(unknown) RUSSIA","C064",1.519,61.7},
{"(unknown) RUSSIA","C066",1.526,60.2},
{"(unknown) RUSSIA","C081",1.53,60.5},
{"(unknown) RUSSIA","C083",1.54,59.7},
{"(unknown) RUSSIA","C085",1.547,62.8},
{"(unknown) RUSSIA","C088",1.569,56.1},
{"(unknown) RUSSIA","C093",1.559,61.2},
{"(unknown) RUSSIA","C107",1.572,57.5},
{"(unknown) RUSSIA","C109",1.611,55.8},
{"(unknown) RUSSIA","C111",1.614,55.1},
{"(unknown) RUSSIA","C113",1.617,54.1},
{"(unknown) RUSSIA","C115",1.569,62.9},
{"(unknown) RUSSIA","C117",1.604,60.6},
{"(unknown) RUSSIA","C119",1.613,60.6},
{"(unknown) RUSSIA","C121",1.613,58.4},
{"(unknown) RUSSIA","C124",1.628,59.4},
{"(unknown) RUSSIA","C125",1.622,56.7},
{"(unknown) RUSSIA","C127",1.657,51.1},
{"(unknown) RUSSIA","C129",1.589,61.2},
{"(unknown) RUSSIA","C166",1.66,57.4},
{"(unknown) RUSSIA","C168",1.687,53.6},
{"(unknown) RUSSIA","C169",1.703,49.7},
{"(unknown) RUSSIA","C170",1.743,50.2},
{"(unknown) RUSSIA","C172",1.738,48.1},
{"(unknown) RUSSIA","C174",1.692,55},
{"(unknown) RUSSIA","C176",1.709,54.8},
{"(unknown) RUSSIA","C177",1.786,45.6},
{"(unknown) RUSSIA","C178",1.744,50.4},
{"(unknown) RUSSIA","C180",1.765,50.3},
{"(unknown) RUSSIA","C191",1.522,76.4},
{"(unknown) RUSSIA","C193",1.55,73.1},
{"(unknown) RUSSIA","C202",1.518,59},
{"(unknown) RUSSIA","C205",1.501,57.1},
{"(unknown) RUSSIA","C207",1.518,51.2},
{"(unknown) RUSSIA","C217",1.525,55},
{"(unknown) RUSSIA","C219",1.548,54},
{"(unknown) RUSSIA","C221",1.57,49.5},
{"(unknown) RUSSIA","C223",1.58,53.9},
{"(unknown) RUSSIA","C225",1.583,46.5},
{"(unknown) RUSSIA","C227",1.622,53.1},
{"(unknown) RUSSIA","C229",1.626,39.1},
{"(unknown) RUSSIA","C231",1.64,48.3},
{"(unknown) RUSSIA","C233",1.671,49.3},
{"(unknown) RUSSIA","C239",1.614,40},
{"(unknown) RUSSIA","C242",1.635,36.8},
{"(unknown) RUSSIA","C243",1.608,46.1},
{"(unknown) RUSSIA","C245",1.65,38.5},
{"(unknown) RUSSIA","C246",1.607,44},
{"(unknown) RUSSIA","C248",1.664,35.4},
{"(unknown) RUSSIA","C249",1.579,46.7},
{"(unknown) RUSSIA","C270",1.756,41.2},
{"(unknown) RUSSIA","C271",1.779,38.1},
{"(unknown) RUSSIA","C274",1.859,36.6},
{"(unknown) RUSSIA","C275",1.808,42.8},
{"(unknown) RUSSIA","C276",1.812,37.5},
{"(unknown) RUSSIA","C277",1.815,33.4},
{"(unknown) RUSSIA","C278",1.833,43.1},
{"(unknown) RUSSIA","C296",1.575,41.3},
{"(unknown) RUSSIA","C298",1.578,41.1},
{"(unknown) RUSSIA","C299",1.558,42},
{"(unknown) RUSSIA","C300",1.58,38},
{"(unknown) RUSSIA","C301",1.548,45.9},
{"(unknown) RUSSIA","C302",1.561,46.8},
{"(unknown) RUSSIA","C304",1.54,44.9},
{"(unknown) RUSSIA","C316",1.613,37},
{"(unknown) RUSSIA","C318",1.617,36.6},
{"(unknown) RUSSIA","C320",1.624,35.9},
{"(unknown) RUSSIA","C322",1.603,37.9},
{"(unknown) RUSSIA","C327",1.614,34.6},
{"(unknown) RUSSIA","C329",1.62,36.4},
{"(unknown) RUSSIA","C331",1.625,35.3},
{"(unknown) RUSSIA","C341",1.648,33.9},
{"(unknown) RUSSIA","C343",1.673,32.2},
{"(unknown) RUSSIA","C345",1.717,29.5},
{"(unknown) RUSSIA","C347",1.74,28.2},
{"(unknown) RUSSIA","C349",1.755,27.5},
{"(unknown) RUSSIA","C351",1.728,28.3},
{"(unknown) RUSSIA","C353",1.689,31.1},
{"(unknown) RUSSIA","C355",1.806,25.4},
{"(unknown) RUSSIA","C357",1.649,31.6},
{"(unknown) RUSSIA","C358",1.785,25.7},
{"(unknown) RUSSIA","C359",1.785,26.3},
{"(unknown) RUSSIA","C366",1.944,20.4},
{"(unknown) RUSSIA","C367",2.17,17},
{"(unknown) RUSSIA","C368",2.056,16.6},
{"(unknown) RUSSIA","C386",1.529,51.8},
{"(unknown) RUSSIA","C389",1.612,44.1},
{"(unknown) RUSSIA","C390",1.651,43.5},
{"(unknown) RUSSIA","C391",1.663,41.8},
{"(unknown) RUSSIA","C392",1.601,51},
{"(unknown) EAST GERMANY","BAF10",1.67,47.2},
{"(unknown) EAST GERMANY","BAF3",1.583,46.5},
{"(unknown) EAST GERMANY","BAF4",1.606,43.9},
{"(unknown) EAST GERMANY","BAF5",1.607,49.3},
{"(unknown) EAST GERMANY","BAF7",1.608,46.2},
{"(unknown) EAST GERMANY","BAF8",1.624,47},
{"(unknown) EAST GERMANY","BAF9",1.643,47.8},
{"(unknown) EAST GERMANY","BAK1",1.572,57.5},
{"(unknown) EAST GERMANY","BAK2",1.54,59.6},
{"(unknown) EAST GERMANY","BAK4",1.569,56},
{"(unknown) EAST GERMANY","BAK4N",1.569,56},
{"(unknown) EAST GERMANY","BAK5",1.557,58.6},
{"(unknown) EAST GERMANY","BALF1",1.562,50.9},
{"(unknown) EAST GERMANY","BALF4",1.58,53.9},
{"(unknown) EAST GERMANY","BALF5",1.547,53.6},
{"(unknown) EAST GERMANY","BALF9",1.593,51.9},
{"(unknown) EAST GERMANY","BALK2",1.518,61},
{"(unknown) EAST GERMANY","BASF1",1.626,39.1},
{"(unknown) EAST GERMANY","BASF2",1.664,35.9},
{"(unknown) EAST GERMANY","BASF3",1.607,40.3},
{"(unknown) EAST GERMANY","BASF4",1.651,38.4},
{"(unknown) EAST GERMANY","BASF5",1.603,42.5},
{"(unknown) EAST GERMANY","BASF6",1.668,41.9},
{"(unknown) EAST GERMANY","BASF7",1.702,41.2},
{"(unknown) EAST GERMANY","BASF8",1.723,38},
{"(unknown) EAST GERMANY","BASF9",1.661,41.5},
{"(unknown) EAST GERMANY","BK3",1.498,65.1},
{"(unknown) EAST GERMANY","BK6",1.531,62.1},
{"(unknown) EAST GERMANY","BK7",1.517,64.1},
{"(unknown) EAST GERMANY","BK7N",1.517,64.1},
{"(unknown) EAST GERMANY","BK9",1.494,66.1},
{"(unknown) EAST GERMANY","D362",1.62,49.4},
{"(unknown) EAST GERMANY","D401",1.697,47},
{"(unknown) EAST GERMANY","D402",1.731,46.3},
{"(unknown) EAST GERMANY","D409",1.77,40.8},
{"(unknown) EAST GERMANY","D423",1.828,30.1},
{"(unknown) EAST GERMANY","D424",1.84,28.4},
{"(unknown) EAST GERMANY","F11",1.621,36},
{"(unknown) EAST GERMANY","F16",1.599,35.4},
{"(unknown) EAST GERMANY","F2",1.62,36.4},
{"(unknown) EAST GERMANY","F3",1.613,37},
{"(unknown) EAST GERMANY","F4",1.617,36.6},
{"(unknown) EAST GERMANY","F5",1.604,38.2},
{"(unknown) EAST GERMANY","F7",1.625,35.6},
{"(unknown) EAST GERMANY","F8",1.596,39.2},
{"(unknown) EAST GERMANY","FK3",1.465,65.7},
{"(unknown) EAST GERMANY","FK5",1.487,70},
{"(unknown) EAST GERMANY","K1",1.51,61.8},
{"(unknown) EAST GERMANY","K10",1.501,56.5},
{"(unknown) EAST GERMANY","K11",1.5,61.6},
{"(unknown) EAST GERMANY","K13",1.522,59.1},
{"(unknown) EAST GERMANY","K14",1.524,58.6},
{"(unknown) EAST GERMANY","K14N",1.524,58.6},
{"(unknown) EAST GERMANY","K3",1.518,59},
{"(unknown) EAST GERMANY","KF3",1.515,54.7},
{"(unknown) EAST GERMANY","KF7",1.523,50.8},
{"(unknown) EAST GERMANY","KF8",1.511,51},
{"(unknown) EAST GERMANY","KZF1",1.551,49.6},
{"(unknown) EAST GERMANY","KZF2",1.529,51.8},
{"(unknown) EAST GERMANY","KZF3",1.524,53.1},
{"(unknown) EAST GERMANY","KZF6",1.527,51.1},
{"(unknown) EAST GERMANY","KZFS1",1.613,44},
{"(unknown) EAST GERMANY","KZFS2",1.558,53.8},
{"(unknown) EAST GERMANY","LF3",1.582,42},
{"(unknown) EAST GERMANY","LF7",1.575,41.3},
{"(unknown) EAST GERMANY","LLF2",1.541,47.3},
{"(unknown) EAST GERMANY","LLF6",1.532,48.9},
{"(unknown) EAST GERMANY","LLF7",1.549,45.4},
{"(unknown) EAST GERMANY","LLF8",1.533,45.2},
{"(unknown) EAST GERMANY","PSK1",1.548,62.8},
{"(unknown) EAST GERMANY","PSK2",1.569,63.1},
{"(unknown) EAST GERMANY","PSK3",1.552,63.5},
{"(unknown) EAST GERMANY","SF1",1.717,29.5},
{"(unknown) EAST GERMANY","SF10",1.728,28.4},
{"(unknown) EAST GERMANY","SF11",1.785,25.7},
{"(unknown) EAST GERMANY","SF13",1.741,27.7},
{"(unknown) EAST GERMANY","SF14",1.762,26.5},
{"(unknown) EAST GERMANY","SF15",1.699,30.1},
{"(unknown) EAST GERMANY","SF19",1.667,33.1},
{"(unknown) EAST GERMANY","SF2",1.648,33.9},
{"(unknown) EAST GERMANY","SF3",1.74,28.2},
{"(unknown) EAST GERMANY","SF4",1.755,27.5},
{"(unknown) EAST GERMANY","SF5",1.673,32.2},
{"(unknown) EAST GERMANY","SF6",1.805,25.5},
{"(unknown) EAST GERMANY","SF7",1.64,34.6},
{"(unknown) EAST GERMANY","SF8",1.689,31.2},
{"(unknown) EAST GERMANY","SF9",1.655,33.4},
{"(unknown) EAST GERMANY","SFS1",1.923,21},
{"(unknown) EAST GERMANY","SK10",1.623,56.9},
{"(unknown) EAST GERMANY","SK11",1.564,60.7},
{"(unknown) EAST GERMANY","SK13",1.592,58.2},
{"(unknown) EAST GERMANY","SK14",1.604,61},
{"(unknown) EAST GERMANY","SK15",1.623,58.1},
{"(unknown) EAST GERMANY","SK16",1.62,60.3},
{"(unknown) EAST GERMANY","SK18",1.639,55.6},
{"(unknown) EAST GERMANY","SK2",1.607,56.7},
{"(unknown) EAST GERMANY","SK22",1.678,55},
{"(unknown) EAST GERMANY","SK24",1.664,56.4},
{"(unknown) EAST GERMANY","SK4",1.613,58.6},
{"(unknown) EAST GERMANY","SK5",1.589,61.3},
{"(unknown) EAST GERMANY","SK6",1.614,56.3},
{"(unknown) EAST GERMANY","SK9",1.614,55.2},
{"(unknown) EAST GERMANY","SSK1",1.617,54.1},
{"(unknown) EAST GERMANY","SSK10",1.693,53.6},
{"(unknown) EAST GERMANY","SSK11",1.753,53.1},
{"(unknown) EAST GERMANY","SSK2",1.622,53.1},
{"(unknown) EAST GERMANY","SSK4",1.618,55.1},
{"(unknown) EAST GERMANY","SSK5",1.658,50.9},
{"(unknown) EAST GERMANY","SSK7",1.618,50.4},
{"(unknown) EAST GERMANY","ZK1",1.533,58},
{"(unknown) EAST GERMANY","ZK2",1.521,60.1},
{"(unknown) EAST GERMANY","ZK5",1.534,55.4},
{"(unknown) EAST GERMANY","ZK7",1.509,61.1},
{"(unknown) CHINA","BAF1",1.583,54},
{"(unknown) CHINA","BAF1",1.549,54},
{"(unknown) CHINA","BAF2",1.57,49.5},
{"(unknown) CHINA","BAF3",1.58,53.9},
{"(unknown) CHINA","BAF4",1.583,46.5},
{"(unknown) CHINA","BAF5",1.606,43.9},
{"(unknown) CHINA","BAF51",1.603,42.5},
{"(unknown) CHINA","BAF6",1.608,46.2},
{"(unknown) CHINA","BAF7",1.614,40},
{"(unknown) CHINA","BAF8",1.626,39.1},
{"(unknown) CHINA","BAK1",1.53,60.5},
{"(unknown) CHINA","BAK11",1.56,61.2},
{"(unknown) CHINA","BAK2",1.54,59.7},
{"(unknown) CHINA","BAK3",1.547,62.8},
{"(unknown) CHINA","BAK4",1.552,63.4},
{"(unknown) CHINA","BAK5",1.561,58.3},
{"(unknown) CHINA","BAK6",1.564,60.8},
{"(unknown) CHINA","BAK7",1.569,56},
{"(unknown) CHINA","BAK8",1.573,57.5},
{"(unknown) CHINA","BAK9",1.574,56.5},
{"(unknown) CHINA","F1",1.603,38},
{"(unknown) CHINA","F12",1.624,36.8},
{"(unknown) CHINA","F13",1.626,35.7},
{"(unknown) CHINA","F2",1.613,37},
{"(unknown) CHINA","F3",1.617,36.6},
{"(unknown) CHINA","F4",1.62,36.4},
{"(unknown) CHINA","F5",1.624,35.9},
{"(unknown) CHINA","F50",1.62,36.3},
{"(unknown) CHINA","F6",1.625,35.6},
{"(unknown) CHINA","F7",1.636,35.4},
{"(unknown) CHINA","FK1",1.486,81.8},
{"(unknown) CHINA","FK2",1.487,84.5},
{"(unknown) CHINA","K1",1.5,62.1},
{"(unknown) CHINA","K10",1.518,59},
{"(unknown) CHINA","K11",1.526,60.2},
{"(unknown) CHINA","K12",1.534,55.5},
{"(unknown) CHINA","K16",1.519,61.7},
{"(unknown) CHINA","K2",1.5,66},
{"(unknown) CHINA","K3",1.505,64.7},
{"(unknown) CHINA","K4",1.508,61.1},
{"(unknown) CHINA","K5",1.51,63.4},
{"(unknown) CHINA","K6",1.511,60.5},
{"(unknown) CHINA","K7",1.515,60.6},
{"(unknown) CHINA","K8",1.516,56.8},
{"(unknown) CHINA","K9",1.517,64.2},
{"(unknown) CHINA","KF1",1.501,57.2},
{"(unknown) CHINA","KF2",1.515,54.5},
{"(unknown) CHINA","KF3",1.526,51},
{"(unknown) CHINA","LAF1",1.694,49.2},
{"(unknown) CHINA","LAF10",1.788,47.4},
{"(unknown) CHINA","LAF2",1.717,47.9},
{"(unknown) CHINA","LAF3",1.744,44.9},
{"(unknown) CHINA","LAF4",1.75,35},
{"(unknown) CHINA","LAF50",1.773,49.6},
{"(unknown) CHINA","LAF6",1.757,47.8},
{"(unknown) CHINA","LAF7",1.782,37.1},
{"(unknown) CHINA","LAF8",1.784,41.3},
{"(unknown) CHINA","LAF9",1.784,43.9},
{"(unknown) CHINA","LAK1",1.66,57.4},
{"(unknown) CHINA","LAK10",1.651,55.9},
{"(unknown) CHINA","LAK11",1.665,54.6},
{"(unknown) CHINA","LAK12",1.697,56.2},
{"(unknown) CHINA","LAK2",1.692,54.5},
{"(unknown) CHINA","LAK3",1.747,51},
{"(unknown) CHINA","LAK4",1.641,60.1},
{"(unknown) CHINA","LAK5",1.678,55.5},
{"(unknown) CHINA","LAK6",1.694,53.4},
{"(unknown) CHINA","LAK7",1.713,53.8},
{"(unknown) CHINA","LAK8",1.72,50.4},
{"(unknown) CHINA","QF1",1.548,45.9},
{"(unknown) CHINA","QF11",1.578,41.1},
{"(unknown) CHINA","QF14",1.596,39.2},
{"(unknown) CHINA","QF2",1.561,46.8},
{"(unknown) CHINA","QF3",1.575,41.3},
{"(unknown) CHINA","QF5",1.582,42},
{"(unknown) CHINA","QF50",1.581,40.9},
{"(unknown) CHINA","QF6",1.532,48.8},
{"(unknown) CHINA","QF9",1.561,45.2},
{"(unknown) CHINA","QK1",1.47,66.8},
{"(unknown) CHINA","QK2",1.478,65.6},
{"(unknown) CHINA","QK3",1.477,70},
{"(unknown) CHINA","TF1",1.529,51.8},
{"(unknown) CHINA","TF3",1.612,44.1},
{"(unknown) CHINA","TF4",1.613,44.3},
{"(unknown) CHINA","TF5",1.654,39.6},
{"(unknown) CHINA","TF6",1.681,37.2},
{"(unknown) CHINA","TIF1",1.533,46},
{"(unknown) CHINA","TIF2",1.58,38},
{"(unknown) CHINA","TIF3",1.593,35.8},
{"(unknown) CHINA","TIF4",1.617,31},
{"(unknown) CHINA","TK1",1.586,61},
{"(unknown) CHINA","UK9",1.517,64.2},
{"(unknown) CHINA","UQF50",1.581,40.9},
{"(unknown) CHINA","ZBAF1",1.622,53.1},
{"(unknown) CHINA","ZBAF11",1.62,49.8},
{"(unknown) CHINA","ZBAF13",1.639,45.2},
{"(unknown) CHINA","ZBAF15",1.651,38.3},
{"(unknown) CHINA","ZBAF16",1.667,48.4},
{"(unknown) CHINA","ZBAF17",1.668,41.9},
{"(unknown) CHINA","ZBAF18",1.67,39.2},
{"(unknown) CHINA","ZBAF2",1.64,48.3},
{"(unknown) CHINA","ZBAF20",1.702,41},
{"(unknown) CHINA","ZBAF21",1.723,38},
{"(unknown) CHINA","ZBAF3",1.657,51.1},
{"(unknown) CHINA","ZBAF4",1.664,35.5},
{"(unknown) CHINA","ZBAF5",1.671,47.3},
{"(unknown) CHINA","ZBAF50",1.658,50.9},
{"(unknown) CHINA","ZBAF51",1.683,44.5},
{"(unknown) CHINA","ZF1",1.648,33.8},
{"(unknown) CHINA","ZF10",1.689,31.2},
{"(unknown) CHINA","ZF11",1.699,30.1},
{"(unknown) CHINA","ZF12",1.762,26.5},
{"(unknown) CHINA","ZF13",1.785,25.8},
{"(unknown) CHINA","ZF14",1.918,21.5},
{"(unknown) CHINA","ZF2",1.673,32.2},
{"(unknown) CHINA","ZF3",1.717,29.5},
{"(unknown) CHINA","ZF4",1.728,28.3},
{"(unknown) CHINA","ZF5",1.74,28.2},
{"(unknown) CHINA","ZF50",1.741,27.8},
{"(unknown) CHINA","ZF51",1.785,26.1},
{"(unknown) CHINA","ZF52",1.847,23.8},
{"(unknown) CHINA","ZF6",1.755,27.5},
{"(unknown) CHINA","ZF7",1.806,25.4},
{"(unknown) CHINA","ZF8",1.654,33.7},
{"(unknown) CHINA","ZK1",1.569,62.9},
{"(unknown) CHINA","ZK10",1.622,56.7},
{"(unknown) CHINA","ZK11",1.639,55.5},
{"(unknown) CHINA","ZK14",1.603,60.6},
{"(unknown) CHINA","ZK15",1.607,59.5},
{"(unknown) CHINA","ZK19",1.614,56.4},
{"(unknown) CHINA","ZK2",1.583,59.5},
{"(unknown) CHINA","ZK20",1.617,53.9},
{"(unknown) CHINA","ZK3",1.589,61.3},
{"(unknown) CHINA","ZK4",1.609,58.9},
{"(unknown) CHINA","ZK5",1.611,55.8},
{"(unknown) CHINA","ZK50",1.607,56.7},
{"(unknown) CHINA","ZK6",1.613,58.4},
{"(unknown) CHINA","ZK7",1.613,60.6},
{"(unknown) CHINA","ZK8",1.614,55.1},
{"(unknown) CHINA","ZK9",1.62,60.3},
{"(unknown) CHINA","ZLAF1",1.802,44.3},
{"(unknown) CHINA","ZLAF2",1.803,46.8},
{"(unknown) CHINA","ZLAF3",1.855,36.6},
{"(unknown) CHINA","ZLAF50",1.804,46.6},
{"(unknown) CHINA","ZLAF51",1.805,39.6},
{"(unknown) CHINA","ZLAF52",1.806,40.7},
{"(unknown) CHINA","ZLAF53",1.934,37.2},
{"(unknown) CHINA","ZLAF54",1.881,41}
};

const int NUM_ALIASES = sizeof(alias_database)/sizeof(GlassAlias);




class OpticalMaterial {

public:

  // default constructor, gives you air
  OpticalMaterial()
  {
    model = mod_sellmeier;
    coeff[0] = 1.0f;
    coeff[1] = 
    coeff[2] = 
    coeff[3] = 
    coeff[4] = 
    coeff[5] = 0.0f;
  }

  // Constructor (6x float): Initialize material with Sellmeier coefficients
  OpticalMaterial(float B1, float B2, float B3, float C1, float C2, float C3):
    model(mod_sellmeier)
  {coeff[0] = B1; coeff[1] = B2; coeff[2] = B3; coeff[3] = C1; coeff[4] = C2; coeff[5] = C3;};

  // Constructor (2x float): Look up best match of refractive index
  // and Abbe number from glass database
  OpticalMaterial(float nd, float vd, bool print_match = false);

  // Constructor (char[]): Look up Schott, Ohara, or Hoya material by name
  OpticalMaterial(const char* query, bool print_match = false);

  // same as above, but fall back to abbe if material name is not found
  OpticalMaterial(const char* query, const float nd, const float vd, bool print_match = false);

  // Return refractive index for given wavelength in nm
  float get_index(float lambda);
  
  bool init(float nd, float vd, bool print_match = false);
  bool init(const char *query, bool print_match = false);
  void init(const char *query, float nd, float vd, bool print_match = false);

  DispersionModel model;
  float coeff[6];
private:

};



OpticalMaterial::OpticalMaterial(const char *query, float nd, float vd, bool print_match)
{
    init(query, nd, vd, print_match);
}

void OpticalMaterial::init(const char *query, float nd, float vd, bool print_match)
{
    // init by mat name, fail over to ior/abbe
    if(!init(query, print_match))
        init(nd, vd, print_match);
}

OpticalMaterial::OpticalMaterial(float nd, float vd, bool print_match)
{
    init(nd, vd, print_match);
}

bool OpticalMaterial::init(float nd, float vd, bool print_match)
{
  float bestDistance 
    = __sqr(glass_database[0].nd-nd) +  1e-4 * __sqr(glass_database[0].vd-vd);
    
  GlassDef bestMaterial = glass_database[0];

  for (int i = 1; i < NUM_MATERIALS; ++i) {
    float newDistance
      = __sqr(glass_database[i].nd-nd) +  1e-4 * __sqr(glass_database[i].vd-vd);
      
    if (newDistance < bestDistance) {
      bestDistance = newDistance;
      bestMaterial = glass_database[i];
    }

  }
    
  // Copy best match over to coefficients:    
  model = bestMaterial.model;
  coeff[0] = bestMaterial.B1;
  coeff[1] = bestMaterial.B2;
  coeff[2] = bestMaterial.B3;
  coeff[3] = bestMaterial.C1;
  coeff[4] = bestMaterial.C2;
  coeff[5] = bestMaterial.C3;
    
  if (print_match) 
    printf("[OpticalMaterial by Abbe number] Best match for nd=%5.3f, vd=%5.2f: %s %s (nd=%5.3f, vd=%5.2f)\n",
	   nd,vd,bestMaterial.mfg, bestMaterial.name,bestMaterial.nd,bestMaterial.vd);

  return true;
};


OpticalMaterial::OpticalMaterial(const char* query, bool print_match)
{
    if(!init(query, print_match)) exit(-1);
}

bool OpticalMaterial::init(const char* query, bool print_match)
{
  char name[32];
  int j = 0;
  // Munch blanks in query, and convert to upper case:

  for (int i = 0; query[i]; ++i) {
    if (query[i]==' ') {}
    else if (query[i] >= 'a' && query[i] <= 'z') {
      name[j++] = query[i] + (int)'A' - (int)'a';
    } else {
      name[j++] = query[i];
    }
  }
  name[j] = 0;

  // user actually requested to not use the material, but ior/abbe, so fail and let the other method try:
  if(!strcmp(name, "ABBE")) return false;
    
  int hit_index = -1;
  float hit_quality = 0.f;

  for (int i = 0; i < NUM_MATERIALS; ++i) 
    if (strstr(glass_database[i].name, name)) {
      float quality = strlen(name) / (float) strlen(glass_database[i].name);
      if (quality > hit_quality) {
	hit_index = i;
	hit_quality = quality;
      }
    }
    
  if (hit_index >= 0) { // Partial match found!

    // If quality of match is not 100%, check for a better name match in the historical database.

    if (hit_quality < 0.99) {
      float alias_hit_quality = hit_quality;
      int alias_hit_index = -1;
      for (int i = 0; i < NUM_ALIASES; ++i) 
	if (strstr(alias_database[i].name, name)) {
	  float quality = strlen(name) / (float) strlen(alias_database[i].name);
	  if (quality > alias_hit_quality) {
	    alias_hit_index = i;
	    alias_hit_quality = quality;
	  }
	}
      
      // Test if historic material actually fits the name better
      if (alias_hit_quality > hit_quality) { 
	GlassAlias alias = alias_database[alias_hit_index];
	if (print_match) 
	  printf("[OpticalMaterial by name] Better match for \"%s\" found in historical database: \n"
		 "                          %s %s (%2.0f%%). Looking up closest Sellmeier material:\n",
		 query, alias.mfg, alias.name, 100*alias_hit_quality);
	
	float nd = alias.nd;
	float vd = alias.vd;
	OpticalMaterial abbe_material(nd, vd, print_match);
	for (int i = 0; i < 6; ++i)
	  coeff[i] = abbe_material.coeff[i];
	model = abbe_material.model;
	return true;
      }

    }

    // Else return best hit
    GlassDef hit = glass_database[hit_index];
    model = hit.model;
    coeff[0] = hit.B1;
    coeff[1] = hit.B2;
    coeff[2] = hit.B3;
    coeff[3] = hit.C1;
    coeff[4] = hit.C2;
    coeff[5] = hit.C3;
    if (print_match) 
      printf("[OpticalMaterial by name] Best match for \"%s\": %s %s (%2.0f%%)\n",
	     query,hit.mfg, hit.name, 100*hit_quality);
    return true;
  }


  if(print_match)
    printf("\x1B[31mERROR: [OpticalMaterial by name] No match found for query \"%s\"\n\x1B[0m",query);
  return false;
};

float OpticalMaterial::get_index(float lambda) {

  switch(model) {

  case mod_sellmeier:
    lambda = lambda * 0.001;
    return sqrt(coeff[0] * lambda * lambda/(lambda*lambda - coeff[3]) 
		+ coeff[1] * lambda * lambda/(lambda*lambda - coeff[4])
		+ coeff[2] * lambda * lambda/(lambda*lambda - coeff[5])
		+ 1);
    break;

  default:
    return -1;
    break;

  }

};
#endif








