import java.lang.Math;

public class Vertex{
    
    public double x,y,z;
    
    public Vertex() {
	x = 0;
	y = 0;
	z = 0;
    }
    
    public Vertex(double x, double y, double z) {
	this.x = x;
	this.y = y;
	this.z = z;
    }
    
    public double getx() { return x;}
    public double gety() { return y;}
    public double getz() { return z;}
    
    public double[] getCoords(){
	double[] c = new double[3];
	
	c[0] = x;
	c[1] = y;
	c[2] = z;
	return c;
    }

    public void update( double a, double b, double c ) {
	x=a; y=b; z=c;
    }
    
    public void setCoords(double x, double y, double z) {
	this.x = x;
	this.y = y;
	this.z = z;
    }

    public static boolean sameVertex(Vertex v1, Vertex v2){
	return v1 == v2 ||
	    (v1.x == v2.x &&
	     v1.y == v2.y &&
	     v1.z == v2.z);
    }

    public static boolean collinear(Vertex v1, Vertex v2, Vertex v3){
	double x1 = v1.x, y1 = v1.y, z1 = v1.z;
	double x2 = v2.x, y2 = v2.y, z2 = v2.z;
	double x3 = v3.x, y3 = v3.y, z3 = v3.z;
	return (z3 - z1) * (y2 - y1) - (z2 - z1) * (y3 - y1) == 0 &&
	    (z2 - z1) * (x3 - x1) - (x2 - x1) * (z3 - z1) == 0 &&
	    (x2 - x1) * (y3 - y1) - (y2 - y1) * (x3 - x1) == 0;
    }

    /**
     * Returns the pair of vertices that bound three collinear vertices.
     *
     * @param  v1, v2, v3  the vertices to test
     * @return An array of two vertices that bound the tested vertices
     */
    public static Vertex[] bounds(Vertex v1, Vertex v2, Vertex v3){
	Vertex min12, max12;
	
	// Compare v1 and v2
	if (v1.x == v2.x){
	    if (v1.y == v2.y){
		if (v1.z <= v2.z){
		    min12 = v1;
		    max12 = v2;
                }else{
		    min12 = v2;
		    max12 = v1;
                }
            }
	    else if (v1.y < v2.y){
		min12 = v1;
		max12 = v2;
            }else{
		min12 = v2;
		max12 = v1;
            }
        }else if (v1.x < v2.x){
	    min12 = v1;
	    max12 = v2;
        }else{
	    min12 = v2;
	    max12 = v1;
        }

	Vertex[] bounds = new Vertex[2];

	// Compare min of v1, v2 with v3
	if (min12.x == v3.x){
	    if (min12.y == v3.y){
		bounds[0] = min12.z <= v3.z ? min12 : v3;
            }else{
		bounds[0] = min12.y < v3.y ? min12 : v3;
            }
        }else{
	    bounds[0] = min12.x < v3.x ? min12 : v3;
        }
	
	
	// Compare max of v1, v2 with v3
	if (max12.x == v3.x){
	    if (max12.y == v3.y){
		bounds[1] = max12.z >= v3.z ? max12 : v3;
            }else{
		bounds[1] = max12.y > v3.y ? max12 : v3;
            }
        }else{
	    bounds[1] = max12.x > v3.x ? max12 : v3;
        }
	return bounds;
    }

    /**
     * Returns the square of the distance between two vertices.
     *
     * @param v1, v2  the vertices
     * @return        the distance squared between them
     */
    public static double distanceSq(Vertex v1,Vertex v2){
	double x1 = v1.x, y1 = v1.y, z1 = v1.z;
	double x2 = v2.x, y2 = v2.y, z2 = v2.z;
	return (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2) + (z1 - z2) * (z1 - z2);
    }

    /**
     * Returns the distance between two vertices.
     *
     * @param v1, v2  the vertices
     * @return        the distance between them
     */
    public static double distance(Vertex v1,Vertex v2){
	return Math.sqrt(Vertex.distanceSq(v1,v2));
    }
    
    /**
     * Returns the square of the norm of a vector
     *
     * @param v  the vector
     * @return   the square of the norm of v
     */
    public double normSq(){
	return x*x+y*y+z*z;
    }

     /**
     * Returns the norm of a vector
     *
     * @param v  the vector
     * @return   the norm of v
     */
    public double norm(){
	return Math.sqrt(x*x+y*y+z*z);
    }
    
    public void normalize(){
	double n = norm();
	x=x/n;
	y=y/n;
	z=z/n;
    }

    //Returns the angle between the two vectors
    public static double angle(Vertex v1,Vertex v2){
	return Math.acos(dotProduct(v1,v2)/(v1.norm()*v2.norm()));
    }
    
    //Returns the dot product of two vectors
    public static double dotProduct(Vertex v1,Vertex v2){
	double x1 = v1.x, y1 = v1.y, z1 = v1.z;
	double x2 = v2.x, y2 = v2.y, z2 = v2.z;
	return x1 * x2 + y1 * y2 + z1 * z2;
    }

    //Returns the vector difference between two vertices
    public static Vertex vectorDiff(Vertex v1,Vertex v2){
	return new Vertex(v1.x - v2.x,v1.y - v2.y,v1.z - v2.z);
    }
    
    //Returns the vector addition between two vertices
    public static Vertex vectorAdd(Vertex v1,Vertex v2){
	return new Vertex(v1.x + v2.x,v1.y + v2.y,v1.z + v2.z);
    }

    //Returns the vector multiple by k
    public static Vertex vectorMult(Vertex v,double k){
	return new Vertex(k*v.x, k*v.y, k*v.z);
    }

    /**
     * Returns the cross product of two vectors
     *
     * @param v1, v2  the vectors
     * @return        v1 x v2
     */
    public static Vertex crossProduct(Vertex v1,Vertex v2){
	return new Vertex(v1.y * v2.z - v1.z * v2.y,
			  v1.z * v2.x - v1.x * v2.z,
			  v1.x * v2.y - v1.y * v2.x);
    }
    
    /* Returns the dot product with another vector v */
    public double dot( Vertex v ) {
	return( x*v.x + y*v.y + z*v.z );
    }
    
    public void add(Vertex v1, Vertex v2) {
	x=v1.x+v2.x;
	y=v1.y+v2.y;
	z=v1.z+v2.z;
    }
    
    public void sub(Vertex v1, Vertex v2) {
	x=v1.x-v2.x;
	y=v1.y-v2.y;
	z=v1.z-v2.z;
    }
    
    public void cross(Vertex v1, Vertex v2) { 
	x=v1.y*v2.z - v1.z*v2.y;
	y=v1.z*v2.x - v1.x*v2.z;
	z=v1.x*v2.y - v1.y*v2.x;
    }
    
    public void scalar(Vertex v, double d) {
	x=d*v.x;
	y=d*v.y;
	z=d*v.z;
    }
    
    public double unit( Vertex v) {
	double tmp = 1/v.norm();
	scalar(v, tmp);
	return tmp;
    }
    
    public String toString() {
	return ("("+x+","+y+","+z+")");
    }
}
