package cve;
import ve.*;

/**
 * A rule collection is a collection of generalised rules. This is
 * much like a Vector, however we probably want to add more indexing
 * at some time.
 *
 * @author David Poole
 * @version 0.1 2001-02-28
 **/

public class RuleCollection {

    private GenRule[] theRules;
    private int size;
    private int maxSize;
    private final int defaultInitialMaxSize=10;

    /** generate an empty rule collection.
     **/
    RuleCollection() {
	theRules=new GenRule[defaultInitialMaxSize];
	maxSize=defaultInitialMaxSize;
	size=0;
    }
    /** generate an empty rule collection with the given initial size.
     **/
    RuleCollection(int initialSize) {
	theRules=new GenRule[initialSize];
	maxSize=initialSize;
	size=0;
    }

    /** add a new element to the collection.
     **/
    public void add(GenRule newrule) {
	if (size == maxSize) {  //double the size
	    maxSize *= 2;
	    GenRule[] newRules = new GenRule[maxSize];
	    for (int i=0; i< size; i++) {
		newRules[i]=theRules[i];
	    }
	    theRules=newRules;
	}
	theRules[size++]=newrule;
    }

    /*
     * Returns an iterator over the values of the factor. This code is
     * based on the Iterator code in java.util.AbstractList.
     */
    public RuleIterator iterator() {
	return new Itr();
    }

    private class Itr implements RuleIterator {
	int curpos=0;
	/** 
	 * returns true if there are more elements. 
	 */
	public boolean hasNext() {
	    return curpos < size;
	}
	/** 
	 * returns the next element in the iterator.
	 */
	public GenRule next() {
	    return theRules[curpos++];
	}
	
    }

    /**
     * eliminates the variable from the current collection of rules.
     **/
    public RuleCollection eliminate(Variable var) {
	RuleCollection result = new RuleCollection();
	CoveringRuleCollection coveringRules = new CoveringRuleCollection();
	RuleCollection toAbsorb = new RuleCollection();
	RuleIterator ruleItr = this.iterator();
	while (ruleItr.hasNext()) {
	    GenRule nextRule=ruleItr.next();
	    if (nextRule.ruleFor(var)) {
		coveringRules.add(nextRule);
	    }
	    else if (nextRule.getContext().involves(var)) {
		toAbsorb.add(nextRule);
	    }
	    else if (nextRule.getFactor().contains(var)) {
		toAbsorb.add(nextRule);
	    }
	    else {
		result.add(nextRule);
	    }
	}
	// Absorb all of the rules that are not part of the covering rules
	ruleItr = toAbsorb.iterator();
	while (ruleItr.hasNext()) {
	    coveringRules=coveringRules.absorb(ruleItr.next());
	}
	// Sum out the rules
	ruleItr = coveringRules.iterator();
	RuleCollection[] varInContext = new RuleCollection[var.getDomain().length];
	Variable[] tosumout={var};
	while (ruleItr.hasNext()) {
	    GenRule nextRule=ruleItr.next();
	    if (nextRule.getFactor().contains(var)) {
		// var is in the factor
		//System.out.println("\n Variable "+var.getName()+" is in the factor");
		GenRule newRule = new GenRule(nextRule.getContext(),
				       new FactorSumOut(nextRule.getFactor(),tosumout),
				       removeVar(nextRule.getRulesFor(),var)
				       );
		if (! newRule.redundant()) {
		    result.add(newRule);
		    System.out.print("  Rule added ");
		    newRule.printBrief();
		}
	    }
	    else {
		// var is in the context
		int val = nextRule.getContext().lookup(var);
		//System.out.println("\n Variable "+var.getName()+" is in the context (with value "+val);
		GenRule newRule = new GenRule(nextRule.getContext().remove(var),
					      nextRule.getFactor(),
					      removeVar(nextRule.getRulesFor(),var)
					      );
		if (varInContext[val]==null) {
		    varInContext[val] = new RuleCollection();
		}
		varInContext[val].add(newRule);
		    System.out.print("  Rule added ");
		    newRule.printBrief();
	    }
	}
	// sum over the resulting rules with var in the context
	if (varInContext[0] != null) {
	    // there are rules with var in the context
	    RuleCollection ruleSum = varInContext[0];
	    for (int i=1; i<varInContext.length; i++) {
		ruleSum = ruleCollectionSum(ruleSum,varInContext[i]);
	    }
	    ruleItr = ruleSum.iterator();
	    while (ruleItr.hasNext()) {
		GenRule newRule = ruleItr.next();
		if (! newRule.redundant()) {
		    result.add(newRule);
		}
	    }
	}
	return result;
    }

    /**
     * update the factors to make the appropriate observations. This
     * assumes that the observed variables are sorted, and we only
     * observe variables that are in the store.
     *
     **/
    public RuleCollection observe(Variable[] observedVars, int[] observedVals) {
	if (observedVars.length == 0) {
	    return this;
	}
	else {
	    RuleCollection result = new RuleCollection(this.size);
	    for (int i=0; i<this.size; i++) {
		theRules[i].observe(observedVars,observedVals,result);
	    }
	    return result;
	}
    }

    /**
     * removes variable var from the list of variables vars. This
     * assumes that var is a member of vars.
     **/
    private static Variable[] removeVar(Variable[] vars, Variable var) {
//  	System.out.print("Removing "+var.getName()+" from [");
//  	for (int i=0; i<vars.length; i++) {
//  	    System.out.print(vars[i].getName()+" ");
//  	}
//  	System.out.println("]");
	Variable[] result = new Variable[vars.length-1];
	int rpos=0;
	for (int i=0; i<vars.length; i++) {
	    if (vars[i] != var) {
		result[rpos++]=vars[i];
	    }
	}
	return result;
    }

    /**
     * constructs a new rule collection by adding two other rule collections.
     */
    private static RuleCollection ruleCollectionSum(RuleCollection c1, RuleCollection c2) {
	RuleCollection result=new RuleCollection();
	RuleIterator it1=c1.iterator();
	while (it1.hasNext()) {
	    GenRule r1=it1.next();
	    RuleIterator it2=c2.iterator();
	    while (it2.hasNext()) {
		GenRule r2=it2.next();
		if (r1.getContext().compatible(r2.getContext())) {
		    result.add(new GenRule(
					   r1.getContext().union(r2.getContext()),
					   new FactorPlus(factorSet(r1.getFactor(),r2.getContext()),
							  factorSet(r2.getFactor(),r1.getContext())),
					   unionVariableArrays(r1.getRulesFor(),r2.getRulesFor())));
		}
	    }
	}
	return result;
    }
    
    /**
     * returns the union of the two variable arrays.
     */
    public static Variable[] unionVariableArrays(Variable[] a1,Variable[] a2){
	if (a1==a2) {
	    return(a1);
	}
	int i=0;
	int i1=0;
	int i2=0;
	// count size of resulting array
	while (i1 < a1.length && i2 < a2.length) {
	    if (a1[i1]==a2[i2]) {
		i++; i1++; i2++;
	    }
	    else if (a1[i1].getId() < a2[i2].getId()) {
		i++; i1++;
	    }
	    else {
		i++; i2++;
	    }
	}
	Variable[] result = new Variable[i+a1.length-i1+a2.length-i2];
	i=0;i1=0;i2=0;
	while (i1 < a1.length && i2 < a2.length) {
	    if (a1[i1]==a2[i2]) {
		result[i]=a1[i1];
		i++; i1++; i2++;
	    }
	    else if (a1[i1].getId() < a2[i2].getId()) {
		result[i]=a1[i1];
		i++; i1++;
	    }
	    else {
		result[i]=a2[i2];
		i++; i2++;
	    }
	}
	while (i1 < a1.length) {
		result[i]=a1[i1];
		i++; i1++;
	}
	while (i2 < a2.length) {
		result[i]=a2[i2];
		i++; i2++;
	}
	return result;
    }

    /**
     * prints the rule collection.
     **/
    public void print() {
	System.out.println("\n\n*** Rule Collection ***");
	int count=0;
	for (int i=0; i<size; i++) {
	    theRules[i].print();
	    System.out.println();
	    count += theRules[i].getFactor().size();
	}
	System.out.println("Number of rules: "+size+". Numer of parameters:"+count);
    }

    /**
     * prints the summary of the rule collection.
     **/
    public void printBrief() {
	System.out.println("\n*** Rule Collection ***");
	int count=0;
	for (int i=0; i<size; i++) {
	    theRules[i].printBrief();
	    count += theRules[i].getFactor().size();
	}
	System.out.println("Number of rules: "+size+". Numer of parameters:"+count);
    }

    /**
     * set the variables in fac to the context con
     **/
    protected static Factor factorSet(Factor fac, Context con) {
	if (disjoint(fac.getVariables(),con.getVars())) {
	    return fac;
	}
	else {
	    return new FactorObserved(fac,con.getVars(),con.getVals());
	}
    }

    private static boolean disjoint(Variable[] vs1, Variable[] vs2) {
	int pos1=0;
	int pos2=0;
	while (pos1<vs1.length && pos2<vs2.length) {
	    if (vs1[pos1]==vs2[pos2]) {
		return false;
	    }
	    else if (vs1[pos1].getId()<vs2[pos2].getId()) {
		pos1++;
	    }
	    else {
		pos2++;
	    }
	}
	return true;
    }

}
