package cve;
import ve.*;

/**
 * A context is a tuple of variables and their values.

 * @author David Poole
 * @version 0.1 2001-02-27
*/

public class Context {

    private Variable[] contextVars;
    private int[] contextVals;

    /**
     * returns the array of variables for the context.
     */
    public Variable[] getVars() {
	return contextVars;
    }

    /**
     * returns the array of values for the context.
     */
    public int[] getVals() {
	return contextVals;
    }

    /**
     * constructs a new context with variables vars and values vals.

     * @param vars the array of variables for the new context
     * @param vals the array of values for the new context
     **/
    Context(Variable[] vars, int[] vals) {
	contextVars=vars;
	contextVals=vals;
    }

    /**
     * constructs a new context with variables vars and values vals,
     * given that len is the number of observed variables.

     * @param vars the array containing variables for the new context
     * @param vals the array containing values for the new context
     * @param int the numner of variables in the context
     **/
    Context(Variable[] vars, int[] vals, int len) {
	contextVars=new Variable[len];
	contextVals=new int[len];
	for (int i=0; i<len; i++) {
	    contextVars[i]=vars[i];
	    contextVals[i]=vals[i];
	}
    }

    /**
     * constructs the empty context.
     **/
    Context() {
	contextVars = new Variable[0];
	contextVals = new int[0];
    }

    /**
     * constructs a new context that is oldcon together with
     * var=val. You can assume that var is not assigned in oldcon.

     * @param oldcon the context being expanded
     * @param var a variable that doesn't appear in oldcon
     * @param val a value in the domain of var
     **/
    Context(Context oldcon, Variable var, int val) {
	int len=oldcon.getVars().length;
	contextVars=new Variable[len+1];
	contextVals=new int[len+1];
	int i=0;
	while (i<len && oldcon.getVars()[i].getId() < var.getId()) {
	    contextVars[i]=oldcon.getVars()[i];
	    contextVals[i]=oldcon.getVals()[i];
	    i++;
	}
	contextVars[i]=var;
	contextVals[i]=val;
	while (i<len) {
	    contextVars[i+1]=oldcon.getVars()[i];
	    contextVals[i+1]=oldcon.getVals()[i];
	    i++;
	}

    }

    /**
     * is true if this context is compatible with c1.
     *
     * @param c1 a context to compare this one to.
     * @return true if if this context is compatible with c1.
     **/
    public boolean compatible(Context c1) {
	int thispos=0;
	int c1pos=0;
	while (thispos<contextVars.length && c1pos<c1.getVars().length) {
	    if (contextVars[thispos]==c1.getVars()[c1pos]) {
		if (contextVals[thispos]!=c1.getVals()[c1pos]) {
		    return false;
		}
		else {
		    thispos++;
		    c1pos++;
		}
	    }
	    else if (contextVars[thispos].getId()<c1.getVars()[c1pos].getId()) {
		thispos++;
	    }
	    else {
		c1pos++;
	    }
	}
	return true;
    }

    /**
     * returns the union of this context and c1. We assume that the
     * contexts are compatible.
     *
     * @param c1 a context to compare this one to.
     * @return a context that the union of this context and c1.
     **/
    public Context union(Context c1) {
	int maxlen = contextVars.length+c1.getVars().length;
	Variable[] newVars = new Variable[maxlen];
	int[] newVals = new int[maxlen];
	int thispos=0;
	int c1pos=0;
	int newpos=0;
	while(thispos<contextVars.length && c1pos<c1.getVars().length) {
	    if (contextVars[thispos]==c1.getVars()[c1pos]) {
		newVars[newpos]=contextVars[thispos];
		newVals[newpos++]=contextVals[thispos++];
		c1pos++;
	    }
	    else if (contextVars[thispos].getId()<c1.getVars()[c1pos].getId()) {
		newVars[newpos]=contextVars[thispos];
		newVals[newpos++]=contextVals[thispos++];
	    }
	    else {
		newVars[newpos]=c1.getVars()[c1pos];
		newVals[newpos++]=c1.getVals()[c1pos++];
	    }
	}
	while(thispos<contextVars.length) {
	    newVars[newpos]=contextVars[thispos];
	    newVals[newpos++]=contextVals[thispos++];
	}
	while (c1pos<c1.getVars().length) {
		newVars[newpos]=c1.getVars()[c1pos];
		newVals[newpos++]=c1.getVals()[c1pos++];
	}
	if (newpos==contextVars.length) {
	    return this;
	}
	if (newpos==c1.getVars().length) {
	    return c1;
	}
	Variable[] finalVars = new Variable[newpos];
	int[] finalVals = new int[newpos];
	for (int i=0; i<newpos; i++) {
	    finalVars[i]=newVars[i];
	    finalVals[i]=newVals[i];
	}
	return new Context(finalVars,finalVals);
    }
    
    /**
     * is true if this context involves variable var.
     **/
    public boolean involves(Variable var) {
	// this could perhaps be improved by taking the ordering into
	// account, but I doubt it!
	for (int i=0; i<contextVars.length; i++) {
	    if (var==contextVars[i]) {
		return true;
	    }
	}
	return false;
    }

    /**
     * returns the value of variable var. This assumes that var has a value.
     **/
    public int lookup(Variable var) {
	// this could perhaps be improved by taking the ordering into
	// account, but I doubt it!
	for (int i=0; i<contextVars.length; i++) {
	    if (var==contextVars[i]) {
		return contextVals[i];
	    }
	}
	return -1;
    }

    /**
     * returns a new context with var removed. This assumes that var
     * is in the context.
     **/
    public Context remove(Variable var) {
	Variable[] newVars = new Variable[contextVars.length-1];
	int[] newVals = new int[contextVars.length-1];
	int newId=0;
	for (int i=0; i<contextVars.length; i++) {
	    if (var!=contextVars[i]) {
		newVars[newId]=contextVars[i];
		newVals[newId++]=contextVals[i];
	    }
	}
	return new Context(newVars,newVals);
    }

    public void print() {
	if (contextVars.length==0) {
	    System.out.print("true");
	}
	else {
	    System.out.print(contextVars[0].getName());
	    System.out.print("=");
	    System.out.print(contextVars[0].getDomain()[contextVals[0]]);
	    for (int i=1; i<contextVars.length; i++) {
		System.out.print(" & ");
		System.out.print(contextVars[i].getName());
		System.out.print("=");
		System.out.print(contextVars[i].getDomain()[contextVals[i]]);

	    }
	}
    }
}
