package ve;

/**
 * This the the class of factors that are determistic (they have only
 * 0-1 probabilities). This is stored much more efficiently as you
 * only need to store which value has the probability of 1. This is
 * the class that is returned when we ask for the policy function in
 * {@link FactorMax}
 *
 * Copyright 2001, David Poole. All rights reserved.
 *
 * @author David Poole
 * @version 0.1 2001-06-15
 **/

public class FactorDeterministic extends Factor {
    private int[] theAction;
    private Variable[] theParents;
    private Variable decisionVar;
    private int preSize;
    private int postSize;
    private int curSize;

    public FactorDeterministic(Variable[] pars, Variable var, int[]actions) {
	super(insert(var,pars),BY_DETERMINISTIC);
	theParents = pars;
	decisionVar = var;
	preSize=1;
	int pos=0;
	while (pos<pars.length && pars[pos].getId()<var.getId()) {
	    preSize *= pars[pos++].getDomain().length;
	}
	curSize=var.getDomain().length;
	postSize=1;
	while (pos<pars.length) {
	    postSize *= pars[pos++].getDomain().length;
	}
	theAction = actions;
    }

    /** prints the policy function with no indent.
     **/
    public void printPolicyFunction() {
	printPolicyFunction("");
    }

    /** prints the policy function with indent inserted at the start
     *of every line..
     **/
    public void printPolicyFunction(String indent) {
	System.out.print(indent);
	for (int i=0; i<getVariables().length; i++) {
	    System.out.print(getVariables()[i].getName());
	    System.out.print("\t");
	}
	System.out.println(decisionVar.getName()+"?");
	int[] index= new int[getVariables().length];
	int policyIndex = 0;
	while (policyIndex < theAction.length) {
	    // output the index
	    System.out.print(indent);
	    for (int i=0; i< getVariables().length; i++) {
		System.out.print(getVariables()[i].getDomain()[index[i]]);
		System.out.print("\t");
	    }
	    // output the value
	    System.out.println(decisionVar.getDomain()[theAction[policyIndex++]]);
	    //	    System.out.print("\n");
	    // update the index
	    boolean allMax=true;
	    for (int i=getVariables().length-1; i>=0 && allMax; i--) {
		if (index[i]<getVariables()[i].getDomain().length-1) {
		    index[i]++;
		    allMax=false;
		}
		else {
		    index[i]=0;
		}
	    }
	}

    }

    /*
     * Returns an iterator over the values of the factor. This code is
     * based on the Iterator code in java.util.AbstractList.
     */
    public EltsIterator iterator() {
	return new Itr();
    }

    private class Itr implements EltsIterator {
	/** 
	 * Index of element to be returned by subsequent call to next.
	 */
	int prepos = 0;
	int curpos = 0;
	int postpos = 0;
	int actionpos = 0;

	public boolean hasNext() {
	    return prepos+1<preSize || curpos+1<curSize || postpos<postSize;
	}

	public double next() {
	    if (postpos == postSize) {
		if (curpos+1<curSize) {
		    curpos++;
		}
		else {
		    curpos=0;
		    prepos++;
		}
		actionpos = prepos*postSize;
		postpos=0;		
	    }
	    postpos++;
	    if (curpos == theAction[actionpos++]) {
		return 1.0;
	    }
	    else {
		return 0.0;
	    }
	}
	
	public int currPos() {
	    return (prepos*curSize + curpos)*postSize+postpos;
	}

	public void backTo(int pos) {
	    postpos = pos % postSize;
	    pos = pos/postSize;
	    curpos = pos % curSize;
	    prepos = pos / curSize;
	}
    }

    /**
     * updates the decision function to have a different action at
     * position pos. This does no error checking.
     *
     * @param pos the position in the total ordering of the values of
     * the parents.
     * @param val the new value at this position (the index into the
     * domain of the variable.
     **/
    public void update(int pos, int val) {
	theAction[pos]=val;
    }

    private static Variable[] insert(Variable var , Variable[] vars) {
	Variable[] result = new Variable[vars.length+1];
	int pos = 0;
	while (pos<vars.length && vars[pos].getId()< var.getId()) {
	    result[pos]=vars[pos++];
	}
	result[pos]=var;
	while (pos<vars.length) {
	    result[pos+1]=vars[pos++];
	}
	return result;
    }

}
