package ve;

/**
 * A factor is a table that given a tuple of values returns a value.
 * This is an abstract class that can be instantiated in many ways,
 * for example the {@link FactorTimes} factor that
 * is the product of other factors.

 * @author David Poole
 * @version 0.1 2001-01-18
*/
public abstract class Factor {

    /**
     * some constants so it can remember how it was created.
     **/

    public static final int BY_CPT = 0;          // FactorCPT
    public static final int BY_V_ELIM = 1;
    public static final int BY_OBSERVED = 2;     // FactorObserved
    public static final int BY_SUM_OUT = 3;      // FactorSumOut
    public static final int BY_MULTIPLYING = 4;  // FactorTimes
    public static final int BY_ASSIGN = 5;       // FactorAssign
    public static final int BY_EXPANDING = 6;    // FactorExpand
    public static final int BY_MAXIMIZING = 7;   // FactorMax
    public static final int BY_NORMALIZING = 8;  // FactorNormalise
    public static final int BY_PLUS = 9;         // FactorPlus
    public static final int BY_REORDERING = 10;  // FactorReorder
    public static final int BY_SAVEIT = 11;      // FactorSaveIt
    public static final int BY_UPDATABLE = 12;   // FactorUpdatable
    public static final int BY_UNIFORM = 13;     // FactorUniform
    public static final int BY_GENRULE = 14;     // GenRule
    public static final int BY_DETERMINISTIC = 15;// FactorDeterministic


    /**
     * returns the integer representing how it was created
     **/

    public int howCreated() {
	return howCreatedVal;
    }

    private int howCreatedVal;

    private static boolean savingForTracing=true;
    /**
     * returns true if the intermediate Factors are being saved to
     * allow for tracing of how a value was computed.
     **/
    public boolean getSavingForTracing() {
	return savingForTracing;
    }
    /**
     * sets the property that the intermediate Factors are being saved
     * to allow for tracing of how a value was computed. Setting it to
     * true uses more space but allows for tracing; setting it to
     * false allows the intermediate Factors to be garbage collected.
     **/
    public void setSavingForTracing(boolean val) {
	savingForTracing = val;
    }


    /**
     * constructs a factor for the variables given.
     */
    public Factor(Variable[] vars, int how) {
	variables=vars;
	thesize=1;
	for (int i=0; i<variables.length; i++) {
	    thesize *= variables[i].getDomain().length;
	}
	howCreatedVal = how;
    }

    /**
     * returns string the variables this is a factor on.
     **/
    public String getName() {
	String nameString = new String("[");
	if (variables.length>0) {
	    nameString += variables[0].getName();
	}
	for (int i = 1; i < variables.length; i++) {
	    nameString += ", " + variables[i].getName();
	}
	return nameString+"]";
    }
    /**
     * variables is the tuple of variables, in order.
    */
    private Variable[] variables;

    /**
     * returns the tuple of variables, in order.
    */
    public Variable[] getVariables() {
	return variables;
    }

    /**
     * Returns an iterator over the values of the factor.
     */
    abstract public  EltsIterator iterator();

    /**
     * the size of the factor table. */
    private int thesize;

    /**
     * returns the size of a factor table.
     */
    public int size(){
	return thesize;
    }

    /**
     * returns true if the factor contains variable v. This should
     * probably do a binary search, although hopefully the factors
     * aren't too big!
     *
     * @param v a variables
     * @return true if v is in the current factor.
     **/
    public boolean contains(Variable v) {
	for (int i=0; i<variables.length; i++) {
	    if (variables[i]==v) {
		return true;
	    }
	}
	return false;
    }

    /**
     * returns a string representation of the Factor.
     **/
    public String toString(){
	StringBuffer output = new StringBuffer();
	for (int i=0; i<variables.length; i++) {
	    output.append(variables[i].getName()).append("\t");
	}
	output.append("Value\n");
	int[] index= new int[variables.length];
	EltsIterator iter = this.iterator();
	while (iter.hasNext()) {
	    // output the index
	    for (int i=0; i< variables.length; i++) {
		output.append(variables[i].getDomain()[index[i]]).append("\t");
	    }
	    // output the value
	    output.append(iter.next()).append("\n");
	    if (iter.hasNext()) {
		// update the index
		boolean allMax=true;
		for (int i=variables.length-1; i>=0 && allMax; i--) {
		    if (index[i]<variables[i].getDomain().length-1) {
			index[i]++;
			for (int j=i+1; j<variables.length; j++) {
			    index[j]=0;
			}
			allMax=false;
		    }
		}
	    }
	}
	return output.toString();
    }

    /**
     * prints the variables
     **/
    public void printVariables() {
	if (variables.length>0) {
	    System.out.print(variables[0].getName());
	    for (int i=1; i<variables.length; i++) {
		System.out.print(", "+variables[i].getName());
	    }
	}
    }

    /**
     * prints the Factor in a table form.
     **/
    public void print(){
	print("");
    }
    /**
     * prints the Factor in a table form, with a string that is
     * printed at the start of each line.
     **/
    public void print(String indent){
	System.out.print(indent);
	for (int i=0; i<variables.length; i++) {
	    System.out.print(variables[i].getName());
	    System.out.print("\t");
	}
	System.out.print("Value\n");
	int[] index= new int[variables.length];
	EltsIterator iter = this.iterator();
	while (iter.hasNext()) {
	    // output the index
	    System.out.print(indent);
	    for (int i=0; i< variables.length; i++) {
		System.out.print(variables[i].getDomain()[index[i]]);
		System.out.print("\t");
	    }
	    // output the value
	    System.out.println(iter.next());
	    //	    System.out.print("\n");
	    // update the index
	    boolean allMax=true;
	    for (int i=variables.length-1; i>=0 && allMax; i--) {
		if (index[i]<variables[i].getDomain().length-1) {
		    index[i]++;
		    allMax=false;
		}
		else {
		    index[i]=0;
		}
	    }
	}
    }

    /**
     * Is true if the factor consists entirely of ones.
     **/
    public boolean allOnes() {
	EltsIterator itr = iterator();
	while (itr.hasNext()) {
	    if (itr.next() != 1.0) {  // do we need a fudge factor???
		return false;
	    }
	}
	return true;
    }

    /**
     * is true if the variable is redundant in the factor. It is
     * redundant if the function represented by the factor has the
     * same value for all of the values of the other variables.
     **/
    public boolean isRedundant(Variable var) {
	int before=1; // domain before var
	int dom = var.getDomain().length;
	int after=1;  //domain after var
	int pos=0;
	while (variables[pos] != var) {
	    before *= variables[pos++].getDomain().length;
	}
	pos++;
	while (pos < variables.length) {
	    after *= variables[pos++].getDomain().length;
	}
	EltsIterator[] its = new EltsIterator[dom];
	for (int i=0; i< dom; i++) {
	    its[i] = this.iterator();
	}
	pos=0;
	for (int b=0; b<before; b++) {
	    for(int i=0; i<dom; i++) {
		its[i].backTo(pos);
		pos+=after;
	    }
	    for (int a=0; a<after; a++) {
		double val = its[0].next();
		for (int i=1; i<dom; i++) {
		    if (val != its[i].next()) {
			return false;
		    }
		}
	    }
	}
	return true;
    }


    /**
     * is true if the variable is redundant in the factor within a threshold. It is
     * redundant if the function represented by the factor has the
     * same value, within the threshold limit for all of the values of the other variables.
     **/
    public boolean isRedundant(Variable var, double threshold) {
	int before=1; // domain before var
	int dom = var.getDomain().length;
	int after=1;  //domain after var
	int pos=0;
	while (variables[pos] != var) {
	    before *= variables[pos++].getDomain().length;
	}
	pos++;
	while (pos < variables.length) {
	    after *= variables[pos++].getDomain().length;
	}
	EltsIterator[] its = new EltsIterator[dom];
	for (int i=0; i< dom; i++) {
	    its[i] = this.iterator();
	}
	pos=0;
	for (int b=0; b<before; b++) {
	    for(int i=0; i<dom; i++) {
		its[i].backTo(pos);
		pos+=after;
	    }
	    for (int a=0; a<after; a++) {
		double min = its[0].next();
		double max = min;
		for (int i=1; i<dom; i++) {
		    double nextval = its[i].next();
		    if (max < nextval) {
			max = nextval;
		    }
		    else if (min > nextval) {
			min=nextval;
		    }
		    if (max-min > threshold) {
			return false;
		    }
		}
	    }
	}
	return true;
    }

    /**
     * displays the resultant factor.
     **/
    public void display() {
    }


}
