package ve;

import java.util.*;

/**
 * A DecisionNetwork contains a tuple of Variables (each with a domain)
 * and a set of conditional probability tables, each represented as a
 * {@link FactorCPT}.

 * @author David Poole
 * @version 0.1 2001-02-29
 **/

public class DecisionNetwork {
    /**
     * The variables of the decision network, in order. The ordering of
     * indexes to that array should be consistent with the ordering of
     * the variables, but the indexes don't have to coincide with the
     * IDs of the variables. The value variables.length does not give the
     * actual number of variables; use getNumVariables() to determine
     * the number of variables. 
     **/
    protected Variable[] variables;

    /**
     * get the variables of the decision network, in order. The
     * ordering of indexes to that array should be consistent with the
     * ordering of the variables, but the indexes don't have to
     * coincide with the IDs of the variables. The value
     * getVariables().length does not give the actual number of
     * variables; use getNumVariables() to determine the number of
     * variables.
     **/
    public Variable[] getVariables(){
	return variables;
    }

    /**
     * factors representing conditional probability tables asssociates
     * with the variables. Decision Variables do not have CPTs, so
     * there are generally fewer factors than variables.
     **/
    protected Factor[] probFactors;

    /**
     * return the conditional probability tables asssociated with the
     * variables. Decision Variables do not have CPTs, so
     * there are generally fewer factors than variables. The value
     * getProbFactors().length does not give the actual number of
     * factors; use getNumProbFactors() to determine the number of
     * factors.
     **/
    public Factor[] getProbFactors() {
	return probFactors;
    }

    /*
     * The number of random variables in the decision network.
     */
    protected int numVariables;


    /*
     * returns number of random variables in the decision network.
     */
     public int getNumVariables() {
	return numVariables;
    }

    /*
     * The number of factors in the decision network.
     */
    protected int numProbFactors;


    /*
     * returns number of factors in the decision network.
     */
     public int getNumProbFactors() {
	return numProbFactors;
    }

    /** the maximum domain size in an input factor. */
    protected final int maxDomainSize=100;
    /** maximum number of variables in an input factor. */
    protected final int maxFactorSize=20;

    /** maps the string name to the variable. */
    //protected HashMap stringToVar = new HashMap();
    protected Hashtable stringToVar = new Hashtable();     // Java 1.1 compatible

    /** returns a map from the string to a variable.
     **/
    public Hashtable getStringToVar() {
	return stringToVar;
    }

    protected DecisionNetwork() {
	numVariables=0;
	numProbFactors=0;
    }

    public DecisionNetwork(Variable[] vars, int numVars) {
	variables = vars;
	numVariables = numVars;
	numProbFactors=0;
    }
    /**
     * Prints the belief network
     **/
    public void print() {
	for (int i=0; i<numProbFactors; i++) {
	    System.out.println("\n Factor #"+i+":");
	    probFactors[i].print();
	}
    }
    /**
     * Prints a summary of the belief network
     **/
    public void printBrief() {
	System.out.println("\n*** Decision Network ***");
	int count=0;
	for (int i=0; i<numProbFactors; i++) {
	    System.out.print("Factor #"+i+": [");
	    this.getProbFactors()[i].printVariables();
	    System.out.println("]. Size="+this.getProbFactors()[i].size());
	    count+=this.getProbFactors()[i].size();
	}
	System.out.println("Number of factors: "+numProbFactors+". Numer of parameters:"+count);
    }

    /**
     * returns the number of paramters in the decision network.
     **/

    public int getSize() {
	int count=0;
	for (int i=0; i<numProbFactors; i++) {
	    count+=this.getProbFactors()[i].size();
	}
	return count;
    }

    static protected void mysort(Variable[] vars) {
	// Should be just Arrays.sort(vars) but we wanted it to be
	// Java 1.1 compatible This is just an insertion sort as we
	// expect the number of variables to be small (<20)
	for (int i=1; i<vars.length; i++) {
	    Variable var=vars[i];
	    int j=i-1;
	    while (j>=0 && vars[j].getId()>var.getId()) {
		vars[j+1]=vars[j];
		j--;
	    }
	    vars[j+1]=var;
	}
    }

}


