package ve;

/**
 * a place to store factors during computation. This is a naive
 * implementation. A better implementation would do better indexing.
  *
 * @author David Poole
 * @version 0.1 2001-04-04
 **/


public class FactorStore {
    private Factor[] theFactors;
    private int numFactors;
    private int factorMaxSize;
    private Variable[] theVariables;
    private int numVariables;
    private int variableMaxSize;

    private final int defaultInitialMaxSize=10;

    /** generate an empty store.
     **/
    public FactorStore() {
	theFactors=new Factor[defaultInitialMaxSize];
	factorMaxSize=defaultInitialMaxSize;
	numFactors=0;
	theVariables=new Variable[defaultInitialMaxSize];
	variableMaxSize=defaultInitialMaxSize;
	numVariables=0;
    }

    /** generate a containing the given initial factors and initial variables.
     **/
    public FactorStore(Factor[] initialFactors, int numFactors, Variable[] initialVariables, int numVars) {
	theFactors=new Factor[numFactors];
	for (int i=0; i<numFactors; i++) {
	    theFactors[i]=initialFactors[i];
	}
	factorMaxSize=numVars;
	this.numFactors=numFactors;
	theVariables=new Variable[numVars];
	for (int i=0; i<numVars; i++) {
	    theVariables[i]=initialVariables[i];
	}
	variableMaxSize=numVars;
	this.numVariables=numVars;
    }

    /**
     * update the factors to make the appropriate observations. This
     * assumes that the observed variables are sorted, and we only
     * observe variables that are in the store.
     **/
    public void observe(Variable[] observedVars, int[] observedVals) {
	if (observedVars.length != 0) {
	    for (int i=0; i<numFactors; i++) {
		theFactors[i]=new FactorObserved(theFactors[i],observedVars,observedVals);
	    }
	    // compact the set of variables
	    int newvarspos=0;
	    int obspos=0;
	    for (int i=0; i<numVariables; i++) {
		if (obspos==observedVars.length ||
		    theVariables[i] != observedVars[obspos]) {
		    theVariables[newvarspos++]=theVariables[i];
		}
		else {
		    obspos++;
		}
	    }
	    numVariables=newvarspos;
	}
    }
    
    /**
     * sets some variables as the query variable. This assumes that
     * the query variables are in the canonical order, and are all
     * appear in the store.
     **/
    public void setQuery(Variable[] queried) {
	// do we need to remember these for some reason??
	// we just remove them from the set of variables
	int newvarspos=0;
	int qpos=0;
	for (int i=0; i<numVariables; i++) {
	    if (qpos==queried.length ||
		theVariables[i] != queried[qpos]) {
		theVariables[newvarspos++]=theVariables[i];
	    }
	    else {
		qpos++;
	    }
	}
	numVariables=newvarspos;
    }

    /**
     * removes the irrelevant variables (and their associated factors)
     * from the store.
     **/
    public void removeIrrelevantVariables() {
	// this needs to be implemented. It only affects efficiency,
	// not correctness.
    }

    /** add a new element to the store.
     **/
    public void add(Factor newFactor) {
	if (numFactors == factorMaxSize) {  //double the store
	    factorMaxSize *= 2;
	    Factor[] newFactors = new Factor[factorMaxSize];
	    for (int i=0; i< numFactors; i++) {
		newFactors[i]=theFactors[i];
	    }
	    theFactors=newFactors;
	}
	theFactors[numFactors++]=newFactor;
    }


    /**
     * returns an iterator over the variables that need to be
     * eliminated. This code is based on the Iterator code in
     * java.util.AbstractList.  This may destroy the set of variables
     * (and the factors).
     *
     * @param how one of "sequential" or "random"
     **/
    public VariableIterator variableSelector(String how) {
	// we also want to implement "minimal deficiency" ...
	if (how == "sequential") {
	    return new VarItr();
	}
	if (how == "random") {
	    return new VarItrRand();
	}
	else {
	    System.out.print("Sorry "+how+" selector is not implemented.");
	    return new VarItr();
	}
    }

    /**
     * returns an iterator over the variables that need to be
     * eliminated for the given elimination ordering.
     *
     * @param eo an elimination ordering
     **/
    public VariableIterator variableSelector(Variable[] eo) {
	return new VarItrEO(eo);
    }

    private class VarItr implements VariableIterator {
	int curpos=0;
	/** 
	 * returns true if there are more elements. 
	 */
	public boolean hasNext() {
	    return curpos < numVariables;
	}
	/** 
	 * returns the next element in the iterator.
	 */
	public Variable next() {
	    return theVariables[curpos++];
	}
	
    }

    private class VarItrEO implements VariableIterator {
	Variable[] elimOrd;
	int curpos;

	VarItrEO(Variable[] eo) {
	    elimOrd=eo;
	    curpos=0;
	}
	/** 
	 * returns true if there are more elements. 
	 */
	public boolean hasNext() {
	    return curpos < elimOrd.length;
	}
	/** 
	 * returns the next element in the iterator.
	 */
	public Variable next() {
	    return elimOrd[curpos++];
	}
	
    }
    private class VarItrRand implements VariableIterator {
	int curpos=0;
	/** 
	 * returns true if there are more elements. 
	 */
	public boolean hasNext() {
	    return numVariables>0;
	}
	/** 
	 * returns the next element in the iterator.
	 */
	public Variable next() {
	    int index = (int) (Math.random() * numVariables);
	    Variable result=theVariables[index];
	    theVariables[index]=theVariables[--numVariables];
	    return result;
	}
	
    }
    /**
     * Returns an iterator over the factors that contain the variable
     * var. Note that we remove the factors as we access them. We
     * should really index properly so that this can be done in time
     * proportional to the number of factors that contain the
     * variable, not linear in the total number of factors.
     **/
    public FactorIterator emunFactorsContaining(Variable var) {
	Factor[] toReturn = new Factor[numFactors];
	int posToReturn=0;
	int newfacpos=0;
	for (int pos=0; pos<numFactors; pos++) {
	    if (theFactors[pos].contains(var)) {
		toReturn[posToReturn++]=theFactors[pos];
	    }
	    else {
		theFactors[newfacpos++]=theFactors[pos];
	    }
	}
	numFactors=newfacpos;
	return new FacItr(toReturn,posToReturn);
    }

    /**
     * Returns an iterator over the the remaining factors.
     **/
    public FactorIterator emunFactors() {
	return new FacItr(theFactors,numFactors);
    }

    private class FacItr implements FactorIterator {
	int curpos=0;
	Factor[] toReturn;
	int numToReturn;

	FacItr(Factor[] toReturn, int numToReturn) {
	    this.toReturn = toReturn;
	    this.numToReturn = numToReturn;
	}

	/** 
	 * returns true if there are more elements. 
	 */
	public boolean hasNext() {
	    return curpos < numToReturn;
	}
	/** 
	 * returns the next element in the iterator.
	 */
	public Factor next() {
	    return toReturn[curpos++];
	}
	
    }
}
