package ve;

/**
 * This the the class of factors that extend other factors to include
 * more variables. It is used in {@link FactorTimes}. The
 * values can be enumerated as needed using the {@link EltsIterator}.

 * @author David Poole
 * @version 0.1 2001-01-25 */
public class FactorExpand extends Factor {

    /**
     * constructs a factor with the given variables.
     *
     * @param f1 the factor to be expanded
     * @param vars the list of variables for the resulting factor
     */
    public FactorExpand(Factor f1, Variable[] vars) {
	super(vars,BY_EXPANDING);
	fact = f1;
	valsExpand = new int[vars.length];
	valsBetween = new int[vars.length+1];
	// we need to construct the tables of the sizes for iterating over.
	int varsIndex=0;
	int f1Index=0;
	int numExpBlocks=0;
	valsBetween[0]=1;
	while (varsIndex < vars.length) {
	    // variables in common
	    while (varsIndex < vars.length && f1Index<f1.getVariables().length 
		   && vars[varsIndex]==f1.getVariables()[f1Index]) {
		valsBetween[numExpBlocks] *= vars[varsIndex].getDomain().length;
		varsIndex++; f1Index++;
	    }
	    if (varsIndex < vars.length) {
		// variables to be expanded
		valsExpand[numExpBlocks] =1;
		while (varsIndex < vars.length && 
		       ( f1Index >= f1.getVariables().length ||
			 vars[varsIndex].getId() < f1.getVariables()[f1Index].getId())){
		    valsExpand[numExpBlocks] *= vars[varsIndex++].getDomain().length;
		}
		valsBetween[++numExpBlocks]=1;
	    }
	}
	numExpandBlocks = numExpBlocks;
    }

    /**
     * the factor to be expanded.
     */
    private Factor fact;
    
    /**
     * the values associated with the variables to be expanded. We
     * don't keep track of the variables, just the number of variables
     * in the domains. */
    private int[] valsExpand;

    /**
     * the values associated with the variables between those to be
     * expanded. valsBetween[i] is the values between variables given
     * by valsExpand[i-1] and valsExpand[i] */
    private int[] valsBetween;

    /**
     * the number of blocks of variables to be exanded.
     */
    private int numExpandBlocks;

    // Iterators

    /**
     * Returns an iterator over the values of the factor. This code is
     * based on the Iterator code in java.util.AbstractList.
     */
    public EltsIterator iterator() {
	return new Itr();
    }

    private class Itr implements EltsIterator {
	/** 
	 * Array that specifies the count of the number of values to
	 * be expanded. This starts from the number of values and
	 * reduces to zero.  */
	int[] countExpand = new int[numExpandBlocks];
	/** 
	 * Array that specifies the count of the number of values in
	 * that are not expanded.
	 */
	int[] countBetween = new int[numExpandBlocks+1];
	/** 
	 * Array that specifies where the values have to go back to.
	 */
	int[] backTos  = new int[numExpandBlocks];

	EltsIterator factItr = fact.iterator();

	/**
	 * default constructor. This
	 */
	Itr() {
	    for (int i=0; i<numExpandBlocks; i++) {
		countExpand[i]=valsExpand[i];
		countBetween[i]=valsBetween[i];
	    }
	    countBetween[numExpandBlocks]=valsBetween[numExpandBlocks];
	}

	public boolean hasNext() {
	    if (countBetween[numExpandBlocks]>0) {
		return true;}
	    for (int i=0; i<numExpandBlocks; i++) {
		if (countExpand[i]>1) {return true;}
		if (countBetween[i]>1) {return true;}
	    }
	    return false;
	}

	/**
	 * returns the next value if there is one.
	 */
	public double next() {
	    if (countBetween[numExpandBlocks] >0){
		// normal case
		countBetween[numExpandBlocks]--;
		return factItr.next();
	    }
	    for (int i=numExpandBlocks-1; i>=0; i--) {
		if (countExpand[i]>1) {
		    countExpand[i]--;
		    factItr.backTo(backTos[i]);
		    for (int j=i+1; j<numExpandBlocks; j++) {
			countExpand[j]=valsExpand[j];
			countBetween[j]=valsBetween[j];
			backTos[j]=backTos[i];
		    }
		    countBetween[numExpandBlocks]=valsBetween[numExpandBlocks]-1;
		    return factItr.next();
		}
		else if (countBetween[i]>1) {
		    countBetween[i]--;
		    for (int j=i; j<numExpandBlocks; j++) {
			countExpand[j]=valsExpand[j];
			countBetween[j+1]=valsBetween[j+1];
			backTos[j]=factItr.currPos();
		    }
		    countBetween[numExpandBlocks]--;
		    return factItr.next();
		}
	    }
	    // We should never get here unless hasNext is false
	    return 100.0f;
	}
	
	public int currPos() {
	    int pos=valsBetween[0]-countBetween[0];
	    for (int i=0; i<numExpandBlocks; i++) {
		pos= ((pos+1)*valsExpand[i]-countExpand[i]+1)*valsBetween[i+1]
		    - countBetween[i+1];
	    }
	    //traceCurrPos(pos);
	    return pos;
	}

	public void backTo(int pos) {
	    int orgpos=pos;
	    for (int i=numExpandBlocks-1; i>=0; i--){
		countBetween[i+1]=valsBetween[i+1]-pos%valsBetween[i+1];
		pos/=valsBetween[i+1];
		countExpand[i]=valsExpand[i]-pos%valsExpand[i];
		pos/=valsExpand[i];
	    }
	    countBetween[0]=valsBetween[0]-pos;
	    // determine where fact should go back to
	    int factBackTo=pos;
	    for (int i=1; i<=numExpandBlocks; i++) {
		factBackTo=(factBackTo+1)*valsBetween[i]-countBetween[i];
	    }
	    computeReturnTos(numExpandBlocks-1,valsBetween[numExpandBlocks]);
	    //traceBackTo(orgpos,factBackTo);
	    factItr.backTo(factBackTo);
	}

	/**
	 * compute the retrunTos array for a backTo. It generates this
	 * from the countBetween array. The value it returns is the
	 * backto position assuming all of the right-most digits are
	 * zero. 
	 *
	 * @param pos the position 
	 * @param posval the */

	 
	private int computeReturnTos(int pos, int posval) {
	    int sigdigs=0;
	    if (pos>0) {
		sigdigs = computeReturnTos(pos-1,posval*valsBetween[pos]);}
	    return backTos[pos]=(valsBetween[pos]-countBetween[pos])*posval+sigdigs;
	}
	private void traceCurrPos(int pos) {
	    System.out.println("\nTracing FactorExpand Iterator CurrPos");
	    System.out.println("   Position =     "+pos);
	    System.out.print("   valsExpand   = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(valsExpand[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   valsBetween  = ");
	    for (int i=0; i<=numExpandBlocks; i++) {
		System.out.print(valsBetween[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   countExpand  = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(countExpand[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   countBetween = ");
	    for (int i=0; i<=numExpandBlocks; i++) {
		System.out.print(countBetween[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   backTos      = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(backTos[i]);
		System.out.print(" ");
	    }
	    System.out.println("\n");
	}

	private void traceBackTo(int pos, int factBackTo) {
	    System.out.println("\nTracing FactorExpand Iterator BackTo");
	    System.out.println("   Position =     "+pos);
	    System.out.print("   valsExpand   = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(valsExpand[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   valsBetween  = ");
	    for (int i=0; i<=numExpandBlocks; i++) {
		System.out.print(valsBetween[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   countExpand  = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(countExpand[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   countBetween = ");
	    for (int i=0; i<=numExpandBlocks; i++) {
		System.out.print(countBetween[i]);
		System.out.print(" ");
	    }
	    System.out.print("\n   backTos      = ");
	    for (int i=0; i<numExpandBlocks; i++) {
		System.out.print(backTos[i]);
		System.out.print(" ");
	    }
	    System.out.println("\n   factBackTo = "+factBackTo+"\n");
	}
    }
}
