package cve;
import ve.*;

/**
 * A generalised rule is a context together with a factor.

 * @author David Poole
 * @version 0.1 2001-02-28
*/

public class GenRule {

    private Context ruleContext;

    /**
     * returns the context.
     */
    public Context getContext() {
	return ruleContext;
    }

    private Factor ruleFactor;
    /**
     * returns the factor of the generalised rule.
     **/
    public Factor getFactor() {
	return ruleFactor;
    }

    /**
     * the variables that this is a rule for.
     **/
    private Variable[] rulesFor;
    /**
     * is true if this rule is a rule for variable var.
     **/
    public boolean ruleFor(Variable var) {
	for (int i=0; i<rulesFor.length; i++) {
	    if (rulesFor[i]==var) {
		return true;
	    }
	}
	return false;
    }
    /**
     * returns the sorted lists of rules this is a rule for
     */
    public Variable[] getRulesFor() {
	return rulesFor;
    }

    /**
     * generate a new instance from context con and factor fact.
     **/
    public GenRule(Context con,Factor fact,Variable[] ruleIsFor) {
	ruleContext = con;
	ruleFactor = fact;
	rulesFor = ruleIsFor;
//  	System.out.println("Generating a rule");
//  	this.print();
    }

    /**
     * generated the new GenRule <true,{}>
     **/
    public GenRule() {
	ruleContext = new Context();
	ruleFactor = new ve.FactorUpdatable( new Variable[0], 1.0);
	rulesFor = new Variable[0];
    }
	
    /**
     * adds the residuals of this with respect to Context con to result.
     *
     * @param con a Context
     * @param result a Rule Collection of GenRule to add the new rules to.
     */
    public void addResiduals(Context con, RuleCollection result) {
	int thispos=0;
	int conpos=0;
	Context currRuleContext=ruleContext;
	Factor currRuleFactor=ruleFactor;
	while (thispos<ruleContext.getVars().length 
	       && conpos<con.getVars().length) {
	    if (ruleContext.getVars()[thispos]==con.getVars()[conpos]) {
		thispos++;
		conpos++;
	    }
	    else if (ruleContext.getVars()[thispos].getId()
		     < con.getVars()[conpos].getId()) {
		thispos++;
	    }
	    else { // found a variable assigned in con and not in ruleContext
		Variable[] newVar = {con.getVars()[conpos]};
		for (int val=0; val< con.getVars()[conpos].getDomain().length; val++) {
		    if (val != con.getVals()[conpos]) {
			int[] newVal = {val};
			result.add(new GenRule(
					       new Context(currRuleContext,
							   con.getVars()[conpos],
							   val),
					       new FactorObserved(currRuleFactor,newVar,newVal),
					       this.getRulesFor()));
		    }
		}
		if (conpos+1<con.getVars().length) {
		    currRuleContext=
			new Context(currRuleContext,
				    con.getVars()[conpos],
				    con.getVals()[conpos]);
		    int[] newValInCon = {con.getVals()[conpos]};
		    currRuleFactor = new FactorObserved(currRuleFactor,newVar,newValInCon);
		}
		conpos++;
	    }
	}
	while (conpos<con.getVars().length){ 
	    // found a variable assigned in con and not in ruleContext
	    Variable[] newVar = {con.getVars()[conpos]};
	    for (int val=0; val< con.getVars()[conpos].getDomain().length; val++) {
		if (val != con.getVals()[conpos]) {
		    int[] newVal = {val};
		    result.add(new GenRule(
					   new Context(currRuleContext,
						       con.getVars()[conpos],
						       val),
					   new FactorObserved(currRuleFactor,newVar,newVal),
					   this.getRulesFor()));
		}
	    }
	    if (conpos+1<con.getVars().length) {
		currRuleContext=
		    new Context(currRuleContext,
				con.getVars()[conpos],
				con.getVals()[conpos]);
		int[] newValInCon = {con.getVals()[conpos]};
		currRuleFactor = new FactorObserved(currRuleFactor,newVar,newValInCon);
	    }
	    conpos++;
	}
    }

    /**
     * adds this rule to a rule collection given the observation.
     **/
    public void observe(Variable[] observedVars, int[] observedVals,RuleCollection resultingRules) {
	boolean ruleChanged=false;
	int conpos=0;    // position in the current context
	int obspos=0;    // position in the observations
	int respos=0;    // position in the results context
	Variable[] resultConVars = new Variable[ruleContext.getVars().length];
	int[] resultConVals = new int[ruleContext.getVars().length];
	while (conpos<ruleContext.getVars().length 
	       && obspos<observedVars.length) {
	    if (ruleContext.getVars()[conpos]==observedVars[obspos]) {
		if (ruleContext.getVals()[conpos]==observedVals[obspos]) {
		    ruleChanged=true;
		    conpos++; obspos++;
		}
		else {
		    // the context is inconsistent with the observation
		    return;
		}
	    }
	    else if (ruleContext.getVars()[conpos].getId()<observedVars[obspos].getId()) {
		// variable in the context isn't observed
		resultConVars[respos]=ruleContext.getVars()[conpos];
		resultConVals[respos++]=ruleContext.getVals()[conpos++];
	    }
	    else {
		// observed variable isn't in the context
		obspos++;
	    }
	}
	Context resContext;
	if (ruleChanged) {
	    resContext = new Context(resultConVars,resultConVals,respos);
	}
	else {
	    resContext=ruleContext;
	}
	resultingRules.add(new GenRule(resContext,
				       new FactorObserved(ruleFactor,observedVars,observedVals),
				       FactorSumOut.setDiff(rulesFor,observedVars)));
    }


    /**
     * is true if the rule is redundant; if it is redundant it
     * doesn't need to be added to the rule collection.
     **/
    public boolean redundant() {
	return rulesFor.length==0 && ruleFactor.allOnes();
    }


    /**
     * print a generalised rule.
     **/
    public void print() {
	ruleContext.print();
	System.out.println(":");
	ruleFactor.print("  | ");
	System.out.print("  (for");
	for (int i=0; i<rulesFor.length; i++) {
	    System.out.print(" "+rulesFor[i].getName());
	}
	System.out.println(")");
	
    }
    
    /**
     * prints a brief summary of the rule (without the probability table).
     **/
    public void printBrief() {
	ruleContext.print();
	System.out.print(": [");
	ruleFactor.printVariables();
	System.out.print("], for:");
	for (int i=0; i<rulesFor.length; i++) {
	    System.out.print(" "+rulesFor[i].getName());
	}
	System.out.println(". Size="+ruleFactor.size());
    }

}
