/* bn.c */
/* routines for manipulating Bayesian networks */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
 
#include "extern.h"
#include "parser.h"
#include "partialFun.h"
#include "operators.h"
#include "tableFun.h"
#include "bn.h"
#include "mis.h"



int noCSISize;
int CSISize;

VAR *varAllocate(numStates)
	int numStates;
{
	VAR *var;
	int *newNumStatesArray, i;

	var = (VAR *) memAllocate(sizeof(VAR));
	var->varID = numVars++;
	var->name = NULL;        
	var->prob = NULL;

	/* put the number of possible states of var onto
	   a global array */
	newNumStatesArray = intArrayAllocate(numVars );
	for (i=0; i<numVars -1; i++)
		newNumStatesArray[i]=numStatesArray[i];
	newNumStatesArray[numVars-1] = numStates;

	intArrayFree(numStatesArray, numVars -1);
	numStatesArray = newNumStatesArray;

	return var;
}

void varFree(var)
	VAR *var;
{
	  if (var->prob != NULL)
		tFuncFree(var->prob);
	  memDeallocate(var, sizeof(VAR));
}

BN *bnAllocate()
{
	BN *bn;

	bn = (BN *) memAllocate(sizeof(BN));
	bn->first = bn->last = NULL;

	return bn;
}


void  printVar(var)
	VAR *var;
{

	printf("var:  %d %s \n", var->varID, var->name);

	printTFunc(var->prob);
}

void printVarInfo( var_type *varInfo, FILE *fp)
{
	int i;
	char *propertyInfo;
	
	fprintf(fp, "variable %s {\n", varInfo->var_name);

	for (i=0; i<varInfo->prop_len; i++)
	{
	  fprintf(fp, "  property ");
	  fprintf(fp, "%s = ", varInfo->property[i]->field);
	  fprintf(fp, "%s;\n", varInfo->property[i]->value);

	}

	fprintf(fp, "  type discrete[%d] {", varInfo->type_size);
	for (i=0; i<varInfo->type_size; i++) {
	     fprintf(fp, "    %s", varInfo->value_list[i]);
	     if (i < varInfo->type_size -1 )
		fprintf(fp, ",");
	     fprintf(fp, "\n");
	}
	fprintf(fp, " };\n");
	fprintf(fp, "}\n\n");
}

/* next combination of values for parents of a variable */
int nextParentValue( parentValues, parents, n)
	int *parentValues;
	var_type **parents;
	int n;
{
	int i, j;
	int nextValueExist =FALSE;

	for (i=n-1; i>=0; i--) 
	{
	   if (parentValues[i] < parents[i]->type_size-1)
	   {
		nextValueExist = TRUE; 
		break;
	   }
	}

	if (nextValueExist == TRUE)
	{
	     parentValues[i] += 1;
	     for (j=i+1; j<n; j++)
	    	parentValues[j] =0;
	}

	return nextValueExist;
}

/* print conditional pribability table of a variable in BIF format */	
void printCPT(var, bn, rawNet, fp)
	VAR *var;
	BN *bn;
	BNet *rawNet;
	FILE *fp;
{
	int i, n, j, k, numStates;
	var_type **parents;
	int *parentValues;
	char *name;
	int nextValueExist;
	
 	fprintf(fp, "probability ( %s", var->name);

	/* number of parents */
	n = var->prob->numVars-1;

	/* number of states */
	numStates = numStatesArray[var->varID];

	/* if no parents */	
	if ( n==0) {
		fprintf(fp, " ) {\n");
		fprintf(fp, "   table ");

		for (i=0; i<var->prob->size -1 ; i++)
		    	fprintf(fp, "%f, ", var->prob->table[i]);
	   	fprintf(fp, "%f;\n", var->prob->table[i]);
	 
		fprintf(fp, "}\n");	

		return;
	}	

	/* var has at least one parent */
	fprintf(fp, " | ");

	/* find out information about parents from rawNet */
	parents = (var_type **) memAllocate( sizeof(var_type *) * n);
	for (i=0; i<n; i++) 
	{
		name = lookUpName( var->prob->varIDs[i], bn);
		parents[i] =  LookupVar(rawNet, name); 
		if (i<n-1) 
			fprintf(fp,"%s, ", name);
		else
			fprintf(fp, "%s ) { \n", name);
	}	

	parentValues = (int *) memAllocate( sizeof(int) * n);
	for (i=0; i<n; i++)
		parentValues[i] = 0; 

	/* now, we are ready to print the conditional probability table */
	j=0;
	do {
	   /* print names of parents values */
	   fprintf(fp, "   (");
	   for (i=0; i<n; i++) 
	   { 
		 if (i<n-1)
                     fprintf(fp, "%s, ", parents[i]->value_list[parentValues[i]]);
                else
                     fprintf(fp, "%s ) ", parents[i]->value_list[parentValues[i]]);
	   }

	   /* probability conditional probability of var 
	      given the parents values */
	   for (k=0; k<numStates; k++) 
	   {
		if (k<numStates-1)
		   fprintf(fp, "%f, ", var->prob->table[j++]);
		else
		   fprintf(fp, "%f; \n", var->prob->table[j++]);

	   }

	   nextValueExist = nextParentValue( parentValues, parents, n);
	} while (nextValueExist == TRUE);

	fprintf(fp, "}\n");

	free(parents);
	free(parentValues);
}


void printBN(bn, rawNet, fileName)
	BN *bn;
	BNet *rawNet;       /* contains raw information such
			       as state names, properties, etc */
	char *fileName;
{
	VAR *var;
	var_type *varInfo;
	FILE *fp;

	fp = fopen( fileName, "w" );
	if ( fp == NULL) {
	    printf(" Cannot open file %s \n", fileName);
	    exit(1);
	}
	if (bn == NULL) 
		printf("printBN: NULL input\n");

	/* total amounts of numbers with and without csi */
	fprintf(fp, "// %d %d %f \n", noCSISize, CSISize, 
			(double)CSISize / (double) noCSISize);

	fprintf(fp, "network Unknown { \n");
	fprintf(fp, "}\n \n");

	/* print variable information */
	var = bn->last;
        while (var != NULL) {
                varInfo = LookupVar(rawNet, var->name); 
		printVarInfo (varInfo, fp );

                var =var->prev;
        }

	/* conditinal probability */
        var = bn->last;
        while (var != NULL) {
 		printCPT(var, bn, rawNet, fp);

                var =var->prev;
        }

	fclose( fp );
}

void bnFree(bn)
	BN *bn;
{
	VAR *var, *var1;

	if (bn == NULL) return;

	var = bn->first;
	while (var != NULL) {
		var1 = var->next;
		varFree(var);
		var = var1;
	}

	memDeallocate(bn, sizeof(BN));
}

REAL *realArrayAllocate(len)
        int len;
{
        int i;
        REAL *array;

        array = (REAL *) memAllocate(len * sizeof(REAL));
        for (i=0; i<len; i++)
                array[i]=0.0;
        return array;
}
void realArrayFree(ar, len)
        REAL *ar;
        int len;
{
        memDeallocate(ar, len * sizeof(REAL));
}


int lookUpID(varName, bn)
	char *varName;
	BN *bn;
{
	VAR *var;

	var = bn->first;
	while (var != NULL) {
		if (strcmp(var->name, varName) == 0)
			return var->varID;
		var = var->next;
	}

	printf("lookUpID: varname %s not found \n", varName);
	exit(1);
}


VAR *lookUpVar(varName, bn)
        char *varName;
        BN *bn;
{
        VAR *var;

        var = bn->first;
        while (var != NULL) {
                if (strcmp(var->name, varName) == 0)
                        return var;
                var = var->next;
        }

        printf("lookUpVar: varname %s not found \n", varName);
        exit(1);
}                          

char *lookUpName(int ID, BN *bn)
{
        VAR *var;

        var = bn->first;
        while (var != NULL) {
                if (var->varID == ID)
                        return var->name;
                var = var->next;
        }

        printf("lookUpName: varID %d not found \n", ID);
        exit(1);
}                

/* from the string value of a variable, find its integer value */
int lookUpVal(BNet *rawNet,  char *varName, char *valName)
{
	var_type *rawVar;
	int i;

	rawVar = LookupVar( rawNet, varName );
	if (rawVar == NULL) 
	{
		printf(" var %s not found \n", varName);
		exit(1);
	}

	for (i=0; i<rawVar->type_size; i++)
	{
		if ( strcmp( valName, rawVar->value_list[i]) == 0)
			return i;
	}

	return UNDEFINED;
}


/* create CPT and detect CSI */
TFUNC *CPTCreate(var, rawProb, bn)
	VAR *var;
	prob_type *rawProb;
	BN *bn;
{
	TFUNC *fun;
	int i, numParentStates=1;
	int numStatesThisVar, stateThisVar, newConf, varID;

	/* header information about this function */
	/* number of vars in fun = num of parents + 1 */
	fun = tFuncAllocate(rawProb->cond_len+1);

	fun->varIDs = intArrayAllocate(fun->numVars);   
	for (i=0; i<fun->numVars-1; i++) {
		varID = lookUpID(rawProb->cond_list[i]->var_name, bn);
		fun->varIDs[i] = varID;
	        numParentStates *= numStatesArray[varID];
	}
	fun->varIDs[fun->numVars-1] = var->varID;

	/* allocate space for probability table */
	numStatesThisVar = numStatesArray[var->varID];
	fun->size = numParentStates *numStatesThisVar;

	fun->table = realArrayAllocate(fun->size);
	/* For data files where raw probability is given as a table,
	   var->varID is the first variable in rawProb. But 
	   in fun, it should be the last.  Some conversion is
           necessary */

	if (tableFormat == TRUE ) { 
		for (i=0; i<fun->size; i++) {
			stateThisVar = i/numParentStates;
			newConf = (i - stateThisVar*numParentStates) * 
				    numStatesThisVar + stateThisVar;	
			fun->table[newConf] = rawProb->prob_table[i];
		}

	} 
	else {
		/*** for data file where CPT is given as a list of tables,
	     	no conversion is necessary */
		for (i=0; i<fun->size; i++)
			fun->table[i] = rawProb->prob_table[i];
	}

	return fun;
}


/* creat BN from a raw BNet */
BN *BNCreate(rawNet)
	BNet *rawNet;
{
	BN *bn;
	VAR *var;
	int i, upper, lower, m;
	
	bn = bnAllocate();

	/* create the variables, rawNet is network obtained by parser */
	for (i=0; i<rawNet->num_var; i++) {
		var = varAllocate(rawNet->variable[i]->type_size);
		var->name = rawNet->variable[i]->var_name;
		ENQ(var, bn)
	}

	/* As a by-product, we have counted the total number of variables numVars
	   Allocate an array for general use */	
	varArray = intArrayAllocate(numVars);

	/* create CPT for all variables. */
	for (i=0; i<rawNet->num_var; i++) {
		var = lookUpVar(rawNet->probability[i]->var->var_name, bn);
		var->prob = CPTCreate(var, rawNet->probability[i], bn); 
	}
	return bn;
}




void  tableCellToPath(int cell, PATH *p, TFUNC *tFun)
{
        int varID, i;

        if (tFun==NULL) {
                printf("tableCellToPath: NULL function\n");
                exit(1);
        }
        if (cell >= tFun->size) {
                printf("tableCellToPath: cell out of range\n");
                exit(1);
        }

	p->val = tFun->table[cell];

        for (i=tFun->numVars - 1; i >= 0; i--) {
                varID = tFun->varIDs[i];
                p->path[i] =   cell - (cell/numStatesArray[varID]) *
                                        numStatesArray[varID];
                cell = cell/numStatesArray[varID];
        }
}                    


/* convert a fun represented as a table into
   a tree */
PFUNC *tableFunToTreeFun( TFUNC *tFun )
{
	PFUNC *pFun;
	int i, cell;
	PATH *p;

	pFun =pFuncAllocate( tFun->numVars);
	for(i=0; i<pFun->numVars; i++)
        	pFun->varIDs[i] = tFun->varIDs[i];

	p = pathAllocate( pFun->numVars );
	for (cell=0; cell < tFun->size; cell++) {
	     tableCellToPath(cell, p, tFun);
	     addPathToFunc(p, pFun);
	} 
	return pFun;
}

/* convert a fun represented as a tree into
   a table. If fun is a partial function, value for the
   undefined points are set at 0.0 */
TFUNC *treeFunToTableFun( PFUNC *pFun )
{
        PATH *p, *p1;
	TFUNC *tFun;
	int i, cell;

        tFun =tFuncAllocate( pFun->numVars);
        for(i=0; i<tFun->numVars; i++)
                tFun->varIDs[i] = pFun->varIDs[i];

	tFun->size =  prodOfStatesOfVarsFun( pFun );
	tFun->table = (double *) memAllocate( tFun->size * sizeof(double));

        p1 = pathAllocate(pFun->numVars);
        for (cell=0; cell < tFun->size; cell++) {
	        tableCellToPath(cell, p1, tFun);

		/* get the path of pFun that corresponds to cell */
		p = getFirstCompPath( pFun, p1->path);
		if (p!= NULL) 
		{
			tFun->table[cell] = p->val;
			pathFree(p, pFun->numVars);
		}
		else
		       tFun->table[cell] = 0.0;
        }
	pathFree(p1, pFun->numVars);
        return tFun;
}
         

int pathToTableCell( PATH *p, TFUNC *fun )
{
        int i, cell = 0;

        for (i=0; i<fun->numVars; i++) {
            cell = cell * numStatesArray[fun->varIDs[i]] + p->path[i];
        }

        return cell;
}                      

int getFirstCell(PFUNC *pFun, PATH *p, TFUNC *tFun,
		PATH *p1, int *mapping)
{
	int i, j;
	
	for (i=0; i<tFun->numVars; i++)
		p1->path[i] = 0;

	for (j=0; j<pFun->numVars; j++)
		p1->path[ mapping[j] ] = p->path[j];

	return pathToTableCell( p1, tFun );
}


int getNextCell(PFUNC *pFun, PATH *p, TFUNC *tFun,
                PATH *p1, int *mapping)
{
        int i, j, k;

	/* find the first variable whose value needs
	   to be increased. it is the last variable that satisfy
	   the following two conditions:
		1. Its value in p1 is less than its maximum possible value
		2. It is not in pFun 
	  denote the location of the variable by k */
	for( k = tFun->numVars-1; k>=0; k-- ) {
	    if ( (p1->path[k] < numStatesArray[tFun->varIDs[k]] -1 )
	        && (positionInArray( tFun->varIDs[k], pFun->varIDs,
			pFun->numVars) == UNDEFINED )) 
		break;
	}

	if (k<0) return UNDEFINED;

	/* increase the value of the variable in p1 by 1 */
	p1->path[k] = p1->path[k] + 1;

	/* values for all varaibles after k and not in pFun should
	   be set to 0 */

        for (i=k+1; i<tFun->numVars; i++) {
	      if (positionInArray( tFun->varIDs[i], pFun->varIDs,
			pFun->numVars) == UNDEFINED ) 
		 p1->path[i] =0;
	}

        return pathToTableCell( p1, tFun );
}                           

void changeTableUsingDecomp( PFUNC_LIST *decomp, TFUNC *tFun )
{
     PFUNC *pFun;
     PATH *p, *p1;
     int *mapping;
     int i, j, cell;


     /* p1 is for inserting values to tFun */
     p1 = pathAllocate( tFun->numVars );

     pFun = decomp->first;
     while (pFun != NULL) {
	 /* mapping is used to store positions of variables of pFun in tFun */
	mapping = intArrayAllocate( pFun->numVars );
	for (i=0; i<pFun->numVars; i++)
		mapping[i] = positionInArray( pFun->varIDs[i],
				tFun->varIDs, tFun->numVars );


	/* get path from pFun one by one */
	p = getFirstPath( pFun) ;
	while (p != NULL) {

	   /* find the cells that this path cover and
	      place the value of p in each of those cells */
	   cell = getFirstCell(pFun, p, tFun, p1, mapping);
	   while (cell != UNDEFINED ) {
		tFun->table[cell] = p->val;

		cell = getNextCell(pFun, p, tFun, p1, mapping );
	   }
	     
	   p = getNextPath( pFun, p);
	}

        free( mapping );
        pFun = pFun->next;
     }
     pathFree(p1, tFun->numVars);
}



TFUNC_LIST *getTableCondProbs(BN *bn,ASSN_ARRAY *evidenceArray) 
{
        VAR *var;
        TFUNC_LIST *list;
	TFUNC *fun;

        list = tFuncListAllocate();
        var = bn->first;
        while (var != NULL) {
		fun = tFuncCopy( var->prob );
		instantiateObservedVarsTFUNC(fun, evidenceArray);
                ENQ(fun , list);
                var = var->next;
        }

        return list;
}


PFUNC_LIST *getTreeCondProbs(BN *bn, ASSN_ARRAY *evidenceArray) 
{
        VAR *var;
	PFUNC_LIST *list;
	PFUNC *fun;

	list = pFuncListAllocate();
        var = bn->first;
        while (var != NULL) {
		fun = tableFunToTreeFun( var->prob);
		instantiateObservedVarsPFUNC(fun, evidenceArray);
		ENQ( fun, list);
                var = var->next;
        }

	return list;
}

PFUNC_LIST_GROUP *getCPTDecomps(BN *bn, ASSN_ARRAY *evidenceArray)
{
        VAR *var;
        PFUNC_LIST_GROUP *g;
	PFUNC_LIST *list;

        g = pFuncListGroupAllocate();
        var = bn->first;
        while (var != NULL) {
                list = var->probDecomp;
		instantiateObservedVarsPFUNCList(list, evidenceArray);
		list->full = TRUE;
                ENQ(  list, g);
		var->probDecomp = NULL;
                var = var->next;
        }

        return g;
}
        
int furtherSplitTest( TFUNC *tFun, double errorBound, int mode )
{
        int n, N, i, j;
	double *mean;
	int tmp;

	/* when in GENERATE_CPT_WITH_CSI mode, we stop
	   further slipt randomly */
	if (mode == GENERATE_CPT_WITH_CSI)
	{
		/* stop further split with some probability */
		tmp = lrand48() % 100;
/*
		printf("(%d, %d ) ", tmp, (int) ( 100.0 * errorBound) );
*/
		if (tmp < (int) (100.0 * errorBound) )
			return FALSE;
		else
			return TRUE;
	}

	/* number of states of the last variable */
	n = numStatesArray[ tFun->varIDs[ tFun->numVars-1 ] ];


	/* number of combinations of states of all other variables */
	N = tFun->size/n;

	
	/* view the table to tFun as a sequence of N vectors, each
	   of length n. compute mean of the vectors */
        mean = (double *) malloc( n * sizeof(double ));
        for (j=0; j<n; j++)
                mean[j]=0.0;

	for (i=0; i<N; i++) {
	   for (j=0; j<n; j++) {
		    mean[j] += tFun->table[ i*n +j];
	   }
	}	   
        for (j=0; j<n; j++)
                mean[j]=mean[j]/ (double)N;

	/* need to split futher if there is one vector not 
	   similar to mean */
	for (i=0; i<N; i++) 
	    if ( similarVecs( tFun->table + i*n, mean, n, 
			errorBound) == FALSE)
		return TRUE;

	return FALSE;
}

         
/* return a function that contain only the last variable.
   the values are averages */
TFUNC *averagingTFunc( TFUNC *fun) 
{
	TFUNC *fun1;
	int i,j,  n, N;

	fun1 = tFuncAllocate(1);
	fun1->varIDs[0] = fun->varIDs[ fun->numVars-1 ];
	
	n = numStatesArray[ fun->varIDs[ fun->numVars-1 ] ];
	fun1->size = n;
	
	fun1->table = (double *) malloc( n * sizeof(double) );

	for (j=0; j<n; j++)
		fun1->table[j] = 0.0;

        N = fun->size/n;


        for (i=0; i<N; i++) {
           for (j=0; j<n; j++) {
                    fun1->table[j] += fun->table[ i*n +j];
           }
        }
        for (j=0; j<n; j++)
                    fun1->table[j] = fun1->table[j]/ (double)N;

	
	return fun1;

}

int similarVecs( double *v1, double *v2,  int len, double delta)
{
	int i;

	for (i=0; i<len; i++) { 
	  if ( fabs( v1[i]-v2[i] ) > delta) 
		return FALSE;
	}
	return TRUE;
}

/* view fun as a sequence of vectors. this computes the number
   of similar vector pairs */
  int numPairSimilarVecs( TFUNC *fun, double errorBound )
{
	int count = 0;
	int n, N, i, j;

	/* number of states of the last variable */
        n = numStatesArray[ fun->varIDs[ fun->numVars-1 ] ];

        /* number of combinations of states of all other variables */
        N = fun->size/n;

        for (i=0; i<N; i++) { 
           for (j=i+1; j<N; j++) { 
		if ( similarVecs( fun->table + i*n, 
				fun->table + j*n, n, errorBound) == TRUE ) 
			count++;
	   }
        }          

	return count;
}

TFUNC *splitTFunc( TFUNC *fun, int var, int state) 
{
	int pos, i, j, n, v, S1, S2, s;
	TFUNC *fun1;
	int N1=1;  	/* number of combinations of states of
			   variables before var */

	int N2=1;  	/* number of combinations of states of
			   variables after var */

	if ( var == fun->varIDs[ fun->numVars-1 ] ) {
	     printf(" splitTFUNC:  cannot split on the last variable \n");
	     exit(1);
	}

	fun1 = tFuncAllocate( fun->numVars - 1);

	/* variables for new fun */
	pos = positionInArray( var, fun->varIDs, fun->numVars);

	if ( pos == UNDEFINED ) {
	     printf(" splitTFUNC: var %d not in functions \n", var);
	     exit(1);
	}
	j=0;
	for (i=0; i< fun->numVars; i++) {
	    if (i != pos ) {
		fun1->varIDs[j] =   fun->varIDs[i];
		j++;
	    }
	}

	/* size for new fun */
	n = numStatesArray[ var ];
	fun1->size = fun->size/n;

	fun1->table = (double *) malloc( fun1->size * sizeof( double ) );
	
	/* compute N1 and N2 */
	  for (i=0; i< fun->numVars; i++) {
            v = fun->varIDs[i];
            if (i < pos ) 
                N1 *= numStatesArray[ v ];
            if (i > pos ) 
                N2 *= numStatesArray[ v ];
        }   

	for (i=0; i<fun->size; i++ ) {
	    /* i the hash value of a combination of states of all vars */
	    /* hash value of the combination of states of var and
		all variable before var */
	     S1 = i/N2;
	
	    /* value for var */
	     s = S1 % n;

	    if (s ==state ) {
		/* hash value for states of variables after var */
		S2 = i % N2 ;

		/* hash value for states of variables before var */
		S1 = i / (N2 * n );
	
		/* new table position */
		j = S1 * N2 + S2;

		fun1->table[j] = fun->table[i];	
	    } 
	}	

	return fun1;
}

int selectSplitVar( TFUNC *fun, double errorBound, int mode )
{
        int var, i,  n, bestVar;
        int m = 0, m1;
        TFUNC *fun1;

	/* In VE_CSI mode, we assume data file contains csi
	   decomposition. We need only to recover it */
	if (mode == VE_CSI)
	{
	      return fun->varIDs[0];
	}


        for (var=0; var<fun->numVars -1 ; var++) {
                n = numStatesArray[ fun->varIDs[var] ];
                m1=0;
                for (i=0; i<n; i++) {
                    /* split fun for var one i */
                    fun1 = splitTFunc( fun, fun->varIDs[var], i );

                    /* number of similar vector pairs in result */
                    m1 += numPairSimilarVecs( fun1, errorBound );

                    tFuncFree( fun1 );
                }

                if (m1 >= m ) {
                        m = m1;
                        bestVar = fun->varIDs[var];
                }
        }

        return  bestVar;
}


/* add var to the front of the variable list of  fun
   the variable can only take value state in the new func */
void addVarToFrontPFunc( int var, int state, PFUNC *fun ) 
{
	int *newVarIDs;
	int i, n;
	TREE *newRoot;

	    /* new array of variables */
	    newVarIDs = intArrayAllocate( fun->numVars + 1);

	    newVarIDs[0] = var;
	    for (i=0; i< fun->numVars; i++)
		newVarIDs[ i+1 ] = fun->varIDs[i];

	    free( fun->varIDs );
	    fun->varIDs = newVarIDs;

	    /* number of variables increase by one */
	    fun->numVars++;

	    /* modify tree. remember var can only take value
	       state and it is the first variable in new function */	
	    newRoot = nodeAllocate(); 
	    n = numStatesArray[ var ];
	    newRoot->subtree = treeArrayAllocate( n );

	    newRoot->subtree[ state ] = fun->tree;
	    fun->tree->parent = newRoot;
	
	    fun->tree = newRoot;
}
void addVarToFrontPFuncList( int var, int state, PFUNC_LIST *list )   
{
	PFUNC *fun;

	fun = list->first;
	while (fun != NULL) {
	     	addVarToFrontPFunc( var, state, fun ); 
		fun = fun->next;
	}
}

PFUNC_LIST *detectCSITFunc0( TFUNC *tFun , int *splitted, int mustSplit, 
			double errorBound, int mode)
{
	PFUNC_LIST *list, *list1;
	PFUNC *pFun;
	TFUNC *tFun1;
	int splitVar, n, i;
	int splitFlag1 = FALSE, splitFlag2;

	list = pFuncListAllocate();

	*splitted = FALSE;
	/* if there is only one variable in tFun,
	   simply convert it into a PFUNC and return */
	if (tFun->numVars == 1) {
	      pFun = tableFunToTreeFun( tFun );
	      ENQ( pFun, list);
	      return list;
	} 

        /* if there is no need to split tFun further */
	if (mustSplit == FALSE &&
	      furtherSplitTest( tFun, errorBound, mode ) == FALSE) {
	      /* get rid of all variables except of the last one
	         do this by  computing average. so, the last variable
		 is idenpendent of the variables eliminated in
		 the given context */
	      tFun1 = averagingTFunc( tFun );

	      pFun = tableFunToTreeFun( tFun1 );
	      tFuncFree( tFun1 );

	      ENQ( pFun, list);
	     
	      return list;
	} 

	/* the function will be splitted */
        *splitted = TRUE;

	/* recursion */
	splitVar = selectSplitVar( tFun, errorBound, mode );
	n = numStatesArray[splitVar];
	for (i=0; i<n; i++) {
		/* split tFun on splitVar at value i,
	           afterward, tFun contains one less variable */
		tFun1 = splitTFunc( tFun, splitVar, i);

	 	if (i==n-1 && splitFlag1 == FALSE) 
		{
			/* we will split tFun according to the value
			   of splitVar into n components 
                           the last component must be further splitted if none 
			   of the first n-1 components are splitted.  otherwise,
                           some variables might be eliminated in all contexts,
                           giving rise to conditional independence rather
                           than CSI */
                         list1 = detectCSITFunc0( tFun1, &splitFlag2, TRUE,
                                                errorBound,mode );

                }      
		else {
			list1 = detectCSITFunc0( tFun1, &splitFlag2, FALSE,
						errorBound,mode );
			/* splitFlag2 == TRUE means i-th component
			   has been further splitted */
			if (splitFlag2 == TRUE) splitFlag1 = TRUE;

		}

	        tFuncFree( tFun1 );

                /* add back splitVar */
                addVarToFrontPFuncList( splitVar, i, list1);

                list = appendPFuncList( list, list1);    
  	}	

	/* merge functions with the same variables */
	mergePFuncList( list );
	return list;
}

PFUNC_LIST *detectCSITFunc( TFUNC *tFun , double errorBound, int mode)
{
	int dummy;

	return	detectCSITFunc0( tFun, &dummy, TRUE, errorBound, mode );
 }
/* detect CSI and decompose conditional probabilities.
   store results to the decomp field of each variable */
void detectCSIBN( BN *bn, double errorBound, int mode) 
{
       	VAR *var;
	PFUNC *fun;
	int size=0, size1=0;

	
        var = bn->first;
        while (var != NULL) {
                var->probDecomp = detectCSITFunc( var->prob, errorBound, mode );
		printf("var%d,  %d %d \n", var->varID, var->prob->size, 
			listSize( var->probDecomp ));

		printPFuncList(var->probDecomp);

		size += var->prob->size;
	        size1 += listSize( var->probDecomp );

		/* approximations are made when detecting CSI.
		   to make sure all algorithms work on the
		   same networks, modify the probability
		   table according to the decomposition */
		changeTableUsingDecomp( var->probDecomp, var->prob );

                var = var->next;
        }
	printf(" ------ \n");
	noCSISize = size;
	CSISize = size1;
	printf(" %d %d %f \n\n", size, size1, 
		(double)size1/(double)size );

	exit(1);
}



