/**
 * File: BayesGraph.java
 * May 3, 2001
 */


package CIspace.bayes;

import ve.*;

import java.awt.Color;
import java.awt.Frame;
import java.awt.List;
import java.io.IOException;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.util.*;

import CIspace.graphToolKit.Graph;
import CIspace.graphToolKit.GraphConsts;
import CIspace.graphToolKit.Point;
import CIspace.graphToolKit.Edge;


/**
 * BayesGraph subclasses Graph, implementing functionality
 * specific to the bayes applet.
 *
 * @since JDK1.1
 */

public class BayesGraph extends Graph implements Cloneable
{

   //
   // Fields
   //

  protected Vector decisionNodes;
  protected Vector valueNodes;
  protected Vector regularNodes;
  
  protected Vector observedNodes;
  
  protected Vector monitoredNodes;
 
  private BayesGraph queryGraph;
   
  private DecisionNetwork dn;
  //private BeliefNetwork dn;
    
  private int queryMode;
  private int selectedIndex;
  
  protected Vector factors;
  protected Vector eliminatedFactors;
  
  protected Integer factorIndex;

  public final static int C_MOD_PROB = 3335;
  public final static int C_MAKE_OBSERVATION = 3336;
  public final static int C_QUERY_NODE = 3337;
  public final static int C_MOD_UTIL = 3338;
  public final static int C_TOGGLE_MONITOR = 3339;
  public final static int C_VIEW_PROB = 3340;
  public final static int C_VIEW_UTIL = 3341;
  public final static int C_VIEW_POLICY = 3342;
  
  public final static int BRIEF_QUERY = 8880;
  public final static int VERBOSE_QUERY = 8881;
  public final static int PROMPT_QUERY = 8882;
  
  public final static int C_SET_REGULAR = 11;
  public final static int C_SET_DECISION = 15;
  public final static int C_SET_VALUE = 16;
    
  //
  // Methods
  //
  
  // constructors and copy constructors

  /**
   * Initializes a new BayesGraph
   */

  public BayesGraph( BayesCanvas canvas ) {
    super( canvas );
    queryMode = BRIEF_QUERY;
    regularNodes = new Vector(5,2);
    decisionNodes = new Vector(5,2);
    valueNodes = new Vector(5,2);
    observedNodes = new Vector(5,2);
    factors = new Vector(5,2);
    eliminatedFactors = new Vector(5,2);
    monitoredNodes = new Vector(5,2);
  }
  
  public BayesGraph( BayesCanvas canvas, String textRep, Vector observed ) {
    super( canvas );
    queryMode = BRIEF_QUERY;
    factors = new Vector(5,2);
    
    eliminatedFactors = new Vector(5,2);
    monitoredNodes = new Vector(5,2);
    
    parse( textRep );
    
    for (int i = 0; i<observed.size(); i++) {
      BayesNode obsNode = (BayesNode)observed.elementAt(i);
      BayesNode newNode = (BayesNode)nodeWithIndex( obsNode.index );
      newNode.makeObservation( obsNode.observation );
      observedNodes.addElement(newNode);
    }
    buildDecisionNetwork();
  }

  protected BayesGraph copy() {
    try {
      return (BayesGraph)clone();
    }
    catch (Exception e) {
      System.out.println(e.toString());
      return this;
    }
  }
  
  // accessors and modifiers

    /** Sets the label at the bottom of the canvas. */

   public void setPromptLabel( String s )
   {
      ((BayesCanvas)canvas).setPromptLabel( s );
   }

    /** Returns the label at the bottom of the canvas. */

   public String getPromptLabel()
   {
      return ((BayesCanvas)canvas).getPromptLabel();
   }

    /** Sets the query mode to be brief, verbose, or prompt. */

    public void setQueryMode( int QueryType ) {
	queryMode = QueryType;
    }

  /** Returns the query mode */

  public int getQueryMode() {
    return queryMode;
  }

    /** Returns an array of Variables that have had observations made. */

    public Variable[] getObservedVars() {
	Variable[] varArray = new Variable[observedNodes.size()];
	for (int i = 0; i<observedNodes.size(); i++) {
	    varArray[i] = ((BayesNode)observedNodes.elementAt(i)).getVariable();
	}
	return varArray;
    }

    /** Returns an array of ints that correspond to the observed Variables. Each
     * value is an index into the domain of the corresponding (same indexed) Variable. */

    public int[] getObservedVals() {
	int[] valArray = new int[observedNodes.size()];
	for (int i = 0; i<observedNodes.size(); i++) {
	    valArray[i] = ((BayesNode)observedNodes.elementAt(i)).observation;
	}
	return valArray;
    }

    /** Returns the Decision or Belief Network. */

    public DecisionNetwork getDN() {
	return dn;
    }

   /**
    * returns the vector of regular nodes in the graph.
    */
   public Vector getRegularNodeIndex() 
   {
      return regularNodes;
   }
   
   /**
    * returns the vector of decision nodes in the graph.
    */
   public Vector getDecisionNodeIndex() 
   {
      return decisionNodes;
   }

  /** returns the vector of value nodes in the graph. */

  public Vector getValueNodeIndex() {
    return valueNodes;
  }

  // the following methods relate to observation

  /** This adds an observation to the graph or modifies an existing observation. */
  
  public void addObservation( BayesNode observedNode ) {
    
    if (!observedNodes.contains(observedNode)) {
      // insert it in the proper place, according to the id of its variable
      int j = 0;
      while (j < observedNodes.size()) {
        if (observedNode.getVariable().getId() < 
            ((BayesNode)observedNodes.elementAt(j)).getVariable().getId()) {
          break;
        }
        j++;
      }
      observedNodes.insertElementAt(observedNode,j);
      stopMonitored( observedNode );
    }
    observedNode.setLabelObserved();
    updateNodeSize( observedNode );
    
    updateMonitored();
  }
  
    /** This clears an observation from the graph. */

    public void clearObservation( BayesNode observedNode ) {
	if (!observedNodes.contains(observedNode)) return;
	observedNodes.removeElement(observedNode);
	observedNode.setLabelClear();
	updateNodeSize( observedNode );
	updateMonitored();
    }

  // The following methods relate to node monitoring

    /** This method adds a node to the list of nodes being monitored. */

    public void addMonitored( BayesNode node ) {
      
      if (node.observation != -1) {
        // do nothing
      }
      else {
	monitoredNodes.addElement(node);
        queryGraph = new BayesGraph( (BayesCanvas)canvas, generateTextRep(), observedNodes );
        queryGraph.copyPolicies( decisionNodes );        
        if (needsPolicy(node)) {
          node.setLabelUndefined();
          updateNodeSize(node);
          return;
        }
        else if (node.nodeType == BayesNode.REGULAR_NODE) {
          queryGraph.removeNonRegular();
        }
        BayesNode otherNode = (BayesNode)queryGraph.nodeWithIndex( node.index );
        queryGraph.removeIrrelevantVars( otherNode );

        queryGraph.buildDecisionNetwork();

	Vector results = queryGraph.briefQuery( (BayesNode)queryGraph.nodeWithIndex(node.index) );
        double[] nodeProbs = new double[results.size()];
        for (int j = 0; j<results.size(); j++) {
          nodeProbs[j] = ((Double)results.elementAt(j)).doubleValue();
        }
        node.monitorProbs = nodeProbs;
        node.setLabelProbs();
	updateNodeSize(node);
      }
    }

  /** This method removes a node from the list of nodes being monitored. */
  
  public void stopMonitored( BayesNode node ) {
    if (monitoredNodes.contains(node)) {
      monitoredNodes.removeElement(node);
      node.setLabelClear();
      updateNodeSize(node);
    }
  }
  
  /** This method updates the labels for the nodes that are being monitored. */
  
  public void updateMonitored() {
    String textRep = generateTextRep();
    
    for (int i = 0; i<decisionNodes.size(); i++) {
      if (monitoredNodes.contains( (BayesNode)decisionNodes.elementAt(i) ) ) {
        stopMonitored( (BayesNode)decisionNodes.elementAt(i) );
      }
    }   
    boolean undef = false;
    for (int i = 0; i<monitoredNodes.size(); i++) {
      queryGraph = new BayesGraph( (BayesCanvas)canvas, textRep, observedNodes );
      queryGraph.copyPolicies( decisionNodes );
      BayesNode tmpNode = (BayesNode)monitoredNodes.elementAt(i);
      System.out.println("updating " + tmpNode.getName() );
      if (tmpNode.nodeType == BayesNode.REGULAR_NODE) {
        undef = false;
        for (int j = 0; j<decisionNodes.size(); j++) {
          BayesNode decisionNode = (BayesNode)decisionNodes.elementAt(j);
          if (decisionNode.policy == null && decisionNode.observation == -1 && 
              descendentOf( tmpNode,decisionNode )) {
            tmpNode.setLabelUndefined();
            updateNodeSize(tmpNode);
            undef = true;
            break;
          }
        }
        if (!undef)
          queryGraph.removeNonRegular();
      }
      //else {
        BayesNode otherNode = (BayesNode)queryGraph.nodeWithIndex( tmpNode.index );
        queryGraph.removeIrrelevantVars( otherNode );
        //}

      if (!undef) {
        queryGraph.buildDecisionNetwork();
        Vector results = queryGraph.briefQuery( (BayesNode)queryGraph.nodeWithIndex(tmpNode.index) );
        double[] nodeProbs = new double[results.size()];
        for (int j = 0; j<results.size(); j++) {
          nodeProbs[j] = ((Double)results.elementAt(j)).doubleValue();
        }
        
        tmpNode.monitorProbs = nodeProbs;
        tmpNode.setLabelProbs();
        updateNodeSize(tmpNode);
      }
    }
  }
  
  
  // end of monitoring methods
  
  /** This method is called when switching to Solve mode, and initialises a new Decision
   *  or Belief Network with all the information required to perform queries. */
  
  public void buildDecisionNetwork() {
    Vector variables = new Vector(regularNodes.size()*2);
    
    for (int i = 0; i<regularNodes.size(); i++) {
      ((BayesNode)regularNodes.elementAt(i)).initVariables();
      variables.addElement((BayesNode)regularNodes.elementAt(i));
    }
    for (int i = 0; i<decisionNodes.size();i++) {
      ((BayesNode)decisionNodes.elementAt(i)).initVariables();
      variables.addElement((BayesNode)decisionNodes.elementAt(i));
    }
    for (int i = 0; i<valueNodes.size();i++) {
      ((BayesNode)valueNodes.elementAt(i)).initVariables();
      variables.addElement((BayesNode)valueNodes.elementAt(i));
    }
    dn = new DecisionNetwork(variables,variables.size());
    
    //dn = new BeliefNetwork(variables, variables.size());
    
    factors = new Vector(dn.getNumProbFactors());
    for (int i = 0; i<dn.getNumProbFactors(); i++) {
      Factor f = (Factor)dn.getProbFactors()[i];
      factors.addElement(f);
      f.facNum = i;
    }
    factorIndex = new Integer(factors.size());
    eliminatedFactors = new Vector();
  }
  
  // The following methods relate to deletion of nodes
  
  /** Removes a node from the graph. */
  
  public void removeNode( BayesNode node ) {
    clearObservation( node );
    stopMonitored( node );
    if (node.getNodeType() == BayesNode.REGULAR_NODE) {
      regularNodes.removeElement( node );
    }
    else if (node.getNodeType() == BayesNode.DECISION_NODE) {
      decisionNodes.removeElement( node );
    }
    else if (node.getNodeType() == BayesNode.VALUE_NODE) {
      valueNodes.removeElement( node );
    }
    node.removeNode();
  }
  
  /** overriding deleteSelected() in CIspace.graphToolkit.Graph so that
   * the nodes can have special treatment when deleted. */
  
  public void deleteSelected() {
    for (int i = 0; i < selectedEdges.size(); i++) {
      Edge e = (Edge)selectedEdges.elementAt(i);
      e.removeFromNodes();
      edges.removeElement(e);
      BayesNode n1 = (BayesNode)e.start;
      BayesNode n2 = (BayesNode)e.end;
      int ind = n2.parents.indexOf(n1);
      n2.removeParent( ind );
      n2.context = n2.buildContext();
      n1.children.removeElement(n2);
    }
    selectedEdges.removeAllElements();
    
    for (int i = 0; i<selectedNodes.size(); i++) {
      BayesNode n = (BayesNode)selectedNodes.elementAt(i);
      Vector[] connected = n.getAllEdges();
      while( connected[0].size() > 0 ) {
        Edge e = (Edge)connected[0].firstElement();
        e.removeFromNodes();
        edges.removeElement(e);
      }
      while( connected[1].size() > 0 ) {
        Edge e = (Edge)connected[1].firstElement();
        e.removeFromNodes();
        edges.removeElement(e);
      }
      nodes.removeElement(n);
      
      // resets node indices
      resetIndices();
      removeNode(n);
    }
    selectedNodes.removeAllElements();
  }
  
  // end of deletion

  private void resetIndices() {
    for (int j = 0; j<numNodes(); j++) {
        BayesNode tmpNode = (BayesNode)nodeAt(j);
        tmpNode.index = j;
      }
  }
  
  private void setNodePos() {
    float x = -200;
    float y = -200;
    for (int i = 0; i<numNodes(); i++) {
      BayesNode tmpNode = (BayesNode)nodeAt(i);
      Point p = new Point( x, y );
      tmpNode.pos = p;
      x += 400;
      if ( ((i+1) % (int)Math.sqrt(numNodes())) == 0 ) {
        y += 400;
        x = -200;
      }
    }
    ((BayesCanvas)canvas).autoscale();
  }
  
  /**
   * set the nodes' properties after a graph has been loaded
   */
  
  public void setNodes() {
    BayesNode tmpNode;
    regularNodes = new Vector(5, 2);
    decisionNodes = new Vector(5, 2);
    valueNodes = new Vector(5, 2);
    
    for( int i = 0; i < numNodes(); i++ ) {
      tmpNode = (BayesNode)nodeAt(i);
      
      if ((tmpNode.getNodeType() == BayesNode.REGULAR_NODE) &&
          (!regularNodes.contains(tmpNode))) {
        regularNodes.addElement(tmpNode);
      }
      if ((tmpNode.getNodeType() == BayesNode.DECISION_NODE) &&
          (!decisionNodes.contains(tmpNode))) {
        decisionNodes.addElement(tmpNode);
      }
      if ((tmpNode.getNodeType() == BayesNode.VALUE_NODE) &&
          (!valueNodes.contains(tmpNode))) {
        valueNodes.addElement(tmpNode);
      }
      for ( int j = 0; j < tmpNode.parentIndexes.size(); j++ ) {
        
        int index = Integer.parseInt((String)tmpNode.parentIndexes.elementAt(j));
        if (validNodeIndex( index )) {
          tmpNode.setParent( nodeWithIndex( index ) );
          BayesEdge edge = new BayesEdge( this, (BayesNode)nodeWithIndex(index), tmpNode );
          addEdge(edge);
        }
      }
      if ( tmpNode.nodeType == BayesNode.DECISION_NODE ) {
        tmpNode.context = tmpNode.buildContext();
        tmpNode.probs = new Vector();
      }
      else {
        tmpNode.context = tmpNode.buildContext();
        tmpNode.initProbs( tmpNode.probsVector );
      }
      
    }
    buildDecisionNetwork();
  }

  /** A method to find the node with the specified index - this is more accurate than
   * nodeAt. */

  protected BayesNode nodeWithIndex( int index ) {
    for (int i = 0; i < numNodes(); i++) {
      BayesNode tmpNode = (BayesNode)nodeAt(i);
      if (tmpNode.index == index) {
        return tmpNode;
      }
    }
    return null;
  }

  protected void setAsValueNode( BayesNode node ) {
    node.setNodeType(BayesNode.VALUE_NODE);
    if (!valueNodes.contains(node)) {
      valueNodes.addElement(node);
      node.setNodeAppearance(BayesNode.VALUE_NODE);
    }
    if (decisionNodes.contains(node)) {
      decisionNodes.removeElement(node);
    }
    else if (regularNodes.contains(node)) {
      regularNodes.removeElement(node);
    }
    for (int i = 0; i<numEdges(); i++) {
      Edge e = edgeAt(i);
      if (e.start == node) {
        e.removeFromNodes();
        edges.removeElement(e);
      }
    }

  }
  
  protected void setAsDecisionNode( BayesNode node ) {
    node.setNodeType(BayesNode.DECISION_NODE);
    if (!decisionNodes.contains(node)) {
      decisionNodes.addElement(node);
      node.setNodeAppearance(BayesNode.DECISION_NODE);
    }
    if (regularNodes.contains(node)) {
      regularNodes.removeElement(node);
    }
    else if (valueNodes.contains(node)) {
      valueNodes.removeElement(node);
    }
  }

  protected void setAsRegularNode( BayesNode node ) {
    node.setNodeType(BayesNode.REGULAR_NODE);
    if (!regularNodes.contains(node)) {
      regularNodes.addElement(node);
    }
    node.setNodeAppearance(BayesNode.REGULAR_NODE);
    if (decisionNodes.contains(node)) {
      decisionNodes.removeElement(node);
    }
    else if (valueNodes.contains(node)) {
      valueNodes.removeElement(node);
    }
  }
  
  public void setNodeAs( int type ) {
    BayesNode tmpNode;
    for (int i = 0; i < selectedNodes.size(); i++) {
      tmpNode = (BayesNode)selectedNodes.elementAt(i);
      if ( type == C_SET_REGULAR && tmpNode.getNodeType() != BayesNode.REGULAR_NODE ) {
        setAsRegularNode( tmpNode );
      }
      else if ( type == C_SET_DECISION && tmpNode.getNodeType() != BayesNode.DECISION_NODE ) {
        setAsDecisionNode( tmpNode );
      }
      else if ( type == C_SET_VALUE && tmpNode.getNodeType() != BayesNode.VALUE_NODE ) {
        setAsValueNode( tmpNode );
      }
    }
    updateNodeSize();
    repaint();
  }

  // The following methods relate to parsing and the text representation of a graph.

  /**
   * Generates the text representation of a graph in the same format that it takes in.
   */
  
  public String generateTextRep() {

    String outString = new String("");
    
    Calendar now = Calendar.getInstance();
    outString = outString + "% Auto-generated on " + now.getTime() + "\n";
    
    outString = outString + "% Nodes\n";
    outString = outString + "% type, index, name, x position, y position, domain, probability table;\n";
    BayesNode tmpNode;
    for ( int i = 0; i < regularNodes.size(); i++ ) {
      tmpNode = (BayesNode)regularNodes.elementAt(i);
      outString = outString + "n" + tmpNode.index + ":" + tmpNode.getName() + "/" + tmpNode.pos.x + "/"
        + tmpNode.pos.y + "/" + tmpNode.domain.toString() + "/" + tmpNode.parentsIndexes().toString()
        + "/" + tmpNode.probsFlatten().toString() + ";\n";
    }
    for ( int i = 0; i < decisionNodes.size(); i++ ) {
      tmpNode = (BayesNode)decisionNodes.elementAt(i);
      outString = outString + "d" + tmpNode.index + ":" + tmpNode.getName() + "/" + tmpNode.pos.x + "/"
        + tmpNode.pos.y + "/" + tmpNode.domain.toString() + "/" + tmpNode.parentsIndexes().toString()
        + "/" + "[]" + ";\n";
    }
    for ( int i = 0; i < valueNodes.size(); i++ ) {
      tmpNode = (BayesNode)valueNodes.elementAt(i);
      outString = outString + "v" + tmpNode.index + ":" + tmpNode.getName() + "/" + tmpNode.pos.x + "/"
        + tmpNode.pos.y + "/" + tmpNode.domain.toString() + "/" + tmpNode.parentsIndexes().toString()
        + "/" + tmpNode.probsFlatten().toString() + ";\n";
    }
    outString = outString + "endBayesGraph;";
    return outString;
  }
  
  /** Generates the text representation of the graph based on the .bif standard. */
  
  public String generateBifTextRep() {
    String outString = new String("");
    BayesNode tmpNode;
    outString = outString + "network Untitled {\n}\n";
    for (int i = 0; i<numNodes(); i++) {
      tmpNode = (BayesNode)nodeAt(i);
      outString = outString + "variable " + tmpNode.getName() + " {\n  type discrete";
      Integer domSize = new Integer(tmpNode.domain.size());
      outString = outString + " [ " + domSize.toString() + " ]";
      outString = outString + " { " + (String)tmpNode.domain.elementAt(0);
      for (int j = 1; j<tmpNode.domain.size(); j++) {
        outString = outString + ", " + (String)tmpNode.domain.elementAt(j);
      }
      outString = outString + " };\n}\n";
    }
    for (int i = 0; i<numNodes(); i++) {
      tmpNode = (BayesNode)nodeAt(i);
      outString = outString + "probability ( " + tmpNode.getName();
      if (tmpNode.parents.size() == 0) {
        outString = outString + " ) {\n  table ";
        Vector probs = tmpNode.probsFlatten();
        outString = outString + ((Double)probs.elementAt(0)).toString();
        for (int j = 1; j<probs.size(); j++) {
          outString = outString + ", " + ((Double)probs.elementAt(j)).toString();
        }
        outString = outString + ";\n}\n";
      }
      else {
        outString = outString + " | " + ((BayesNode)tmpNode.parents.elementAt(0)).getName();
        for (int j = 1; j<tmpNode.parents.size(); j++) {
          outString = outString + ", " + ((BayesNode)tmpNode.parents.elementAt(j)).getName();
        }
        outString = outString + " ) {\n";
        for (int j = 0; j<tmpNode.context.size(); j++) {
          Vector curContext = (Vector)tmpNode.context.elementAt(j);
          Vector curProbs = (Vector)tmpNode.probs.elementAt(j);
          outString = outString + "  (" + (String)curContext.elementAt(0);
          for (int k = 1; k<curContext.size(); k++) {
            outString = outString + ", " + (String)curContext.elementAt(k);
          }
          
          outString = outString + ") " + ((Double)curProbs.elementAt(0)).toString();
          for (int k = 1; k<curProbs.size(); k++) {
            outString = outString + ", " + ((Double)curProbs.elementAt(k)).toString();
          }
          outString = outString + ";\n";
        }
        outString = outString + "}\n";
      }
      
    }
    return outString;
  }

    /** 
     * This builds a new graph based on a string generated from a .bif file. */

    public String parseBif( String file ) {

	String errorMessage = new String("");
	String inString = file;
	int p1 = -1;
	int p2 = file.indexOf("}\n");
	
	nodes = new Vector(10,10);
	edges = new Vector(10,10);
	decisionNodes = new Vector(5,2);
	regularNodes = new Vector(10,10);
	valueNodes = new Vector(5,2);
	observedNodes = new Vector(5,2);
        monitoredNodes = new Vector(5,2);
	scale = 1;
	int lineCount = 0;
	p1 = p2;
	p2 = file.indexOf( "}\n",p1+1);


       	try {
	    while (p2 != -1) {
		inString = file.substring(p1+1, p2).trim();
		errorMessage = parseBifLine( inString );
		
		if (errorMessage.length() > 0) {
		    throw (new Exception());
		}
		p1 = p2;
		p2 = file.indexOf("}\n",p1+1);
		if (p2 == -1) { 
		    p2 = file.indexOf("};\n", p1+1);
		    p1 = p1+1;
		}
		lineCount++;
	    }
	    setNodePos();
	    setNodes();
	    
	    return "OK";
	}
	catch (Exception e) {
	   String message = "";
	    if (errorMessage.length() > 0) {
		message = "Error at line " + (lineCount + 1) + " -- " + errorMessage;
	    }
	    else {
		message = "Error at line " + (lineCount + 1) + " -- " + e.toString();
	    }
	    return message;  
	}
    }

  /** Parses an individual "line" of a .bif file. A line is the definition of a variable or
   * a variable's conditional probability table. */
	
  public String parseBifLine( String inString ) throws IOException {
    String errorMessage = "";
    StringTokenizer tok = new StringTokenizer( inString );
    String doing = tok.nextToken();
    
    //parsing a variable
    if (doing.equalsIgnoreCase("variable")) {
      String nodeName = tok.nextToken();
      String blank = tok.nextToken();
      while (!blank.equals("[")) {
        blank = tok.nextToken();
      }
      String domSize = tok.nextToken();
      Vector domain = new Vector( Integer.parseInt( domSize.trim() ) );
      while (!blank.startsWith("{")) {
        blank = tok.nextToken();
      }
      String domElement = tok.nextToken();
      while (!domElement.endsWith(";")) {
        if (domElement.endsWith(",")) {
          domElement = domElement.substring(0,domElement.indexOf(","));
        }
        domain.addElement(domElement);
        domElement = tok.nextToken();
      }
      BayesNode node = new BayesNode( this );
      node.setName(nodeName);
      node.setNodeType( BayesNode.REGULAR_NODE );
      node.setNodeAppearance( BayesNode.REGULAR_NODE );
      node.domain = domain;
      node.parentIndexes = new Vector();
      node.probsVector = new Vector();
      addNode( node );
    }
    else if (doing.equalsIgnoreCase("probability")) {
      String blank = tok.nextToken();
      String nodeName = tok.nextToken();
      BayesNode curNode = nodeNamed( nodeName );
      if (curNode == null) {
        return "Invalid node name";
      }
      curNode.parentIndexes = new Vector();
      curNode.probsVector = new Vector();
      blank = tok.nextToken();
      if (blank.equals("|")) {
        nodeName = tok.nextToken().trim();
        while (!nodeName.equals(")")) {
          
          if (nodeName.endsWith(",")) {
            nodeName = nodeName.substring(0, nodeName.indexOf(","));
          }
          
          BayesNode parent = nodeNamed( nodeName );
          if (parent == null) {
            return "Invalid parent";
          }
          
          Integer ind = new Integer( parent.index );
          curNode.parentIndexes.addElement( ind.toString() );
          nodeName = tok.nextToken().trim();
        }
      }
      if (!tok.hasMoreTokens()) return errorMessage;
      blank = tok.nextToken();
      while (!blank.startsWith("}") && tok.hasMoreTokens()) {
        blank = tok.nextToken();
        if (!blank.equals("table")) {
          while (!blank.endsWith(")")) {
            blank = tok.nextToken();
          }
        }
        String prob = "";
        while (!prob.endsWith(";") && tok.hasMoreTokens()) {
          prob = tok.nextToken();
          String toAdd = prob;
          if (prob.endsWith(",")) {
            toAdd = prob.substring(0, prob.indexOf(","));
          }
          else if (prob.endsWith(";")) {
            toAdd = prob.substring(0, prob.indexOf(";"));
          }
          curNode.probsVector.addElement( toAdd );
          //prob = tok.nextToken();
        }
       }
     }
    return errorMessage;
  }
	 
  /**
   * This finds the node with the given name. */
  
  private BayesNode nodeNamed( String name ) {
    for (int i = 0; i<numNodes(); i++) {
      if (((BayesNode)nodeAt(i)).getName().equals(name)) {
        return (BayesNode)nodeAt(i);
      }
    }
    return null;
  }
  
  /** 
   * This builds a new graph based on a file in string form. It calls parseLine
   * to build each new node.
   */
  
  public String parse( String file ) {
    String errorMessage = new String("");
    String inString;
    int p1 = -1;
    int p2 = file.indexOf("\n");
    int lineCount = 0;
    
    nodes = new Vector(10,10);
    edges = new Vector(10,10);
    decisionNodes = new Vector(5,2);
    regularNodes = new Vector(10,10);
    valueNodes = new Vector(5,2);
    observedNodes = new Vector(5,2);
    monitoredNodes = new Vector(5,2);
    scale = 1;
    
    try { 
      while (p2 != -1) {
        inString = file.substring( p1 + 1, p2).trim();
        if ((inString).length() > 0) {
          if ((inString.charAt(0) != '%') && (inString.indexOf(";") == -1)) {
            errorMessage = new String("Missing semicolon");
            throw (new Exception());
          }
        }
        errorMessage = parseLine(inString.trim());
        
        if (errorMessage.length() > 0) {
          throw (new Exception());
        }
        p1 = p2;
        p2 = file.indexOf("\n", p1 + 1);
        lineCount++;
      }
      setNodes();
      if (decisionNodes.size() > 0 || valueNodes.size() > 0) {
        ((BayesCanvas)canvas).decisionStatus( true );
      }
      return "OK";
    }
    catch (Exception e) {
      String message = "";
      if (errorMessage.length() > 0) {
        message = "Error at line " + (lineCount + 1) + " -- " + e.toString();
      }
      else {
        message = "Error at line " + (lineCount + 1) + " -- " + e.toString();
      }
      return message;
    }
  }
  
  /** Parses each individual line in the text representation. */    
  
  public String parseLine( String inString) throws IOException {
    
    String errorMessage = "";
    if (inString.charAt(0) == '%' || inString.charAt(0) == ' ' || inString.charAt(0) == '\n' 
        || inString.indexOf("endBayesGraph") != -1 ) 
      return "";
    
    int p = inString.indexOf(":");
    
    Integer ind = new Integer(inString.substring(1,p).trim());
    int index = ind.intValue();
    StringTokenizer nodeTokenizer = new StringTokenizer(inString.substring(p+1,inString.length()),"/");
    String nodeName = nodeTokenizer.nextToken();
    Float xpos = new Float(nodeTokenizer.nextToken());
    Float ypos = new Float(nodeTokenizer.nextToken());
    Point pos = new Point(xpos.floatValue(), ypos.floatValue());
    String domainString = nodeTokenizer.nextToken();
    String parentsString = nodeTokenizer.nextToken();
    String probsString = nodeTokenizer.nextToken();
    int type = BayesNode.REGULAR_NODE;
    if (inString.charAt(0) == 'v') {
      type = BayesNode.VALUE_NODE;
    }
    else if (inString.charAt(0) == 'd') {
      type = BayesNode.DECISION_NODE;
    }
    else {
      type = BayesNode.REGULAR_NODE;
    }
    
    BayesNode node = new BayesNode( this );
    node.setName(nodeName);
    node.setNodeType(type);
    node.setNodeAppearance(type);
    node.pos = pos;
    node.parentIndexes = stringToVector(parentsString);
    node.domain = stringToVector(domainString);
    node.probsVector = stringToVector(probsString);
    node.updateSize();
    
    addNode( node );
    
    node.index = index;
    return errorMessage;
  }
  
  /** turns a string array into a vector of strings */
  
  private Vector stringToVector( String str ) {
    Vector tmpVec = new Vector();
    int lpar = str.indexOf('[');
    int rpar = str.indexOf(']');
    StringTokenizer tokenizer = new StringTokenizer(str.substring(lpar+1,rpar),",");
    while (tokenizer.hasMoreTokens()) {
      tmpVec.addElement(tokenizer.nextToken().trim());
    }
    return tmpVec;
  }

  // The following methods relate to querying

  /** Constructs a query for the node in question, depending on what has been selected as the
   * query mode. */
  
  public void queryNode( BayesNode node ) {
    
    if (needsPolicy( node )) {
        ((BayesCanvas)canvas).showMessage("Error","Node " + node.getName() + " requires a policy to be defined.");
       return;
      }

    factors = new Vector(dn.getNumProbFactors());
    for (int i = 0; i<dn.getNumProbFactors(); i++) {

      factors.addElement((Factor)dn.getProbFactors()[i]);
      ((Factor)factors.elementAt(i)).facNum = i;
    }

    eliminatedFactors = new Vector();
    if (queryMode == BRIEF_QUERY) {
      queryGraph = new BayesGraph( (BayesCanvas)canvas, generateTextRep(), observedNodes );
      BayesNode queryGraphNode = (BayesNode)queryGraph.nodeWithIndex( node.index );
      queryGraph.copyPolicies( decisionNodes );

      if (queryGraph.needsPolicy( queryGraphNode )) {
        ((BayesCanvas)canvas).showMessage("Error","Node " + node.getName() + " requires a policy to be defined.");
       return;
      }
      queryGraph.removeIrrelevantVars( queryGraphNode );
      queryGraph.buildDecisionNetwork();
      Vector results = queryGraph.briefQuery( queryGraphNode );
      String givenString = new String("");
      for (int i = 0; i<observedNodes.size(); i++) {
        Variable v = ((BayesNode)observedNodes.elementAt(i)).getVariable();
        givenString = givenString + "[" + v.getName() + "=" 
          + v.getDomain()[((BayesNode)observedNodes.elementAt(i)).observation] + "] ";
      }
      ((BayesCanvas)canvas).setAnswerFrame( new AnswerFrame( node, results, givenString ) );
    }
    else if (queryMode == VERBOSE_QUERY) {
      ((BayesCanvas)canvas).setQueryWindow( new QueryWindow( node, generateTextRep(), 
                                                             getObservedVars(), getObservedVals() ) );
    }
    else {
      ((BayesCanvas)canvas).setQTypeDialog(new QueryTypeDialog( (Frame)canvas.parent, 
                                                                (BayesCanvas)canvas, node, this));
    }
  }

  /** This optimises the decisions for the first value node */

  protected void optimizeDecisions() {
    if (valueNodes.size() == 0) return;
    for (int i = 0; i<observedNodes.size(); i++) {
      BayesNode tmpNode = (BayesNode)observedNodes.elementAt(i);
      if (tmpNode.nodeType == BayesNode.DECISION_NODE) {
        canvas.showMessage("Error","Decision nodes must be unobserved.");
        return;
      }
      for (int j = 0; j<tmpNode.children.size(); j++) {
        if (decisionNodes.contains( tmpNode.children.elementAt(j) )) {
          canvas.showMessage("Error","Parents of decision nodes must be unobserved.");
          return;
        }
      }
    }
    BayesNode node = (BayesNode)valueNodes.elementAt(0);
    queryGraph = new BayesGraph( (BayesCanvas)canvas, generateTextRep(), observedNodes );
    BayesNode queryGraphNode = (BayesNode)queryGraph.nodeWithIndex( node.index );
    queryGraph.removeIrrelevantVars( queryGraphNode );
    queryGraph.buildDecisionNetwork();
    Vector results = queryGraph.briefQuery( queryGraphNode );
    copyPolicies( queryGraph.decisionNodes );
    updateMonitored();
    String givenString = new String("");
    for (int i = 0; i<observedNodes.size(); i++) {
      Variable v = ((BayesNode)observedNodes.elementAt(i)).getVariable();
      givenString = givenString + "[" + v.getName() + "=" 
        + v.getDomain()[((BayesNode)observedNodes.elementAt(i)).observation] + "] ";
    }
    ((BayesCanvas)canvas).setAnswerFrame( new AnswerFrame( node, results, givenString ) );
  }

  /** This optimises a single decision node. */

  protected void optimizeDecision( BayesNode dNode ) {
    if (valueNodes.size() == 0) return;
    BayesNode node = (BayesNode)valueNodes.elementAt(0);
    queryGraph = new BayesGraph( (BayesCanvas)canvas, generateTextRep(), observedNodes );
    BayesNode queryGraphNode = (BayesNode)queryGraph.nodeWithIndex( node.index );
    queryGraph.removeIrrelevantVars( queryGraphNode );
    queryGraph.buildDecisionNetwork();
    Vector results = queryGraph.briefQuery( queryGraphNode );
    BayesNode otherNode = (BayesNode)queryGraph.nodeWithIndex( dNode.index );
    dNode.setPolicy( otherNode.getPolicy() );
  }

  /** This method assumes that the query is ready to go and sets up the query node
   * along with any relevant decision variables. */

  protected Variable[] getQueryVars( BayesNode node ) {

    Vector nonAncestors = getNonAncestors( node );
    Variable[] tempVar = new Variable[numNodes()];
    tempVar[0] = node.getVariable();
    int elements = 1;
    if (needsPolicy(node)) {
      for (int i = 0; i<decisionNodes.size(); i++) {
        BayesNode tmpNode = (BayesNode)decisionNodes.elementAt(i);
        if (tmpNode.observation == -1 && !nonAncestors.contains( tmpNode ) ) {  
          if (!contains( tempVar, tmpNode ) ) {
            tempVar[elements] = tmpNode.getVariable();
            elements++;
          }
          // but in case this node has parents, we need to take the policy into account
          // for value nodes
          for (int j = 0; j<tmpNode.parents.size(); j++) {
            BayesNode parentNode = (BayesNode)tmpNode.parents.elementAt(j);
            if (!contains( tempVar, parentNode ) && node.nodeType == BayesNode.VALUE_NODE
                && parentNode.observation == -1 && needsPolicy(node)) {
              tempVar[elements] = parentNode.getVariable();
              elements++;
            }
          }
        }
      }
    }
    Variable[] vars = new Variable[elements];
    for (int i = 0; i<elements; i++) {
      vars[i] = tempVar[i];
      System.out.println("querying " + vars[i].getName() );
    }
    DecisionNetwork.mySort( vars );
    return vars;
  }
  
  /** Performs a brief query for the node. This method is generally never called by the main Bayes Graph,
   *  but rather by the temporary query graph created, so nodes can be pruned. The very last portion
   * mirrors the methods defined in finishQuery(), when the node being queried is a value node. */
  
  public Vector briefQuery( BayesNode node ) {
    
    // we don't need to query observed nodes.
    if (node.observation != -1) {
      Vector results = new Vector(node.domain.size());
      for (int i = 0; i<node.domain.size(); i++) {
        if (node.observation == i) {
          results.addElement( new Double(1.0) );
        }
        else {
          results.addElement( new Double(0.0) );
        }
      }
      return results;
    }

    buildDecisionNetwork();
    observeFactors();

    Variable[] tempVar = getQueryVars( node );
    List l1 = new List();
    List l2 = new List();
    Variable[] theVars = new Variable[dn.getVariables().length - tempVar.length];
    int count = 0;
    int elements = 0;
    while (count < dn.getVariables().length) {
      if (!contains( tempVar, dn.getVariables()[count].getNode() ) ) {
        theVars[elements] = dn.getVariables()[count];
        elements++;
      }
      count++;
    }
    
    Vector factorsToEliminate = new Vector(factors.size());
    for (int i = 0; i<factors.size(); i++) {
      Factor tmpFactor = (Factor)factors.elementAt(i);
      if (tmpFactor.getVariables().length < 1) {
        factorsToEliminate.addElement(tmpFactor);
      }
      else {
        l1.add( tmpFactor.getName() );
      }
    }
    for (int i = 0; i<factorsToEliminate.size(); i++) {
      factors.removeElement( factorsToEliminate.elementAt(i) );
    }
  
    if (elements > 0) {
      FactorStore it = new FactorStoreIndexed( theVars, dn.getProbFactors(), dn.getNumProbFactors() );
      while (it.hasNext()) {
        Variable var = it.next();
        BayesNode tmpNode = var.getNode();
        eliminateVariable( var, l1, l2 );
        tmpNode.eliminated = true;
        if (factors.size() > 0) {
          it.addFactor( (Factor)factors.elementAt( factors.size() - 1 ) );
        }
      }
    }
    return getQueryResults( l1, l2, node );
  }

  /** This method sets up the answer properly when the factor contains multiple variables */

  private Vector fixAnswer( Factor queryAnswer, BayesNode node ) {
    Vector vars = new Vector(queryAnswer.getVariables().length - 1);
    Vector vecAnswer = new Vector();
    int contexts = 1;
    for (int i = 0; i<queryAnswer.getVariables().length; i++) {
      if (node != queryAnswer.getVariables()[i].getNode()) {
        vars.addElement( queryAnswer.getVariables()[i].getNode() );
        contexts *= queryAnswer.getVariables()[i].getNode().domain.size();
      }
    }
    
    int[] index = new int[vars.size()];
    while (contexts > 0) {
      Vector results = new Vector();
      Vector stringResults = new Vector();
      
      for (int i = 0; i < vars.size(); i++) {
        BayesNode tmpNode = (BayesNode)vars.elementAt(i);
        tmpNode.makeObservation( index[i] );
        if (!observedNodes.contains( tmpNode ) ) {
          // insert it in the proper place, according to the id of its variable
          int j = 0;
          while (j < observedNodes.size()) {
            if (tmpNode.getVariable().getId() < 
                ((BayesNode)observedNodes.elementAt(j)).getVariable().getId()) {
              break;
            }
            j++;
          }
          observedNodes.insertElementAt(tmpNode,j);
        }
        String str = new String( "[ " + tmpNode.getName() + " = " + (String)tmpNode.domain.elementAt( index[i] )
                                 + " ]");
        stringResults.addElement( str );
      }
      Vector tempAnswer = getQueryResults( node );
      for (int i = 0; i<tempAnswer.size(); i++) {
        results.addElement( (Double)tempAnswer.elementAt(i) );
      }
      vecAnswer.addElement( stringResults );
      vecAnswer.addElement( results );
      
      boolean allMax = true;
      for (int i = vars.size() - 1; i>=0 && allMax; i--) {
        if (index[i]< ((BayesNode)vars.elementAt(i)).domain.size()-1) {
          index[i]++;
          for (int j = i+1; j<vars.size(); j++) {
            index[j] = 0;
          }
          allMax = false;
        }
      }
      contexts--;
    }
    return vecAnswer;
  }


  /** Automatically eliminates variables. */

  public void autoEliminate( List factorList, List eliminatedFactorList, FactorStore iter, BayesNode queryNode ) {
    if (iter != null && factors.size() > 1) {
      while (iter.hasNext()) {
        Variable var = iter.next();
        BayesNode node = var.getNode();
        eliminateVariable( var, factorList, eliminatedFactorList );
        node.eliminated = true;
        node.setNodeAppearance( BayesNode.ELIMINATED_NODE );
        addLastFactor();
        if (factors.size() > 0) {
          iter.addFactor( (Factor)factors.elementAt( factors.size() - 1 ) );
        }
      }
    }
    finishQuery( factorList, eliminatedFactorList, queryNode );
    repaint();
  }

  /** Eliminates a Variable in verbose query mode. It assumes all variables are observed. */
  
  public void eliminateVariable( Variable var, List factorList, List eliminatedFactorList ) {
    Variable[] tempVar = new Variable[1];
    tempVar[0] = var;
    
    Vector derivedFrom = new Vector();
    Vector derivedFromVars = new Vector();
    derivedFromVars.addElement(var);
    Vector toEliminate = new Vector(factors.size());
    for (int i = 0; i<factors.size(); i++) {
      Factor tempFactor = (Factor)factors.elementAt(i);
      for (int j = 0; j<tempFactor.getVariables().length; j++) {
        if (tempFactor.getVariables()[j] == var) {
          toEliminate.addElement(tempFactor);
          derivedFrom.addElement(tempFactor);
        }
      }
    }
    if (toEliminate.size() == 0) return;
    
    Factor toSumOut = (Factor)toEliminate.elementAt(0);
    
    Integer ind = new Integer(toSumOut.facNum);
    int index = factors.indexOf( toSumOut );
    
    factors.removeElement(toSumOut);
    
    eliminatedFactorList.addItem("f"+ind.toString()+toSumOut.getName()  );
    factorList.delItem(index);
    eliminatedFactors.addElement(toSumOut);
    
    for (int i = 1; i<toEliminate.size(); i++) {
      Factor tempFactor = (Factor)toEliminate.elementAt(i);
      ind = new Integer(tempFactor.facNum);
      
      factorList.delItem(factors.indexOf(tempFactor));
      factors.removeElement(tempFactor);
      
      eliminatedFactorList.addItem( "f"+ind.toString()+tempFactor.getName() );
      eliminatedFactors.addElement(tempFactor);
      toSumOut = new FactorTimes(tempFactor,toSumOut);
    }
    Factor newFactor = new FactorSumOut(toSumOut, tempVar);
    newFactor.createdFrom = derivedFrom;
    newFactor.createdFromVars = derivedFromVars;
    newFactor.facNum = factorIndex.intValue();
    factors.addElement(newFactor);
    factorList.addItem("f"+factorIndex.toString()+newFactor.getName());
    factorIndex = new Integer(factorIndex.intValue() + 1);
  }

  protected void observeFactors() {
    for (int i = 0; i<factors.size(); i++) {
      Factor f = (Factor)factors.elementAt(i);
      if (needsObservation(f)) {
        System.out.println( "observing " + f.getName() );
        f = new FactorObserved( f, getObservedVars(), getObservedVals() );
        f.facNum = i;
        factors.setElementAt(f, i);
      }
    }
  }

  private boolean needsObservation( Factor f ) {
    if (f.howCreated() == Factor.BY_OBSERVED) return false; 
    Variable[] obsVars = getObservedVars();
    for (int i = 0; i < obsVars.length; i++) {
      for (int  j = 0; j < f.getVariables().length; j++) {
        if (obsVars[i].getId() == f.getVariables()[j].getId())
          return true;
      }
    }
    return false;
  }

  /** Copies the policies over from a vector of decision nodes, to the nodes
   * with the corresponding indices. */
  
  protected void copyPolicies( Vector nodes ) {
    for (int i = 0; i<nodes.size(); i++) {
      BayesNode tmpNode = (BayesNode)nodes.elementAt(i);
      BayesNode thisNode = nodeWithIndex( tmpNode.index );
      if (tmpNode.policy != null && tmpNode.observation == -1) {
        tmpNode.getPolicy().print();
        thisNode.setPolicy( tmpNode.getPolicy() );
      }
    }
  }

  private boolean needsPolicy( BayesNode node ) {
    for (int i = 0; i<decisionNodes.size(); i++) {
      BayesNode tmpNode = (BayesNode)decisionNodes.elementAt(i);
      if (tmpNode.observation == -1 && tmpNode.policy == null && descendentOf( node, tmpNode )) {
        return true;
      }
    }
    return false;
  }
  
  /** A helper method to determine whether a factor contains a variable with
   * an observed value. */
  
  protected static boolean contains( Variable[] list, BayesNode node ) {
    if (node == null) return false;
    for (int i = 0; i<list.length; i++) {
      if (list[i] != null && list[i].getNode().index == node.index) {
        return true;
      }
    }
    return false;
  }
  
  /** This method multiplies all remaining factors in a query and normalises them
   * to produce the query's result. */
  
  public void finishQuery( List factorList, List eliminatedFactorList, BayesNode node ) {
    if (factors.size() == 0) return;
    String givenString = new String("");
    for (int i = 0; i<observedNodes.size(); i++) {
      Variable v = ((BayesNode)observedNodes.elementAt(i)).getVariable();
      givenString = givenString + "[" + v.getName() + "=" 
        + v.getDomain()[((BayesNode)observedNodes.elementAt(i)).observation] + "] ";
    }
    Vector results = getQueryResults( factorList, eliminatedFactorList, node );
    ((BayesCanvas)canvas).setAnswerFrame( new AnswerFrame( node, results, givenString ) );    
  }

  /** This method is similar to the one below, but doesn't modify the lists, and assumes that
   * you are only ever querying a node that doesn't depend on decisions. */

  private Vector getQueryResults( BayesNode node ) {
    Variable[] tempVar = new Variable[1];
    tempVar[0] = node.getVariable();
    Query q = new Query( tempVar, dn, getObservedVars(), getObservedVals(), "" );
    Factor answer = q.getResult();
    if (node.nodeType == BayesNode.VALUE_NODE) {
      answer = q.getUnNormResult();
    }
    answer.facNum = factorIndex.intValue();
    factorIndex = new Integer( factorIndex.intValue() + 1 );
    factors.addElement(answer);
    EltsIterator iter = answer.iterator();
    Vector results = new Vector(node.domain.size());
    while (iter.hasNext()) {
      results.addElement( new Double( iter.next() ) );
    }
    iter.backTo(0);
    return results;
  }

  /** This is a generalised method to finish queries. */

  private Vector getQueryResults( List factorList, List eliminatedFactorList, BayesNode node ) {
    for (int i = factors.size() - 1; i>=0; i--) {
      if (((Factor)factors.elementAt(i)).getVariables().length < 1) {
        factors.removeElement(factors.elementAt(i));
      }
    }
    Vector derivedFrom = new Vector();
    Factor result = (Factor)factors.elementAt(0);
    derivedFrom.addElement(result);
    Integer ind = new Integer(result.facNum);
    eliminatedFactors.addElement(result);
    eliminatedFactorList.addItem(factorList.getItem(0));
    for (int i = 1; i<factors.size(); i++) {
      Factor tempFactor = (Factor)factors.elementAt(i);
      eliminatedFactors.addElement(tempFactor);
      eliminatedFactorList.addItem(factorList.getItem(i));
      derivedFrom.addElement(tempFactor);
      ind = new Integer(tempFactor.facNum);
      result = new FactorTimes(result,tempFactor);
    }
    // don't normalise results for value nodes
    Factor normresult;
    if (node.nodeType == BayesNode.REGULAR_NODE) {
      normresult = new FactorNormalise(result);
      normresult.createdFrom = derivedFrom;
    }
    else {
      normresult = result;
      if (derivedFrom.size() > 1) {
        normresult.createdFrom = derivedFrom;
      }
    }
    factors.removeAllElements();
    factorList.clear();
    Vector results = new Vector(node.domain.size());
    // if there's only one variable  
    if (normresult.getVariables().length == 1) {
      EltsIterator iter = normresult.iterator();
      for (int i = 0; i < node.domain.size(); i++) {
        Double d = new Double(iter.next());
        results.addElement(d);
      }
      iter.backTo(0);
      
      factors.addElement(normresult);


      if (eliminatedFactors.contains(normresult)) {
        eliminatedFactorList.remove( eliminatedFactors.indexOf(normresult) );
        eliminatedFactors.removeElement(normresult);
        factorIndex = new Integer( factorIndex.intValue() - 1 ); 
      }
      normresult.facNum = factorIndex.intValue();
      factorList.addItem("Answer:" + "f" + factorIndex.toString() + normresult.getName());
    }

    // otherwise it depended on some decisions
    // find the optimal policy, if one exists.

    else {
      Factor oldAnswer = normresult;
      BayesNode theNode = node;
      Vector remainingDecisionNodes = new Vector( decisionNodes.size() );
      for (int i = 0; i<decisionNodes.size(); i++) {
        remainingDecisionNodes.addElement( decisionNodes.elementAt(i) );
      }
      for (int i = 0; i<decisionNodes.size(); i++) {
        BayesNode nextNode = nextToEliminate( oldAnswer, theNode );
        if (nextNode != null) {
          oldAnswer.print();
          nextNode.eliminated = true;
          nextNode.setNodeAppearance( BayesNode.ELIMINATED_NODE );
          System.out.println( "next to eliminate: " + nextNode.getName() );
          remainingDecisionNodes.removeElement( nextNode );          
          Vector derivedFromVars = new Vector(1);
          derivedFromVars.addElement(nextNode.getVariable());
          Vector createdFrom = new Vector(1);
          createdFrom.addElement( oldAnswer );
          Factor theAnswer;

          theAnswer = new FactorMax( oldAnswer, nextNode.getVariable() );
          theAnswer.print();
          ((FactorMax)theAnswer).getPolicyFunction().print();
          nextNode.setPolicy( ((FactorMax)theAnswer).getPolicyFunction() );

          theAnswer.createdFrom = createdFrom;
          theAnswer.createdFromVars = derivedFromVars;
          theAnswer.facNum = factorIndex.intValue();
          factorIndex = new Integer( factorIndex.intValue() + 1 );
          eliminatedFactors.addElement( theAnswer );
          ind = new Integer( theAnswer.facNum );
          eliminatedFactorList.addItem("f" + ind.toString() + theAnswer.getName() );
          oldAnswer = sumOutParents( theAnswer, nextNode, eliminatedFactorList, remainingDecisionNodes );
        }
        if ( nextToEliminate( oldAnswer, theNode ) == null ) {
          theNode = nextNode;
        }

        // and continue eliminating with the new factor until all the variables except one are gone
      }
      // then output the answer
       EltsIterator iter = oldAnswer.iterator();
       for (int i = 0; i < node.domain.size(); i++) {
         Double d = new Double(iter.next());
         results.addElement(d);
       }
       iter.backTo(0);
       if (eliminatedFactors.contains( oldAnswer )) {
         eliminatedFactorList.remove( eliminatedFactors.indexOf(oldAnswer) );
         eliminatedFactors.removeElement( oldAnswer );
       }
       factors.addElement(oldAnswer);
       oldAnswer.facNum = factorIndex.intValue() - 1;
       ind = new Integer( oldAnswer.facNum );
       factorList.addItem("Answer:" + "f" + ind.toString() + oldAnswer.getName());
    }
    return results;
  }

  /** A method to find the last item in a policy, or any item if they're not ordered. 
   * It takes in a factor that's presumably the factor of all remaining decision nodes
   * and their parents and finds the last decision that has to be made, or its parents.
   * The node it takes in is the node that the factor corresponds to. */

  protected BayesNode nextToEliminate( Factor f, BayesNode node ) {
    Vector theParents = new Vector(node.parents.size());
    for (int i = 0; i<node.parents.size(); i++) {
      BayesNode tmpNode = (BayesNode)node.parents.elementAt(i);
      if (contains( f.getVariables(), tmpNode ) && tmpNode.nodeType == BayesNode.DECISION_NODE
          && !tmpNode.eliminated && tmpNode.observation == -1) {
        theParents.addElement( node.parents.elementAt(i) );
      }
    }
    if (theParents.size() == 1) {
      return (BayesNode)theParents.elementAt(0);
    }
    else if (theParents.size() == 0) {
      return null;
    }
    // otherwise find the node with the most ancestors among theParents
    else {
      int[] numParents = new int[theParents.size()];
      // count each node's ancestors
      for (int i = 0; i<theParents.size(); i++) {
        int count = 0;
        BayesNode firstNode = (BayesNode)theParents.elementAt(i);
        for (int j = 0; j<theParents.size(); j++) {
          BayesNode secondNode = (BayesNode)theParents.elementAt(j);
          if (descendentOf( firstNode, secondNode )) {
            count++;
          }
        }
        numParents[i] = count;
        System.out.println( count );
      }
      // now find the node with the most parents
      int max = numParents[0];
      for (int i = 0;i<numParents.length; i++) {
        if (max < numParents[i]) {
          max = i;
        }
      }
      BayesNode theNode = (BayesNode)theParents.elementAt(max);
      
      return theNode;
    }
  }

  /** This method should sum out all the random node parents of a decision node.
   * and returns the last answer */

  private Factor sumOutParents( Factor f, BayesNode dNode, List eliminatedFactorList, Vector remaining ) {
    Factor lastAnswer = f;
    for (int i = 0; i<dNode.parents.size(); i++) {
      
      BayesNode tmpNode = (BayesNode)dNode.parents.elementAt(i);
      
      if (tmpNode.nodeType != BayesNode.DECISION_NODE && !isParentOf( tmpNode, remaining )) {
        System.out.println( "summing out " + tmpNode.getName() );
        tmpNode.eliminated = true;
        tmpNode.setNodeAppearance( BayesNode.ELIMINATED_NODE );
        Vector createdFrom = new Vector(1);
        Vector createdFromVars = new Vector(1);
        Variable[] tempVar = new Variable[1];
        tempVar[0] = tmpNode.getVariable();
        Factor sumOut = new FactorSumOut( lastAnswer, tempVar );
        createdFrom.addElement( lastAnswer );
        createdFromVars.addElement( tmpNode.getVariable() );
        sumOut.createdFrom = createdFrom;
        sumOut.createdFromVars = createdFromVars;
        sumOut.facNum = factorIndex.intValue();
        eliminatedFactorList.addItem("f" + factorIndex.toString() + sumOut.getName() );        
        eliminatedFactors.addElement( sumOut );
        factorIndex = new Integer( factorIndex.intValue() + 1 );
        lastAnswer = sumOut;
      }
    }
    return lastAnswer;
  }  
 
  private static boolean isParentOf( BayesNode node, Vector nodeList ) {
    for (int i = 0; i<nodeList.size(); i++) {
      BayesNode tmpNode = (BayesNode)nodeList.elementAt(i);
      if (tmpNode.parents.contains( node )) {
        return true;
      }
    }
    return false;
  }

  /** A helper method to add the last factor created to the graph, joining its variables
   * together. */
  
  protected void addLastFactor() {
    if (factors.size() == 0) return;
    Factor tempFactor = (Factor)factors.elementAt(factors.size() - 1);
    joinFactor( tempFactor );
  }
  
    /** A method to draw an edge between all of a factor's variables. */

  protected void joinFactor( Factor f ) {
    Variable[] factorVars = f.getVariables();
    for (int i = 0; i<factorVars.length; i++) {
      for (int j = 0; j<factorVars.length; j++) {
        if (i != j) {
          BayesNode n1 = factorVars[i].getNode();
          BayesNode n2 = factorVars[j].getNode();
          if (getEdge(n1.index,n2.index) == null && getEdge(n2.index,n1.index) == null) {
            Edge e = new Edge(this,n1,n2);
            e.edgeType = GraphConsts.NON_DIRECTIONAL;
            addEdge(e);
          }
        }
      }
    }
  }
  
 /** Removes decision and value nodes from the query */

  public void removeNonRegular() {
    for (int i = decisionNodes.size() - 1; i >= 0; i--) {
      BayesNode tmpNode = (BayesNode)decisionNodes.elementAt(i);
      if (tmpNode.observation == -1 && tmpNode.policy == null) {
        tmpNode.eliminated = true;
        tmpNode.setNodeAppearance( BayesNode.ELIMINATED_NODE );
        removeNode( tmpNode );
      }
    }
    
    for (int i = valueNodes.size() - 1; i >= 0; i--) {
      BayesNode tmpNode = (BayesNode)valueNodes.elementAt(i);
      tmpNode.eliminated = true;
      tmpNode.setNodeAppearance( BayesNode.ELIMINATED_NODE );
      removeNode( tmpNode );
    }
    //buildDecisionNetwork();
  }

  /** This method prunes irrelevant variables from the graph. Irrelevant variables are value 
   * nodes (unless it's the node being queried) and variables that are neither ancestors of
   * the queried node nor of observed nodes. */

  public void removeIrrelevantVars( BayesNode node ) {
    Vector nonAncestors = getNonAncestors( node );
    Vector curPath = new Vector(5,2);
    curPath.addElement(node);
    
    Vector dConnected = new Vector();
    dConnected = findAllConnected( node, curPath, observedNodes, dConnected );
    for (int i = 0; i<numNodes(); i++) {
      BayesNode tmpNode = (BayesNode)nodeAt(i);
      if (!dConnected.contains( tmpNode ) && tmpNode != node && tmpNode.observation == -1) { 
        removeNode( tmpNode );
      }
      else if ( tmpNode != node && tmpNode.nodeType == BayesNode.VALUE_NODE ) {
        removeNode( tmpNode );
      }
      else if (nonAncestors.contains(tmpNode) && tmpNode.observation == -1) {
        removeNode( tmpNode );
      }
    }
    //buildDecisionNetwork();
  }

  /** This method returns all the nodes that are neither ancestors of node nor
   * ancestors of observed nodes. It uses the d-separation methods as helpers */

  protected Vector getNonAncestors( BayesNode node ) {
    Vector vec = new Vector(numNodes());
    for (int i = 0; i<numNodes(); i++) {
      BayesNode tmpNode = (BayesNode)nodeAt(i);
      if (tmpNode == node || descendentOf( node, tmpNode )) {
        vec.addElement(tmpNode);
      }
      else {
        boolean isAncestor = false;

        for (int j = 0;j<observedNodes.size();j++) {
          if (descendentOf( (BayesNode)observedNodes.elementAt(j), tmpNode ) && 
              !node.parents.contains( (BayesNode)observedNodes.elementAt(j) ) ) {
            isAncestor = true;
            break;
          }
        }
        if (isAncestor) {
          vec.addElement(tmpNode);
        }
      }
    }
    Vector nonAncestors = new Vector(numNodes());
    for (int i = 0; i<numNodes(); i++) {
      if (!vec.contains(nodeAt(i))) {
        nonAncestors.addElement(nodeAt(i));
      }
    }
    return nonAncestors;
  }

  // end of querying methods
   
  // The following methods relate to d-separation and conditional independence

  /** This method checks to see whether all the parents of the queryNode have
   * known values (and thus whether the queryNode is conditionally independent of
   * the other. */
  
  private boolean parentsGiven( BayesNode queryNode, Vector given ) {
    if (queryNode.parents.size() == 0) return false;
    for (int i = 0; i< queryNode.parents.size(); i++) {
      if (!given.contains( (BayesNode)queryNode.parents.elementAt(i))) {
        return false;
      }
    }
    return true;
  }
  
  /** This method checks to see whether the first node is a descendent of the second. 
   * Currently, it does depth-first search, since we assume the graphs are acyclic. */
  
  protected static boolean descendentOf( BayesNode firstNode, BayesNode secondNode ) {
    // if it is a direct descendent
    if (firstNode.parents.contains( secondNode )) {
      return true;
    }
    for (int i = 0; i<secondNode.children.size(); i++) {
      BayesNode tmpNode = (BayesNode)secondNode.children.elementAt(i);
      if (descendentOf(firstNode, tmpNode)) return true;
    }
    return false;
  }
  
  /** This method checks whether the first node is an ancestor of the second 
   * or a descendent of an ancestor. */
  
  protected boolean ancestorOf( BayesNode firstNode, BayesNode secondNode ) {
    // if it is a parent
    if( secondNode.parents.contains( firstNode ) ) {
      return true;
    }
    for (int i = 0; i<secondNode.parents.size();i++) {
      BayesNode tmpNode = (BayesNode)secondNode.parents.elementAt(i);
      if (descendentOf( firstNode, tmpNode )) return true;
      if (ancestorOf( firstNode, tmpNode )) return true;
    }
    return false;
  }
  
  /** This method finds all nodes that are d-connected to queryNode
   * and adds them to the Vector dConnected.  */
  
  protected Vector findAllConnected( BayesNode firstNode, Vector curPath, Vector given, Vector dConnected ) {
    // checking for things that shouldn't happen.
    if (curPath.size() >= numNodes() || allInside(firstNode, curPath)) return dConnected; 
    
    Vector connected = new Vector( firstNode.parents.size() + firstNode.children.size() );
    
    for (int i = 0; i<firstNode.parents.size(); i++) {
      if (!curPath.contains( (BayesNode)firstNode.parents.elementAt(i) )) {
        connected.addElement(firstNode.parents.elementAt(i));
      }
    }
    for (int i = 0; i<firstNode.children.size(); i++) {
      if (!curPath.contains( (BayesNode)firstNode.children.elementAt(i) )) {
        connected.addElement(firstNode.children.elementAt(i));
      }
    }
    
    for (int i = 0; i<connected.size(); i++) {
      BayesNode tmpNode = (BayesNode)connected.elementAt(i);
      System.out.println(tmpNode.toString());
      Vector nextPath = new Vector();
      nextPath = addToVector(tmpNode, curPath);
      System.out.println(nextPath.toString());
      if (!isBlockedBy( nextPath, given ) ) {
        if (!dConnected.contains( tmpNode) ) {
          dConnected.addElement( tmpNode );
          Vector blank = findAllConnected( tmpNode, nextPath, given, dConnected );
        }
      }
    }
    return dConnected;
  }
  
  /** This method implements the d-separation criteria, taken from Pearl 2000
   * A path p is said to be d-separated (or blocked) by a set of nodes Z iff
   * 1) p contains a chain i->m->j or a fork i<-m->j s.t. the middle node m
   * is in Z, or
   * 2) p contains an inverted fork (or collider) i->m<-j s.t. the middle node
   * m is not in Z and such that no descendent of m is in Z. 
   */
  
  private boolean isBlockedBy(Vector path, Vector Z) {
    if (path.size() < 3) return false;
    BayesNode node = (BayesNode)path.elementAt(path.size() - 2);
    int type = getTypeOf( node, path );
    
    // Testing for the first case
    if ( (type == QuizCanvas.CHAIN || type == QuizCanvas.FORK) && Z.contains(node)) return true;
    
    // Testing for the second case
    boolean descendentInZ = false;
    if (type == QuizCanvas.COLLIDER && !Z.contains(node)) {
      
      // checking whether a descendent of m is in Z
      for (int j = 0; j<Z.size(); j++) {
        // if an element in Z is a descendent of the node 
        if ( descendentOf( (BayesNode)Z.elementAt(j), node ) ) {
          descendentInZ = true;
        }
      }
      if (!descendentInZ) return true;
    }
    
    return false;
  }

  /** This method is similar to the one above, except it tests every node on
   * a path to see whether that node blocks the path. */

  private boolean isBlockedByComplete( Vector path, Vector Z ) {
    if (path.size() < 3) return false;
    for (int i = 1; i< path.size() - 1; i++) {
      BayesNode node = (BayesNode)path.elementAt(i);
      int type = getTypeOf( node, path );
      if ( (type == QuizCanvas.CHAIN || type == QuizCanvas.FORK) && Z.contains(node)) return true;
      boolean descendentInZ = false;
      if (type == QuizCanvas.COLLIDER && !Z.contains(node)) {
        for (int j = 0; j<Z.size(); j++) {
          if ( descendentOf( (BayesNode)Z.elementAt(j), node ) ) {
            descendentInZ = true;
          }
        }
        if (!descendentInZ) return true;
      }
    }
    return false;
  }
  
  /** This method returns an integer representing whether a particular node is part of
   * a chain, is a fork, or is a collider. It assumes that the node is neither the
   * first nor the last element of the Vector representing its path as well as that the
   * node actually is in the path, returning an error code of -1 if this assumption fails. 
   */
  
  private int getTypeOf( BayesNode node, Vector path ) {
    int index = path.indexOf(node);
    if (index < 1 || index > path.size() - 1) return -1; // this should never happen.
    BayesNode earlier = (BayesNode)path.elementAt(index-1);
    BayesNode later = (BayesNode)path.elementAt(index+1);
    if (node.parents.contains(earlier)) {
      if (node.parents.contains(later)) {
        return QuizCanvas.COLLIDER;
      }
      else {
        return QuizCanvas.CHAIN;
      }
    }
    else if (node.children.contains(later)) {
      return QuizCanvas.FORK;
    }
    else return QuizCanvas.CHAIN;
  }
  
  /** This method creates a new Vector that is a copy of the old, with an element added to
   * the end. It's static because it's useful. */
  
  public static Vector addToVector( Object o, Vector vec ) {
    Vector tmpVec = new Vector(vec.size() + 1);
    for (int i = 0; i<vec.size(); i++) {
      tmpVec.addElement( vec.elementAt(i) );
    }
    tmpVec.addElement(o);
    return tmpVec;
  }
  
  
  /** This method determines whether all the nodes in the Vector are parents or children of
   * the node in question. */
  
  private boolean allInside( BayesNode node, Vector path ) {
    for (int i = 0; i<node.parents.size(); i++) {
      if (!path.contains((BayesNode)node.parents.elementAt(i))) {
        return false;
      }
    }
    for (int i = 0; i<node.children.size(); i++) {
      if (!path.contains((BayesNode)node.children.elementAt(i))) {
        return false;
      }
    }
    return true;
  }
  
  // End of graph theory stuff.	
  
}











