
/**
 * An immutable class representing a particular arrangement of x's and y's on a
 * tic tac toe game board
 */
public class TicTacToeGame {
	private boolean gameOver = false;
	private int emptySquares = 9;
	private int[][] game = { { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } };

	/**
	 * Default Constructor Creates an empty TicTacToeGame
	 */
	public TicTacToeGame() {
	}

	/**
	 * Constructor to create a TicTacToeGame from an array of ints where 1
	 * represents player x, -1 represents player y, and 0 represents an empty
	 * square.
	 * 
	 * @param game
	 *            a 3x3 array of ints containing only members of {-1, 0, 1}
	 */
	public TicTacToeGame(int[][] game) {
		this.game = game;
	}

	/**
	 * Copy Constructor
	 */
	public TicTacToeGame(TicTacToeGame s) {
		gameOver = s.gameOver;
		emptySquares = s.emptySquares;
		for (int i = 0; i < 3; i++) {
			System.arraycopy(s.game[i], 0, game[i], 0, 3);
		}
	}

	/**
	 * Tests whether a move is allowed.
	 * 
	 * @param row
	 *            the row, a member of {0, 1, 2}
	 * @param col
	 *            the column, a member of {0, 1, 2}
	 * @return <code>true</code> if <code>row</code> and <code>col</code>
	 *         represent a valid, empty square and the game has not been won by
	 *         either player
	 */
	public boolean legalMove(int row, int col) {
		return 0 <= row && row < 3 && 0 <= col && col < 3 && !gameOver
				&& game[row][col] == 0;
	}

	/**
	 * Peform a move for player x. Returns a new TicTacToeGame which differs
	 * from the current instance only in that there is an x in the specified
	 * square on the board.
	 * 
	 * @param row
	 *            the row, a member of {0, 1, 2}
	 * @param col
	 *            the column, a member of {0, 1, 2}
	 * @return the new TicTacToeGame
	 */
	public TicTacToeGame xMove(int row, int col) {
		return doMove(row, col, 1);
	}

	/**
	 * Peform a move for player y. Returns a new TicTacToeGame which differs
	 * from the current instance only in that there is a y in the specified
	 * square on the board.
	 * 
	 * @param row
	 *            the row, a member of {0, 1, 2}
	 * @param col
	 *            the column, a member of {0, 1, 2}
	 * @return the new TicTacToeGame
	 */
	public TicTacToeGame yMove(int row, int col) {
		return doMove(row, col, -1);
	}

	/**
	 * Helper method to perform a move. This method does not check whether the
	 * move is legal.
	 * 
	 * @param row
	 *            the row, a member of {0, 1, 2}
	 * @param col
	 *            the column, a member of {0, 1, 2}
	 * @param player
	 *            an int representing the player, 1 for x, -1 for y
	 * @return
	 */
	private TicTacToeGame doMove(int row, int col, int player) {
		TicTacToeGame s = new TicTacToeGame(this);
		if (s.game[row][col] == 0) {
			--s.emptySquares;
		}
		s.game[row][col] = player;
		if (s.winner() != 0 || s.emptySquares == 0) {
			s.gameOver = true;
		}
		return s;
		// if (legalMove(row, col)) {
		// GameState s = new GameState(this);
		// s.game[row][col] = player;
		// if (s.winner() != 0) {
		// s.gameOver = true;
		// }
		// return s;
		// } else
		// return null;
	}

	/**
	 * @return <code>true</code> if the game has been won or if no empty
	 *         squares remain, <code>false</code> otherwise
	 */
	public boolean gameOver() {
		return gameOver;
	}

	/**
	 * This method tests if this <code>TicTacToeGame</code> represents a win
	 * for x or y.
	 * 
	 * @return 1 if X has won, -1 if Y has won, and 0 otherwise
	 */
	public int winner() {
		// If a row, column or diagonal sums to 3 then X has won. If they
		// sum to -3 then Y has won.
		// First check the diagonals
		int diag1 = game[0][0] + game[1][1] + game[2][2];
		int diag2 = game[0][2] + game[1][1] + game[2][0];
		if (diag1 == 3 || diag2 == 3) {
			return 1;
		} else if (diag1 == -3 || diag2 == -3) {
			return -1;
		} else {
			// Check each row and each column
			for (int j = 0; j < 3; j++) {
				int row = 0, col = 0;
				for (int k = 0; k < 3; k++) {
					row += game[j][k];
					col += game[k][j];
				}
				if (row == 3 || col == 3) {
					return 1;
				} else if (row == -3 || col == -3) {
					return -1;
				}
			}
		}
		return 0;
	}

	/**
	 * Method to print the game.
	 */
	public void print() {
		System.out.println("Game State:");
		for (int j = 0; j < 3; j++) {
			for (int k = 0; k < 3; k++) {
				char c = ' ';
				switch (game[j][k]) {
				case -1:
					c = 'y';
					break;
				case 1:
					c = 'x';
					break;
				case 0:
					c = '-';
					break;
				}
				System.out.print(c + " ");
			}
			System.out.println();
		}
	}
}
