
import java.io.IOException;

/**
 * This class implements the minimax algorithm for tic tac toe and allows the
 * user to play against it on the console.
 * 
 */
public class MinimaxTicTacToe {

	/**
	 * A simple class representing a pair of ints.
	 */
	public static class Pair {
		public int x, y;

		public Pair(int x, int y) {
			this.x = x;
			this.y = y;
		}
	}

	/**
	 * This method takes a <code>TicTacToeGame</code> and computes the best
	 * move for player x, returning a float representing the relative value of
	 * the move and returning the move in the given Pair if it is not
	 * <code>null</code>.
	 * 
	 * @param g
	 *            the <code>TicTacToeGame</code>
	 * @param move
	 *            a Pair to be filled in with the best move, can be
	 *            <code>null</code>
	 * @return the relative value of the move, where positive values favour x
	 *         and negative values favour y
	 */
	public static double maxMove(TicTacToeGame g, Pair move) {
		Pair bestMove = null;
		double bestValue = -100;
		// For each legal move, compute the game state resulting from that move
		// and the value of that state assuming the opponent makes the best move
		// it can. The best move is the one that results in the state where the
		// opponent's best possible move is worst.
		for (int j = 0; j < 3; j++) {
			for (int k = 0; k < 3; k++) {
				if (g.legalMove(j, k)) {
					TicTacToeGame newState = g.xMove(j, k);
					double val = minMove(newState);
					if (val > bestValue) {
						bestMove = new Pair(j, k);
						bestValue = val;
					}
				}
			}
		}
		if (bestMove == null) {// No legal moves left
			// Note that this is not strictly necessary as each player assumes
			// the worst outcome (for themselves) when they have no legal moves
			// (such as when the game has been won)
			return g.winner();
		}
		if (move != null) {
			move.x = bestMove.x;
			move.y = bestMove.y;
		}
		return 0.9 * bestValue;
	}

	/**
	 * This method takes a <code>TicTacToeGame</code> and computes the best
	 * move for player y, returning a float representing the relative value of
	 * the move.
	 * 
	 * @param g
	 *            the <code>TicTacToeGame</code>
	 * @return the relative value of the move, where positive values favour x
	 *         and negative values favour y
	 */
	public static double minMove(TicTacToeGame g) {
		Pair bestMove = null;
		double bestValue = 100;
		for (int j = 0; j < 3; j++) {
			for (int k = 0; k < 3; k++) {
				if (g.legalMove(j, k)) {
					TicTacToeGame newState = g.yMove(j, k);
					double val = maxMove(newState, null);
					if (val < bestValue) {
						bestMove = new Pair(j, k);
						bestValue = val;
					}
				}
			}
		}
		if (bestMove == null) {// No legal moves left
			// Note that this is not strictly necessary as each player assumes
			// the worst outcome (for themselves) when they have no legal moves
			// (such as when the game has been won)
			return g.winner();
		}
		return 0.9 * bestValue;
	}

	public static void printDirections() {
		System.out.println("Minimax Tic Tac Toe\n\n"
				+ "Computer is x, you are y. To move, enter the number \n"
				+ "corresponding to the square to be marked with a y. \n"
				+ "The squares are numbered as follows:\n\n"
				+ "\t1 2 3\n\t4 5 6\n\t7 8 9\n");
	}

	public static void printResult(TicTacToeGame game) {
		int winner = game.winner();
		System.out.println("Game Over. "
				+ (winner == 0 ? "Result is a draw."
						: winner == 1 ? "Computer wins."
								: winner == -1 ? "You won!" : ""));
	}

	/**
	 * Prompts the user to enter a move, waiting until a legal move for the
	 * specified <code>TicTacToeGame</code> is entered
	 * 
	 * @param game
	 *            the <code>TicTacToeGame</code>
	 * @return the move as a Pair
	 */
	public static Pair promptForMove(TicTacToeGame game) {
		System.out.print("Enter your move: ");
		boolean validMove = false;
		char move;
		int row = -1, col = -1;
		while (!validMove) {
			try {
				move = (char) System.in.read();
			} catch (IOException e) {
				break;
			}
			switch (move) {
			case '1':
				col = 0;
				row = 0;
				break;
			case '2':
				col = 1;
				row = 0;
				break;
			case '3':
				col = 2;
				row = 0;
				break;
			case '4':
				col = 0;
				row = 1;
				break;
			case '5':
				col = 1;
				row = 1;
				break;
			case '6':
				col = 2;
				row = 1;
				break;
			case '7':
				col = 0;
				row = 2;
				break;
			case '8':
				col = 1;
				row = 2;
				break;
			case '9':
				col = 2;
				row = 2;
				break;
			}
			validMove = game.legalMove(row, col);
		}
		return new Pair(row, col);
	}

	/**
	 * Plays a game of tic tac toe, starting from the supplied game state
	 * 
	 * @param game
	 *            the <code>TicTacToeGame</code> to start from
	 * @param computerStarts
	 *            if <code>true</code>, the computer will make the first move
	 */
	public static void play(TicTacToeGame game, boolean computerStarts) {
		Pair move = new Pair(-1, -1);
		if (computerStarts) {
			maxMove(game, move);
			game = game.xMove(move.y, move.x);
			if (game.gameOver()) {
				game.print();
				int winner = game.winner();
				printResult(game);
				return;
			}
		}
		game.print();
		while (true) {
			move = promptForMove(game);
			game = game.yMove(move.x, move.y);
			if (game.gameOver()) {
				game.print();
				printResult(game);
				return;
			}
			maxMove(game, move);
			game = game.xMove(move.y, move.x);
			game.print();
			if (game.gameOver()) {
				int winner = game.winner();
				printResult(game);
				return;
			}
		}
	}

	public static void main(String[] args) {
		printDirections();
		TicTacToeGame game = new TicTacToeGame();
		play(game, true);
	}
}
