// scene.h

#ifndef SCENES_H
#define SCENES_H

#include <Afxtempl.h>
#include "mesh.h"
#include "renderer.h"

/** \nosubgrouping */

/** Manages all the available scenes. 
	A scene is made of a Mesh and a corresponding Renderer, and there can be as many scenes defined as the user wishes.
	A scene is identified by a unique SceneID.
*/
class Scenes {

private:
	CArray<Mesh*, Mesh*> meshes;
	CArray<Renderer*, Renderer*> renderers;
	SceneID	count;
	SceneID curr;
	bool checkSceneId(SceneID sID);

public:

#ifndef SKIP_THIS
	Scenes(Mesh* nmesh, Renderer* nrenderer);
	~Scenes();
#endif
	/** Create a new scene and set it as the current one
		\param sID \out The ID of the new created scene
		\retval OK Operation succeeded
		\note The new added scene becomes the current scene, and can be accessed with the variables \e mesh and \e renderer (no need to explicitly call \e getCurrentScene);
	*/
	RESULT addNewScene(SceneID &sID);
	/** Remove the given scene
		\param sID The ID of the scene to be removed
		\retval OK Operation succeeded
		\retval	SceneNotFound  No scene was found with the ID \e sID
		\retval SceneUnremovable Given SceneID is the ID of the current scene (unable to remove current scene). 
	*/
	RESULT removeScene(SceneID sID);
	/** Set current scene to scene with ID \e sID
		\param sID The ID of the scene to be set as current
		\retval OK Operation succeeded
		\retval	SceneNotFound  No scene was found with the ID \e sID
		\note The new added scene becomes the current scene, and can be accessed with the variables \e mesh and \e renderer (no need to explicitly call \e getCurrentScene);
	*/
	RESULT setCurrentScene(SceneID sID);
	/** Get the current scene's Mesh and Renderer. Example of usage: \par
		<CODE>SceneID sID;\n
			  Mesh* newMesh;\n
			  Renderer* newRenderer;\n
			  scenes->getCurrentScene(sID, &newMesh, &newRenderer);
		</CODE>
		\param sID \out The ID of the current scene
		\param nmesh \out The address of a pointer to the Mesh of the scene
		\param nrenderer \out The address of a pointer to the Renderer of the scene
		\retval OK Operation succeeded
	*/
	RESULT getCurrentScene(SceneID &sID, Mesh** nmesh=NULL, Renderer** nrenderer=NULL);
	/** Get the scene's (with ID \e sID) Mesh and Renderer. Example of usage: \par
		<CODE>Mesh* newMesh; \n
			  Renderer* newRendered; \n
			  scenes->getScene(sID, &newMesh, &newRenderer);
		</CODE>
		\param sID The ID of the scene
		\param nmesh \out The address of a pointer to a Mesh (\e *nmesh will point to the scene's Mesh)
		\param nrenderer \out The address of a pointer to the Renderer (\e *nrenderer will point to the scene's Renderer)
		\retval OK Operation succeeded
		\retval	SceneNotFound  No scene was found with the ID \e sID
	*/
	RESULT getScene(SceneID sID, Mesh** nmesh, Renderer** nrenderer);

};

#endif