// Renderer.h

#ifndef RENDERER_H
#define RENDERER_H

#include "params.h"
#include "color.h"
#include "linkedlist.cpp"
#include "line.h"
#include "sphere.h"
#include "cylinder.h"
#include "BitMapImage.h"
#include "vr.h"
#include "wrl.h"

class Mesh;

/** \nosubgrouping */

/** Handles the rendering functions and modes, such as backface culling, 
	perspective/orthographic view, colors, highlights, transformations etc.
*/

class Renderer {

	friend Mesh;

public:

#ifndef SKIP_THIS
	Renderer();
	~Renderer();
#endif

	/************************************************/
	/** \name Rendering modes */
	//@{

	/** Set the rendering mode
		\param mode The new render mode
		\retval OK Operation succeeded
	*/
	RESULT setRenderMode(RENDER_MODE mode = WIREFRAME);
	/** Get current rendering mode 
		\param mode \out Current rendering mode
		\retval OK Operation succeeded
	*/
	RESULT getRenderMode(RENDER_MODE &mode);
	
	/** Set to perspective/orthographics mode
		\param perspective The new perspective/orthographics mode (true for perspective, else false)
		\retval OK Operation succeeded
	*/
	RESULT setPerspective(bool perspective = true);
	/** Get current perspective/orthographics mode
		\param perspective \out If true, current mode is perspective. Otherwise, mode is orthographics
		\retval OK Operation succeeded
	*/
	RESULT getPerspective(bool &perspective);
	
	/** Enable/disable backface culling
		\param enable If true, backface culling is enabled 
		\retval OK Operation succeeded
	*/
	RESULT setBackfaceCulling(bool enable = true); 
	/** Get current backface culling mode (enabled/disabled)
		\param enable \out If true, backface culling is enabled. Otherwise, it's disabled.
		\retval OK Operation succeeded
	*/
	RESULT getBackfaceCulling(bool &enable);
	
	/** Toggles rendering of lines
		\param enable Toggles on (true) or off (false)
		\retval OK Operation succeeded
	*/
	RESULT setLineDrawing(bool enable = true);
	/** Gets the current line drawing policy 
		\param enable \out If true, line drawing is enabled. Otherwise, it's disabled.
		\retval OK Operation succeeded
	*/
	RESULT getLineDrawing(bool &enable);

	/** Toggles rendering of spheres
		\param enable Toggles on (true) or off (false)
		\retval OK Operation succeeded
	*/
	RESULT setSphereDrawing(bool enable = true);
	/** Gets the current sphere drawing policy 
		\param enable \out If true, sphere drawing is enabled. Otherwise, it's disabled.
		\retval OK Operation succeeded
	*/
	RESULT getSphereDrawing(bool &enable);

	/** Toggles rendering of cylinders
		\param enable Toggles on (true) or off (false)
		\retval OK Operation succeeded
	*/
	RESULT setCylinderDrawing(bool enable = true);
	/** Gets the current cylinder drawing policy 
		\param enable \out If true, cylinder drawing is enabled. Otherwise, it's disabled.
		\retval OK Operation succeeded
	*/
	RESULT getCylinderDrawing(bool &enable);

	/** Displays text at given screen coordinates
		\param text The text to be displayed 
		\param coord The coordinate in which the text will be displayed on the screen, where (0,0) is the top left corner 
		\retval OK Operation succeeded
		\retval OversizedString Text is longer than MAX_STRING_SIZE
		\note 
			- If text is NULL, the text is cleared (equivalent to displayText("", coord)). 
			- The text's maximal allowed size is MAX_STRING_SIZE.
			- If coord is NULL, the text is displayed at (0,0).
			- With each call to this function, the previous text is cleared (no matter where it was displayed on the screen).

	*/
	RESULT displayText(const char* text = NULL, const Coord2D *coord = NULL);

	/** Toggles auto rendering after each model change. 
		If set to false, rendering of the window will occur only when the user calls \e refreshScreen().
		Otherwise, the window will be re-rendered after each mouse move and model change.
		\param enable Toggles on (true) or off (false)
		\retval OK Operation succeeded
		\sa refreshScreen()
	*/
	RESULT setAutoRefresh(bool enable = true); 
	/** Refreshes the model viewed in the window (in case the user made a change which has not yet been shown).
		\retval OK Operation succeeded
		\sa setAutoRefresh()
	*/
	RESULT refreshScreen();

	/** Toggles auto scaling after each model change. 
		If set to true, rescaling of the model will occur automatically after every model change.
		\param enable Toggles on (true) or off (false)
		\retval OK Operation succeeded
		\note This function is usually used with the Mesh::startNewModel() function, in order to rescale the model after each change, to fit nicely in the render window.
		\sa Mesh::startNewModel()
	*/
	RESULT setAutoScale(bool enable = true);

	/** Toggles interacting with the model (using the mouse events).
		\param enable If true, all mouse events are enabled (the model can transformed, scaled and rotated); else, they're disabled
		\retval OK Operation succeeded
	*/
	RESULT setMouseInteraction(bool enable = true);

#ifndef SKIP_THIS
	bool   interactionEnabled;	// mouse interaction
#endif
	//@}

	/************************************************/
	/** \name Transformations */
	//@{

	// transformations - rotate, translate and scale

	// functions used directly by the user
	/** Rotate the mesh around a given axis at a given angle
		\param axis The axis to rotate around
		\param angle The angel to rotate by (in degrees)
		\retval OK Operation succeeded
	*/
	RESULT rotate(AXIS axis, int angle);
	/** Translate the mesh
		\param dx How much to translate in the X axis
		\param dy How much to translate in the Y axis
		\param dz How much to translate in the Z axis
		\retval OK Operation succeeded
	*/
	RESULT translate(int dx, int dy, int dz=0);
	/** Scale the mesh (uniformly on each axis).
		\param dy How much to scale the model. If dy>0, the model will be enlarged, otherwise it will be shrinked.
		\retval OK Operation succeeded
	*/
	RESULT scale(int dy);

#ifndef SKIP_THIS
	// functions used when the mouse moves on the screen
	RESULT screen_rotate(float angle, int axisx, int axisy, int axisz);
	RESULT screen_translate(int dx, int dy, int dz=0);
	RESULT screen_scale(int dy);
#endif
	//@}

	/************************************************/
	/** \name Colors */
	//@{

	/** Set background color
		\param color The new color for the background
		\retval OK Operation succeeded
	*/
	RESULT setBgColor(Color color);	
	/** Set background color
		\param color \out The current background color
		\retval OK Operation succeeded
	*/
	RESULT getBgColor(Color& color);

	/** Set the default vertices color 
		\param color The new default vertices color
		\retval OK Operation succeeded
	*/
	RESULT setVerticesDefaultColor(Color color);
	/** Get the default vertices color 
		\param color \out The default vertices color
		\retval OK Operation succeeded
	*/
	RESULT getVerticesDefaultColor(Color& color);

	/** Set the default edges color 
		\param color The new default edges color
		\retval OK Operation succeeded
	*/
	RESULT setEdgesDefaultColor(Color color);
	/** Get the default edges color 
		\param color \out The default edges color
		\retval OK Operation succeeded
	*/
	RESULT getEdgesDefaultColor(Color& color);

	/** Set the default faces color 
		\param color The new default faces color
		\retval OK Operation succeeded
	*/
	RESULT setFacesDefaultColor(const Color color);
	/** Get the default faces color 
		\param color \out The default faces color
		\retval OK Operation succeeded
	*/
	RESULT getFacesDefaultColor(Color& color);

	/** Set the default lines color 
		\param color The new default lines color
		\retval OK Operation succeeded
	*/
	RESULT setLinesDefaultColor(const Color color);
	/** Get the default lines color 
		\param color \out The default lines color
		\retval OK Operation succeeded
	*/
	RESULT getLinesDefaultColor(Color& color);

	/** Set the default spheres color 
		\param color The new default spheres color
		\retval OK Operation succeeded
	*/
	RESULT setSpheresDefaultColor(const Color color);
	/** Get the default spheres color 
		\param color \out The default spheres color
		\retval OK Operation succeeded
	*/
	RESULT getSpheresDefaultColor(Color& color);

	/** Set the default cylinders color 
		\param color The new default cylinders color
		\retval OK Operation succeeded
	*/
	RESULT setCylindersDefaultColor(const Color color);
	/** Get the default cylinders color 
		\param color \out The default cylinders color
		\retval OK Operation succeeded
	*/
	RESULT getCylindersDefaultColor(Color& color);

	/** Set the color of the given vertex
		\param vID The ID of the vertex
		\param color A pointer to the new color. If NULL, the default vertices color is used
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT setVertexColor(VertexID vID, const Color* color = NULL);
	/** Get the color of the given vertex
		\param vID The ID of the vertex
		\param color \out The color of the given vertex
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT getVertexColor(VertexID vID, Color& color);

	/** Set the color of the given edge
		\param eID The ID of the edge
		\param color A pointer to the new color. If NULL, the default edges color is used
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT setEdgeColor(EdgeID eID, const Color* color = NULL);
	/** Get the color of the given edge
		\param eID The ID of the edge
		\param color \out The color of the given edge
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT getEdgeColor(EdgeID eID, Color& color);

	/** Set the color of the given face
		\param fID The ID of the face
		\param color A pointer to the new color. If NULL, the default faces color is used
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
	*/
	RESULT setFaceColor(FaceID fID, const Color* color = NULL);
	/** Get the color of the given face
		\param fID The ID of the face
		\param color \out The color of the given face
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
	*/
	RESULT getFaceColor(FaceID fID, Color& color);

	/** Set the color of the given line
		\param lID The ID of the line
		\param color A pointer to the new color. If NULL, the default lines color is used
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT setLineColor(LineID lID, const Color* color = NULL);
	/** Get the color of the given line
		\param lID The ID of the line
		\param color \out The color of the given line
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT getLineColor(LineID lID, Color& color);

	/** Set the color of the given sphere
		\param sID The ID of the sphere
		\param color A pointer to the new color. If NULL, the default spheres color is used
		\retval OK Operation succeeded
		\retval SphereNotFound No sphere was found with the given ID
	*/
	RESULT setSphereColor(SphereID sID, const Color* color = NULL);
	/** Get the color of the given sphere
		\param sID The ID of the sphere
		\param color \out The color of the given sphere
		\retval OK Operation succeeded
		\retval SphereNotFound No sphere was found with the given ID
	*/
	RESULT getSphereColor(SphereID sID, Color& color);

	/** Set the color of the given cylinder
		\param cID The ID of the cylinder
		\param color A pointer to the new color. If NULL, the default cylinders color is used
		\retval OK Operation succeeded
		\retval CylinderNotFound No cylinder was found with the given ID
	*/
	RESULT setCylinderColor(CylinderID cID, const Color* color = NULL);
	/** Get the color of the given cylinder
		\param cID The ID of the cylinder
		\param color \out The color of the given cylinder
		\retval OK Operation succeeded
		\retval CylinderNotFound No cylinder was found with the given ID
	*/
	RESULT getCylinderColor(CylinderID cID, Color& color);

	/** Sets all the vertices to use the default vertex color. 
		The default vertices color will be changed to the given color 
		\param color The new default vertices color
		\retval OK Operation succeeded
	*/
	RESULT setAllVerticesColor(Color color);
	/** Sets all the edges to use the default edge color. 
		The default edges color will be changed to the given color 
		\param color The new default edges color
		\retval OK Operation succeeded
	*/
	RESULT setAllEdgesColor(Color color);
	/** Sets all the faces to use the default face color. 
		The default faces color will be changed to the given color 
		\param color The new default faces color
		\retval OK Operation succeeded
	*/
	RESULT setAllFacesColor(Color color);
	/** Sets all the lines to use the default line color. 
		The default lines color will be changed to the given color 
		\param color The new default lines color
		\retval OK Operation succeeded
	*/
	RESULT setAllLinesColor(Color color);
	/** Sets all the spheres to use the default sphere color. 
		The default spheres color will be changed to the given color 
		\param color The new default spheres color
		\retval OK Operation succeeded
	*/
	RESULT setAllSpheresColor(Color color);
	/** Sets all the cylinders to use the default cylinder color. 
		The default cylinders color will be changed to the given color 
		\param color The new default cylinders color
		\retval OK Operation succeeded
	*/
	RESULT setAllCylindersColor(Color color);

	/** Set the text color
		\param color The new text color
		\retval OK Operation succeeded
	*/
	RESULT setTextColor(Color color);
	/** Get the text color
		\param color \out The text color
		\retval OK Operation succeeded
	*/
	RESULT getTextColor(Color& color);

	/** Use random colors to color the different faces.
		\param faces A list of all the FaceIDs to color. If NULL, all the faces of the mesh will be randomly colored.
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with one of the given FaceIDs.
	*/
	RESULT colorFaces(const LinkedList<FaceID>* faces = NULL); 
	//@}

	/************************************************/
	/** \name Highlights */
	//@{

	/** Toggles rendering of all vertices as small cubes
		\param highlight If true, all vertices will be rendered as small cubes.
		\retval OK Operation succeeded
		\note Highlighted vertices will be shown in all rendering modes
	*/
	RESULT highlightAllVertices(bool highlight=true);
	/** Toggles rendering of all edges as thick lines
		\param highlight If true, all edges will be rendered as thick lines.
		\retval OK Operation succeeded
		\note Highlighted edges will be shown in all rendering modes
	*/
	RESULT highlightAllEdges(bool highlight=true);
	/** Toggles highlighting of all faces
		\param highlight If true, all faces will be highlighted.
		\retval OK Operation succeeded
		\note Highlighted faces will be shown in all rendering modes
	*/
	RESULT highlightAllFaces(bool highlight=true);
	
	/** Gets the mode of the vertices highlighting (enabled/disabled).
		\param highlight \out If true, vertices highlighting is enabled. Else, it's false
		\retval OK Operation succeeded
	*/
	RESULT getHighlightAllVertices(bool &highlight);
	/** Gets the mode of the edges highlighting (enabled/disabled).
		\param highlight \out If true, edges highlighting is enabled. Else, it's false
		\retval OK Operation succeeded
	*/
	RESULT getHighlightAllEdges(bool &highlight);
	/** Gets the mode of the faces highlighting (enabled/disabled).
		\param highlight \out If true, faces highlighting is enabled. Else, it's false
		\retval OK Operation succeeded
	*/
	RESULT getHighlightAllFaces(bool &highlight);
	
	/** Toggles highlighting of a given vertex
		\param vID The ID of the vertex to highlight
		\param highlight If true, the vertex will be highlighted
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT highlightVertex(VertexID vID, bool highlight=true);
	/** Toggles highlighting of a given edge
		\param eID The ID of the edge to highlight
		\param highlight If true, the edge will be highlighted
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT highlightEdge(EdgeID eID, bool highlight=true);
	/** Toggles highlighting of a given face
		\param fID The ID of the face to highlight
		\param highlight If true, the face will be highlighted
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
	*/
	RESULT highlightFace(FaceID fID, bool highlight=true);
	//@}

	/************************************************/
	/** \name Picking objects */
	//@{

	/** Allows the user to pick a vertex with a mouse left-click
		\param timeOut The desired time until time-out occurs, in milliseconds. 
					If 0, time-out will never occur and the function will wait infinitely until the user picks a vertex 
		\param vID \out The ID of the picked vertex
		\retval OK Operation succeeded
		\retval TimeOut The user didn't respond in the given time
	*/
	RESULT pickVertex(VertexID& vID, unsigned int timeOut = 0);
	/** Allows the user to pick an edge with a mouse left-click
		\param timeOut The desired time until time-out occurs, in milliseconds. 
					If 0, time-out will never occur and the function will wait infinitely until the user picks an edge 
		\param eID \out The ID of the picked edge
		\retval OK Operation succeeded
		\retval TimeOut The user didn't respond in the given time
	*/
	RESULT pickEdge(EdgeID& eID, unsigned int timeOut = 0);
	/** Allows the user to pick a face with a mouse left-click
		\param timeOut The desired time until time-out occurs, in milliseconds. 
					If 0, time-out will never occur and the function will wait infinitely until the user picks a face 
		\param fID \out The ID of the picked face
		\retval OK Operation succeeded
		\retval TimeOut The user didn't respond in the given time
	*/
	RESULT pickFace(FaceID& fID, unsigned int timeOut = 0);
#ifndef SKIP_THIS
	RESULT processSelection(GLint hits, GLuint selectBuf[512]);
	bool   inPickingMode;			// are we in picking mode?
#endif
	//@}

	/************************************************/
	/** \name Texture mapping */
	//@{

	/** Enable/disable texture mapping
		\param enable If true, texture mapping is enabled
		\retval OK Operation succeeded
	*/
	RESULT setTextureMapping(bool enable=true);
	/** Get the current texture mapping state
		\param enable \out If true, texture mapping is enabled
		\retval OK Operation succeeded
	*/
	RESULT getTextureMapping(bool &enable);		
	/** Set the image to use in the texture mapping
		\param fileName The file name of the bitmap image to be used in the texture mapping
		\retval OK Operation succeeded
		\retval NullPointerGiven \a fileName is NULL
		\retval FileNotFound The file was not found in the given location
		\note The image can be only of type 24-bit bitmap (i.e. *.bmp). If the image's height or width are not a power of 2, the image will be automatically cropped to the proper size (the closest lower power of 2).
	*/
	RESULT setTextureImage(const char* fileName);
	/** Get the image used in the texture mapping
		\param fileName The file name of the bitmap image used in the texture mapping
		\retval OK Operation succeeded
		\retval NullPointerGiven \a fileName is NULL
		\note If \e fileName is not NULL, the function will copy the file name into it. \e fileName needs to be allocated by the user before use!
	*/
	RESULT getTextureImage(char* fileName);
	/** Set the (u,v) texture mapping coordinate of a vertex
		\param vID The ID of the vertex to set
		\param u The u-coordinate of the texture mapping (for this vertex)
		\param v The v-coordinate of the texture mapping (for this vertex)
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
		\note \e u and \e v should usually be in the range [0, 1]. 
				\e u is the coordinate in the x-axis of the bitmap, and \e v is the coordinate in the y-axis. 
				The coordinate (0,0) is the bottom left corner of the bitmap file, and the coordinate (1,1) is the top right corner. 
				If \e u or \e v are outside that range, the image will be repeated as needed.
	*/
	RESULT setVertexUV(VertexID vID, double u, double v);	
	/** Set the (u,v) texture mapping coordinate for a list of vertices
		\param vertices The list of VertexIDs of the vertices to set (u,v) coordinates for
		\param coords The (u,v) coordinates of the texture mapping, given as a 2D coord. 
					The first cell in each coordinate (i.e. coord[0]) represents the u-coordinate, the second cell (i.e. coord[1]) represents the v-coordinate.
		\retval OK Operation succeeded
		\retval NullPointerGiven One of the given lists is NULL
		\retval VertexNotFound No vertex was found one of the given IDs in \e vertices
		\note	The 2 lists should be of the same size. The i'th coord in \e coords defines the (u,v) coordinate for the i'th VertexID in \e vertices.
				If the lists have different lengths (meaning there are extra vertices IDs or extra coordinates), the extra data in the longer list will be ignored.\n
				\e u and \e v should usually be in the range [0, 1]. 
				\e u is the coordinate in the x-axis of the bitmap, and \e v is the coordinate in the y-axis. 
				The coordinate (0,0) is the bottom left corner of the bitmap file, and the coordinate (1,1) is the top right corner. 
				If \e u or \e v are outside that range, the image will be repeated as needed.
	*/
	RESULT setVerticesUV(const LinkedList<VertexID> *vertices, const LinkedList<Coord2D> *coords);
	/** Get the (u,v) texture mapping coordinate of a vertex
		\param vID The ID of the vertex to get (u,v) coordinates of
		\param u \out The u-coordinate of the texture mapping (for this vertex)
		\param v \out The v-coordinate of the texture mapping (for this vertex)
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT getVertexUV(VertexID vID, double &u, double &v);	
	//@}

	/************************************************/
	/** Misc. */
#ifndef SKIP_THIS
	// set the projection matrix
	RESULT setProjection();
	// set the viewport according to new window size
	RESULT setSize(int cx, int cy);
	// reset
	RESULT reset();
	RESULT resetModel();

	// render the given mesh
	RESULT setMesh(Mesh* mesh);
	RESULT render(bool selectionMode=false);
#endif
	
private:

	// rendering
	Mesh*		mesh;				// the mesh to render
	RENDER_MODE renderMode;			// rendering mode
	bool		perspective;		// perspective/orthographic view
	bool		backfaceRemoval;	// backface culling
	bool		autoRefresh;		// auto refreshing the screen
	bool		refreshCalled;		// was refresh called?
	bool		autoScale;			// auto scaling the model
	RESULT		renderFaces(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderWireframe(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderVertices(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderHighlightedFaces(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderHighlightedEdges(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderHighlightedVertices(VRIndexedFaceSet *vrIFS, bool selectionMode);
	RESULT		renderLines(bool selectionMode);
	RESULT		renderSpheres(bool selectionMode);
	RESULT		renderCylinders(bool selectionMode);
	void		findOrthonormalVectors(Coord &z, Coord &x, Coord &y);

	// drawing lines
	bool		drawLines;			// drawing lines
	double		lineDefaultWidth;	// line default width

	// drawing spheres
	bool		drawSpheres;			// drawing spheres

	// drawing Cylinder
	bool		drawCylinders;			// drawing spheres

	char		text[MAX_STRING_SIZE];
	Coord2D		textCoord;
	bool		newText;
	void		drawText(double TR);

	// colors	
	Color		bgColor;			// background color		
	Color		vertexDefaultColor; // vertices' default color
	Color		edgeDefaultColor;	// edges' default color
	Color		faceDefaultColor;	// faces' default color
	Color		lineDefaultColor;	// lines' default color
	Color		sphereDefaultColor;	// spheres' default color
	Color		cylinderDefaultColor;	// spheres' default color
	Color		textColor;
	double		ambient, diffuse;

	// highlights
	bool		doHighlightAllVertices;
	bool		doHighlightAllEdges;
	bool		doHighlightAllFaces;
	LinkedList<VertexID>	*verticesToHighlight;
	LinkedList<EdgeID>		*edgesToHighlight;
	LinkedList<FaceID>		*facesToHighlight;
	
	// center of rotation
	Coord	centerRot;  
	float	transformMatrix[16];
	float	rotateMatrix[16];
	void	unifyTransformMatrix();
	void	unifyRotateMatrix();
	float	scalingParameter;
	void	initialTranslate(double x, double y, double z);
	void	initialScale(double val);
	void	drawBox(GLfloat size);

	// text

	/********* texture mapping ***********/

	bool			textureMapping;	// texture mapping enabled/disabled
	CBitMapImage	img;			// the bmp image to use as texture mapping

	/********* picking ***********/

	HANDLE pickingMutex;
	PICKING_MODE pickingMode;			// the item we want to pick (edge/vertex/face)
	long pickedItem;				// the name of the item that was picked
	bool processHits(GLint hits, GLuint buffer[]);
};

#ifndef SKIP_THIS

// a function used for redrawing the screen
extern HWND getHWND();
extern void refresh();
extern RECT getClientRect();

#endif

extern Renderer *renderer;

#endif