// params.h

#ifndef PARAMS_H
#define PARAMS_H

#ifndef SKIP_THIS

#define MESH_RESET		WM_USER+1
#define MESH_PROJECTION WM_USER+2
#define MESH_ROTATE		WM_USER+3
#define MESH_SCALE		WM_USER+4
#define MESH_TRANSLATE	WM_USER+5

typedef struct translate_params {
	int dx, dy, dz;
} translate_params;

typedef struct rotate_params {
	float angle;
	int dx, dy, dz;
} rotate_params;

typedef enum {TRANSLATE, SCALE, ROTATE, NONE} TRANSFORM_MODE;

typedef enum {FACE, EDGE, VERTEX} PICKING_MODE;

#endif

/** \name Max and min functions */
//@{

/// Calculate maximum of 3 values
#define MAX3(a, b, c) ((a>b) ? (a>c ? a : c) : (b>c ? b : c))
/// Calculate maximum of 2 values
#define MAX(a, b)	 ((a>b) ? a : b)

/// Calculate minimum of 3 values
#define MIN3(a, b, c) ((a<b) ? (a<c ? a : c) : (b<c ? b : c))
/// Calculate minimum of 2 values
#define MIN(a, b)	 ((a<b) ? a : b)

//@}

/** \name Constants */
//@{

/// Maximum allowed string size.
#define MAX_STRING_SIZE  100
/// Maximum number of vertices in a Face (currently, only triangles are allowed).
#define MAX_FACE_VERTICES 3

/// Default Line width.
#define DEFAULT_LINE_WIDTH 5.0
/// Default Sphere radius.
#define DEFAULT_SPHERE_RADIUS 1.0
/// Default Cylinder radius.
#define DEFAULT_CYLINDER_RADIUS 1.0

//@}

/** \name Object IDs */
//@{

/** Identifies a vertex. A vertex will retain its ID throughout the program unless it has been removed. */
typedef long	VertexID;
	
/** Identifies a half-edge, which is an edge with a direction (or side). For example, the edge between vertices 0 and 2 has 2 half-edges: one is (0,2) and the other is (2,0). A half-edge will retain its ID throughout the program unless it has been removed. This leads to an edge having 2 ID's (one for each half-edge). */
typedef long	EdgeID;		

/** Identifies a face. A face will retain its ID throughout the program unless it has been removed.
	The connection between a FaceID and an EdgeID is as follows:
	An edge of a face (with a faceID of j), will have one of the following ID's: (j*3, j*3+1, j*3+2). The edge with the ID of j*3 defines the 1st and 2nd  vertices of the face. The edge with the ID of j*3+1 defines the face's 2nd and 3rd vertices, and the edge with the ID of j*3+2 defines the face's 3rd and 1st vertices. This is illustrated below:
	\image html faceid.jpg
	It can be seen from this example, that most edges will have more than one edgeID (one for every face with which they are associated).
*/
typedef long	FaceID;	
/** Identifies a line. A line will retain its ID throughout the program unless it has been removed. */
typedef long	LineID;		
/** Identifies a sphere. A sphere will retain its ID throughout the program unless it has been removed. */
typedef long	SphereID;	
/** Identifies a cylinder. A cylinder will retain its ID throughout the program unless it has been removed. */
typedef long	CylinderID;	
/** Identifies a scene. A scene will retain its ID throughout the program unless it has been removed. */
typedef long	SceneID;	

//@}

/** The result of each operation. */
typedef enum { OK /** Operation completed successfully */, 
	NullPointerGiven /** One of the input parameters was NULL */ , 
	OversizedString /** A problem with the given string length */ , 
	FileNotFound /** Couldn't find the file to open */ , 
	MeshNotValid /** %Mesh is either non valid or is not specified */ , 
	WrongFileFormat /** VRML file in the wrong format (has no IndexedFaceSets or has other Geometry shapes as well) */ , 
	VertexNotFound /** No vertex was found with the given VertexID */ , 
	EdgeNotFound /** No edge was found with the given EdgeID */ , 
	FaceNotFound /** No face was found with the given FaceID */ , 
	LineNotFound /** No line was found with the given LineID */ , 
	SphereNotFound /** No sphere was found with the given SphereID */ , 
	CylinderNotFound /** No cylinder was found with the given CylinderID */ , 
	SceneNotFound /** No scene was found with given SceneID */ , 
	VerticesNotIncident /** The 2 given vertices are not incident (do not share an edge) */ , 
	EdgesNotIncident /** The 2 given edges are not incident (do not share a common vertex) */ , 
	FacesNotIncident /** The 2 given faces are not incident (do not share an edge)  */ , 
	SameVertexGiven /** The 2 given VertexID's represent the same vertex */ , 
	SameEdgeGiven /** The 2 given EdgeID's represent the same edge */ , 
	SameFaceGiven /** The 2 given FaceID's represent the same face */ , 
	FaceExists /** Given face already exists in the mesh */ , 
	TimeOut /** There was a time-out while waiting for the user to pick an object */ , 
	VertexIsOrphan /** The given vertex is orphan (is not associated with any face, therefore doesn't have a neighborhood) */ , 
	SceneUnremovable  /** Scene cannot be removed (usually because the user asked to remove the current scene) */ 
} RESULT;

/** Describes the render mode. In all these modes, the highlighted items are always shown */
typedef enum { SOLID /** Render faces only */, 
	WIREFRAME /** Render edges only */, 
	SOLID_AND_WIREFRAME /** Render faces with the wireframe rendered on top */, 
	VERTICES /** Render vertices only */, 
	HIGHLIGHT_ONLY /** Render only highlighted items */
} RENDER_MODE;

/// Defines an axis - X, Y or Z
typedef enum {AXIS_X, AXIS_Y, AXIS_Z} AXIS;

/// Defines an R, G or B color component
typedef enum {R=0, G, B} COLOR;

/** Get the current sotware version. 
	\param version An allocated 
	\retval OK Operation succeeded
	\retval NullPointerGiven \a version is NULL
	\note If \a version is not NULL, the function will copy the version into it. \a version needs to be allocated by the user before use, and it should be long enough to contain the current version
*/
RESULT getVersion(char* version);

#endif



