// mesh.h

#ifndef MESH_H
#define MESH_H

#include "renderer.h"
#include "params.h"
#include "info.h"
#include "corner.h"
#include "edge.h"
#include "face.h"
#include "vr.h"
#include "wrl.h"

class Renderer;
class Scenes;

/** \nosubgrouping */

/**  Handles the mesh functions, such as I/O, VRML export, mesh geometry functions etc. 
*/
class Mesh {

	friend Renderer;
	friend Scenes;

public:

#ifndef SKIP_THIS
	Mesh();
	~Mesh();
#endif

	/************************************************/
	/** \name I/O Operations */
	//@{

	/** Loads a VRML file and renders it
		\param fileName The path of the *.wrl file to load
		\retval OK Operation succeeded
		\retval NullPointerGiven \e fileName is NULL.
		\retval OversizedString \e fileName is longer than MAX_STRING_SIZE.
		\retval FileNotFound File isn't found in given path.
		\retval WrongFileFormat VRML is not in the right format (has no IndexedFaceSets or has other geometry shapes (Box etc.)).
		\note \e fileName should be shorter than MAX_STRING_SIZE characters. \n
				The vertices' IDs are initialized according to each vertex's index in the IFS->coord->point array. \n
				The faces' IDs are initialized according to the face's position in the IFS->coordIndex array, such that each face is defined by 4 values in the coordIndex array (the 3 vertices and an extra '-1'). 
				The first face (with FaceID=0) is defined by the first 4 values in the array, the second face (with FaceID=1) by the next 4 and so on
	*/
	RESULT openVRMLFile(const char* fileName);	
	/** Saves the current mesh to a VRML file
		\param fileName The path of the *.wrl file to be saved. If NULL, the mesh will be saved to the predefined VRML file (when previously saved or loaded)
		\retval OK Operation succeeded
		\retval NullPointerGiven \e fileName is NULL and there is no predefined VRML file (the mesh wasn't saved before).
		\retval OversizedString \e fileName is longer than MAX_STRING_SIZE.
		\retval FileNotFound File isn't found in given path.
		\retval MeshNotValid Mesh is either non valid or is not specified
		\note \e fileName should be shorter than MAX_STRING_SIZE characters.\n
				The file saved contains only the geometry and connectivity (i.e. vertices and faces), and disregards colors, highlights, spheres, lines and cylinders
	*/
	RESULT saveVRMLFile(const char* fileName = NULL);
	/** Saves the current mesh to a VRML file, saving all the highlighted vertices/edges/faces and their colors
		\retval OK Operation succeeded
		\retval NullPointerGiven \e fileName is NULL and there is no predefined VRML file (the mesh wasn't saved before).
		\retval OversizedString \e fileName is longer than MAX_STRING_SIZE.
		\retval FileNotFound File isn't found in given path.
		\retval MeshNotValid Mesh is either non valid or is not specified
		\note \e fileName should be shorter than MAX_STRING_SIZE characters.\n
				The file saved contains the geometry and connectivity (i.e. vertices and faces), as well as colors, highlights, spheres, lines and cylinders
	*/
	RESULT VRMLExport(const char* fileName);
	/** Get the name of the current VRML file
		\param fileName \out A pointer to a string to which the file name will be copied
		\retval OK Operation succeeded
		\retval NullPointerGiven \e fileName is NULL and there is no predefined VRML file (the mesh wasn't saved before).
		\retval MeshNotValid Mesh is either non valid or is not specified
	*/
	RESULT getFileName(char* fileName);

	//@}

	/************************************************/
	/** \name IFS/model */
	//@{

	/** Get the current mesh's bounding box
		\param bBoxMin \out The minimum coordinate of the bounding box
		\param bBoxMax \out The maximum coordinate of the bounding box
		\param bBoxDim \out The dimensions of the bounding box (=bBoxMax-bBoxMin)
		\param center \out The center coordinate of the bounding box
		\retval OK Operation succeeded
		\retval MeshNotValid Mesh is either non valid or is not specified
		\note If line/sphere/cylinder drawing is enabled, the bounding box is calculated according to the model as well as the lines/spheres/cylinders to draw. 
			Otherwise, they lines are not added to the calculation
	*/
	RESULT getBoundingBox(Coord &bBoxMin, Coord &bBoxMax, Coord &bBoxDim, Coord &center);
	/** Get a copy of the current IndexedFaceSet which represents the mesh
		\param IFS \out A copy of the IndexedFaceSet of the mesh
		\retval OK Operation succeeded
		\retval NullPointerGiven \e IFS is NULL
		\retval MeshNotValid Mesh is either non valid or is not specified
		\note \e IFS should be allocated by the user before the call to the function. 
			The structure itself is changed inside the function to be a copy of the mesh. 
			This means that changing the updated structure returned to the user, won't change the data structure used by the API, only the user's local copy
	*/
	RESULT getIFS(VRIndexedFaceSet* IFS);
	/** Sets the current model to be a copy of the given IndexedFaceSet
		\param IFS The IndexedFaceSet of the mesh
		\retval OK Operation succeeded
		\retval NullPointerGiven \e IFS is NULL
	*/
	RESULT setIFS(const VRIndexedFaceSet* IFS);	
	/** Starts a new empty model (clears the current mesh and the screen).
		\retval OK Operation succeeded
		\note This function is usually followed by a call to Renderer::setAutoScale()
		\sa Renderer::setAutoScale()
	*/
	RESULT startNewModel();

	//@}

	/************************************************/
	/** \name Line drawing */
	//@{

	/** Adds a line to be drawn.
		\param line The line to be added.
		\param color The color of the new line, or NULL for the default line color
		\param lineID \out The ID of the new added line
		\retval OK Operation succeeded
	*/
	RESULT addLine(Line line, LineID &lineID, const Color* color = NULL);
	/** Set the line's coordinates
		\param lineID The ID of the line
		\param line The Line object with the new coordinates
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT setLine(LineID lineID, Line line);
	/** Get a copy of a line with a given ID
		\param lineID The ID of the line
		\param line \out A copy of the line.
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT getLine(LineID lineID, Line& line);
	/** Set the width of a line with a given ID
		\param lineID The ID of the line
		\param width The new width of the line
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT setLineWidth(LineID lineID, double width=DEFAULT_LINE_WIDTH);
	/** Get the width of a line with a given ID
		\param lineID The ID of the line
		\param width \out The width of the line
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT getLineWidth(LineID lineID, double &width);
	/** Remove the given line
		\param lineID The ID of the line to remove
		\retval OK Operation succeeded
		\retval LineNotFound No line was found with the given ID
	*/
	RESULT removeLine(LineID lineID);
	/** Remove all lines
		\retval OK Operation succeeded
	*/
	RESULT removeAllLines();

	//@}

	/************************************************/
	/** \name Sphere drawing */
	//@{

	/** Adds a sphere to be drawn.
		\param sphere The sphere to be added.
		\param color The color of the new sphere, or NULL for the default sphere color
		\param sId \out The ID of the new added sphere
		\retval OK Operation succeeded
	*/
	RESULT addSphere(Sphere sphere, SphereID &sId, const Color* color = NULL);
	/** Set the sphere's parameters (center and radius)
		\param sId The ID of the sphere
		\param sphere The Sphere object with the new parameters
		\retval OK Operation succeeded
		\retval SphereNotFound No sphere was found with the given ID
	*/
	RESULT setSphere(SphereID sId, Sphere sphere);
	/** Get a copy of a sphere with a given ID
		\param sId The ID of the sphere
		\param sphere \out A copy of the sphere.
		\retval OK Operation succeeded
		\retval SphereNotFound No sphere was found with the given ID
	*/
	RESULT getSphere(SphereID sId, Sphere& sphere);
	/** Remove the given sphere
		\param sId The ID of the sphere to remove
		\retval OK Operation succeeded
		\retval SphereNotFound No sphere was found with the given ID
	*/
	RESULT removeSphere(SphereID sId);
	/** Remove all spheres
		\retval OK Operation succeeded
	*/
	RESULT removeAllSpheres();

	//@}

	/************************************************/
	/** \name Cylinder drawing */
	//@{

	/** Adds a cylinder to be drawn.
		\param cylinder The cylinder to be added.
		\param color The color of the new cylinder, or NULL for the default cylinder color
		\param cId \out The ID of the new added cylinder
		\retval OK Operation succeeded
	*/
	RESULT addCylinder(Cylinder cylinder, CylinderID &cId, const Color* color = NULL);
	/** Set the cylinder's parameters
		\param cId The ID of the cylinder
		\param cylinder The Cylinder object with the new parameters
		\retval OK Operation succeeded
		\retval CylinderNotFound No cylinder was found with the given ID
	*/
	RESULT setCylinder(CylinderID cId, Cylinder cylinder);
	/** Get a copy of a cylinder with a given ID
		\param cId The ID of the cylinder
		\param cylinder \out A copy of the cylinder.
		\retval OK Operation succeeded
		\retval CylinderNotFound No sphere was found with the given ID
	*/
	RESULT getCylinder(CylinderID cId, Cylinder& cylinder);
	/** Remove the given cylinder
		\param cId The ID of the cylinder to remove
		\retval OK Operation succeeded
		\retval CylinderNotFound No sphere was found with the given ID
	*/
	RESULT removeCylinder(CylinderID cId);
	/** Remove all cylinders
		\retval OK Operation succeeded
	*/
	RESULT removeAllCylinders();

	//@}

	/************************************************/
	/** \name Mesh Geometry */
	//@{

	/** Get the total number of vertices in the mesh model
		\param count \out The number of vertices
		\retval OK Operation succeeded
	*/
	RESULT getVerticesCount(VertexID& count);

	/** Get the total number of unique geometric edges in the mesh model. 
		If an edge has 2 EdgeID's, the edge is counted only once.
		\param count \out The number of vertices
		\retval OK Operation succeeded
	*/
	RESULT getEdgesCount(EdgeID& count);

	/** Get the total number of edges ID's in the mesh model.
		If an edge has 2 EdgeID's, both ID's will be counted.
		\param count \out The number of edges
		\retval OK Operation succeeded
	*/
	RESULT getEdgesIDsCount(EdgeID& count);

	/** Get the total number of faces in the mesh model
		\param count \out The number of faces
		\retval OK Operation succeeded
	*/
	RESULT getFacesCount(FaceID& count);

	/** Get the number of lines to draw
		\param count \out The number of lines
		\retval OK Operation succeeded
	*/
	RESULT getLinesCount(LineID& count);

	/** Get the number of spheres to draw
		\param count \out The number of spheres
		\retval OK Operation succeeded
	*/
	RESULT getSpheresCount(SphereID& count);

	/** Get the number of cylinders to draw
		\param count \out The number of cylinders
		\retval OK Operation succeeded
	*/
	RESULT getCylindersCount(SphereID& count);

	/** Move given vertex to a given location
		\param vID The ID of the vertex to move.
		\param coord The new vertex coordinate
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT setCoord(VertexID vID, Coord coord);	

	/** Move given vertices to given locations
		\param vertices A list of the IDs of the vertices to move
		\param coords The new vertices coordinates
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval NullPointerGiven One of the given lists is NULL
		\note The entries in \e coords list should correspond to the entries in \e vertices. 
			That is, the first entry in \e coords should be the new coordinate for the first entry in \e vertices, and so on.\n
			The changes will be visible on the screen only after all vertices are moved (or if a problem occurs before that).\n
			If the lists have different lengths (meaning there are extra vertices IDs or extra coordinates), the extra data in the longer list will be ignored.
	*/
	RESULT setCoords(const LinkedList<VertexID> *vertices, const LinkedList<Coord> *coords);

	/** Get the coordinate of the given vertex
		\param vID The ID of the vertex
		\param coord \out The coordinate of the vertex
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
	*/
	RESULT getCoord(VertexID vID, Coord &coord);

	/** Get the coordinates of the given vertices
		\param vertices A list of all the vertices IDs we want to get coords from
		\param coords \out The corresponding vertices' coordinates
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval NullPointerGiven One of the given lists is NULL
		\note The entries in \e coords correspond to the entries in the \e vertices. 
			That is, the first entry in \e coords is the coordinate for the first entry in \e vertices, and so on.
	*/
	RESULT getCoords(const LinkedList<VertexID> *vertices, LinkedList<Coord> *coords);

	/** Get the coordinates of all the valid mesh vertices, including their IDs
		\param vertices \out All the valid vertices' IDs
		\param coords \out The corresponding vertices' coordinates
		\retval OK Operation succeeded
		\retval NullPointerGiven One of the given lists is NULL
		\note \e vertices contain all the valid mesh vertices' IDs.\n
			The entries in \e coords correspond to the entries in the \e vertices. 
			That is, the first entry in \e coords is the coordinate for the first entry in \e vertices, and so on.
	*/
	RESULT getAllCoords(LinkedList<VertexID> *vertices, LinkedList<Coord> *coords);

	/** Get the edge with the given ID
		\param eID The edge ID
		\param edge \out A copy of the edge
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT getEdge(EdgeID eID, Edge& edge);

	/** Get the face with the given ID
		\param fID The face ID
		\param face \out A copy of the face
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
	*/
	RESULT getFace(FaceID fID, Face& face);

	/** Get the faces of the given faces IDs
		\param facesids A list of all the faces IDs we want to get faces from
		\param faces \out The corresponding faces
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with one of the given IDs
		\retval NullPointerGiven One of the given lists is NULL
		\note The entries in \e faces correspond to the entries in the \e facesids. 
			That is, the first entry in \e faces is the face of the first entry in \e facesids, and so on.
	*/
	RESULT getFaces(const LinkedList<FaceID> *facesids, LinkedList<Face> *faces);

	/** Get all valid mesh faces, including their IDs
		\param facesids \out All the valid faces' IDs
		\param faces \out The corresponding faces
		\retval OK Operation succeeded
		\retval NullPointerGiven One of the given lists is NULL
		\note \e facesids contain all the valid faces' IDs.\n
			The entries in \e faces correspond to the entries in the \e facesids. 
			That is, the first entry in \e faces is the face of the first entry in \e facesids, and so on.
	*/
	RESULT getAllFaces(LinkedList<FaceID> *facesids, LinkedList<Face> *faces);

	/** Get the other edgeID of the edge with the given edgeID. Since each edge is actually made of 2 half-edges,
		each edge can have 2 EdgeID's (if there are 2 triangles attached to it).
		\param eID The edge ID
		\param otherEdgeID \out The other edgeID of the given edge. If equals '-1', there is only one ID for the given edge
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT getOtherEdgeID(EdgeID eID, EdgeID& otherEdgeID);

	/** Get the ID of the edge joining the 2 given faces. 
		Since the edge actually has 2 IDs (one when appearing in the first face, and another when appearing in the second face), 
		this function actually returns both of the IDs
		\param fID1 The ID of the first face
		\param fID2 The ID of the second face
		\param eID1 \out The ID of the joined edge (associated with the first face)
		\param eID2 \out The ID of the joined edge (associated with the second face)
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with one of the given IDs
		\retval FacesNotIncident The 2 faces are not incident.
		\retval SameFaceGiven The 2 given face IDs are the same

	*/
	RESULT getJoinedFacesEdge(FaceID fID1, FaceID fID2, EdgeID &eID1, EdgeID &eID2);

	/** Get the ID of the joined edge between the 2 given vertices. If the joined edge has 2 IDs, this function returns both of them. 
		If it has only one (there's only one face connecting the two vertices), the ID is returned in eID1, and eID2 is -1
		\param vID1 The ID of the first vertex
		\param vID2 The ID of the second vertex
		\param eID1 \out The ID of the joined edge
		\param eID2 \out The other ID of the joined edge
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval VerticesNotIncident The 2 vertices are not incident.
		\retval SameVertexGiven The 2 given vertex IDs are the same
	*/
	RESULT getJoinedVerticesEdge(VertexID vID1, VertexID vID2, EdgeID &eID1, EdgeID &eID2);

	/** Get the ID of the joined face of the 3 given vertices
		\param vID1 The ID of the first vertex
		\param vID2 The ID of the second vertex
		\param vID3 The ID of the third vertex
		\param fID \out The ID of the joined face
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval VerticesNotIncident The 3 vertices are not incident (do not form a face)
		\retval SameVertexGiven At least 2 of the given vertex IDs are the same
	*/
	RESULT getJoinedVerticesFace(VertexID vID1, VertexID vID2, VertexID vID3, FaceID &fID);

	/** Get the ID of the joined vertex between the two given edges
		\param eID1 The ID of the first edge
		\param eID2 The ID of the second edge
		\param vID \out The ID of the joined vertex
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with one of the given IDs
		\retval EdgesNotIncident The 2 edges are not incident (do not share a vertex)
		\retval SameEdgeGiven The 2 given edge IDs represent the same edge
	*/
	RESULT getJoinedEdgesVertex(EdgeID eID1, EdgeID eID2, VertexID &vID);

	/** Get the 2 vertices which define the given edge
		\param eID The ID of the edge
		\param vID1 \out The ID of the first vertex
		\param vID2 \out The ID of the second vertex
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT getVerticesOfEdge(EdgeID eID, VertexID &vID1,VertexID &vID2);

	/** Get the face associated with this half edge
		\param eID The ID of the edge
		\param fID \out The ID of the associated face
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with the given ID
	*/
	RESULT getFaceOfEdge(EdgeID eID, FaceID &fID);

	/** Get the neighborhood of the given face (its vertices, edges, and neighboring faces). 
		\param fID The ID of the face
		\param vertices \out If not NULL, will contain the face vertices' IDs (in ccw order)
		\param edges \out If not NULL, will contain the face edges' IDs (in ccw order)
		\param faces \out If not NULL, will contain the face's neighboring faces IDs (in ccw order)
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
		\retval NullPointerGiven All 3 given lists are NULL
		\note The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).\n
			Some of the lists can be NULL, if you don't want to get that data. For example, if you don't want to get the list of edges, just send NULL
			intead of \e edges
	*/
	RESULT getFaceNeighborhood(FaceID fID, LinkedList<VertexID>* vertices=NULL, LinkedList<EdgeID>* edges=NULL, LinkedList<FaceID>* faces=NULL);

	/** Get the neighborhood of the given vertex (adjacent vertices, edges, and neighboring faces). 
		\param vID The ID of the vertex
		\param vertices \out If not NULL, will contain the IDs of the vertices which are adjacent to the given vertex (in ccw order)
		\param edges \out If not NULL, will contain the IDs of the edges which are adjacent to the given vertex (in ccw order)
		\param faces \out If not NULL, will contain the IDs of the faces which are adjacent to the given vertex (in ccw order)
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
		\retval VertexIsOrphan The given vertex is an orphan (is not associated with any face).
		\retval NullPointerGiven All 3 given lists are NULL
		\note This function does not work well if the vertex is complex.\n
			The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).\n
			Some of the lists can be NULL, if you don't want to get that data. For example, if you don't want to get the list of edges, just send NULL
			intead of \e edges
	*/
	RESULT getNeighborhood(VertexID vID, LinkedList<VertexID>* vertices=NULL, LinkedList<EdgeID>* edges=NULL, LinkedList<FaceID>* faces=NULL);

	/** Remove the given vertex and all its incident edges and faces, leaving a hole where the faces were
		\param vID The ID of the vertex to remove
		\param removedFaces \out If not NULL, will contain the IDs of the faces which were removed during the process
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed by this operation (the removed faces' edges) 
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
		\note The output list needs to be allocated before use. Also, the function does not empty the list before use, so make sure the list is empty (unless you want the new data to be added to previous list elements).
		\sa removeVertexTriangulate()
	*/
	RESULT removeVertex(VertexID vID, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Remove the given vertex and all its associated faces, then retriangulate the hole that was created, according to \e newFacesIndices. 
		Each element in the list is a Face, which represents a new face to be created.
		\param vID The ID of the vertex to remove
		\param newFaces A list of new faces which represents new faces to be created
		\param newFacesIDs \out If not NULL, will contain the IDs of the new faces which were created during the retriangulation
		\param removedFaces \out If not NULL, will contain the IDs of the faces which were removed during the process
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed by this operation (the removed faces' edges) 
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
		\retval NullPointerGiven \e newFaces is NULL
		\note The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
		\sa removeVertex()
	*/
	RESULT removeVertexTriangulate(VertexID vID, const LinkedList<Face>* newFaces, LinkedList<FaceID>* newFacesIDs=NULL, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Removes the given face
		\param fID The ID of the face to remove
		\param affectedVertices \out If not NULL, will contain the IDs of the vertices affected by this operation (the removed face's vertices)
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed by this operation (the removed face's edges) 
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with the given ID
		\note The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
	*/
	RESULT removeFace(FaceID fID, LinkedList<VertexID>* affectedVertices=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Performs an "edge collapse": move the first vertex until it collapses with the second vertex. 
		Because of this action 2 edges will be joined and 2 faces will be removed
		\param vID1 The ID of the vertex to move from
		\param vID2 The ID of the vertex to move to
		\param coordinate The new coordinate of the joined vertex.
		\param removedFaces \out If not NULL, will contain the IDs of the faces which were removed during the process 
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed during the process
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval VerticesNotIncident The 2 vertices are not incident
		\retval SameVertexGiven The 2 given vertex IDs are the same
		\note The ID of the merged vertex is vID2. If \e coordinate is not NULL, this vertex will be moved to the new coordinate. Otherwise, its coordinate will be the same as of vID2.\n
			The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
		\sa edgeCollapseGraphic()
	*/
	RESULT edgeCollapse(VertexID vID1, VertexID vID2, const Coord *coordinate=NULL, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Does the same operation as edgeCollapse(), but shows it nicely on the screen. 
		The first vertex is seen as moving towards the second vertex until they merge
		\param vID1 The ID of the vertex to move from
		\param vID2 The ID of the vertex to move to
		\param steps The number of steps to show on the screen
		\param coordinate The new coordinate of the joined vertex.
		\param removedFaces \out If not NULL, will contain the IDs of the faces which were removed during the process 
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed during the process
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
		\retval VerticesNotIncident The 2 vertices are not incident
		\retval SameVertexGiven The 2 given vertex IDs are the same
		\note The ID of the merged vertex is vID2. If \e coordinate is not NULL, this vertex will be moved to the new coordinate. Otherwise, its coordinate will be the same as of vID2.\n
			The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
		\sa edgeCollapse()
	*/
	RESULT edgeCollapseGraphic(VertexID vID1, VertexID vID2, int steps, const Coord *coordinate=NULL, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Split the given vertex to a new vertex, then move it to the given location. 
		The left and right vertices define the vertices which will be attached to the new vertex. All the vertices between left and right (in ccw order) will become the adjacent vertices of the split vertex
		\param vID The ID of the vertex to split
		\param coordinate The coordinate of the new vertex.
		\param left ID of the vertex to the left of the original vertex.
		\param right ID of the vertex to the right of the original vertex
		\param newVertex \out The ID of the new vertex.
		\param newFacesIDs \out If not NULL, will contain the ID's of the new faces which were created
		\param removedFaces \out If not NULL, will contain the ID's of the faces which were removed during the process 
		\param removedEdges \out If not NULL, will contain the IDs of the edges which were removed during the process
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with the given ID
		\retval VerticesNotIncident One of the left and right vertices is not incident to the given vertex to be split
		\note The output lists need to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
	*/
	RESULT vertexSplit(VertexID vID, Coord coordinate, VertexID left, VertexID right, VertexID& newVertex, LinkedList<FaceID>* newFacesIDs=NULL, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);

	/** Adds a new vertex at the specified coordinate, and adds new faces to the model. The new faces that will be created are:  (in the correct orientation)\n
		\e newVertex  \e neighbouringVertices[0].v1  \e neighbouringVertices[0].v2 \n
		\e newVertex  \e neighbouringVertices[1].v1  \e neighbouringVertices[1].v2 \n
		\e newVertex  \e neighbouringVertices[2].v1  \e neighbouringVertices[2].v2 \n
		etc. \n
		Which means that in each new face, the new added vertex is the first vertex, then the 2 other vertices are defined as an Edge in \e neighbouringVertices
		\param coordinate Coordinate of the new vertex.
		\param vID \out The ID of the new added vertex 
		\param neighbouringVertices A list of Edges. Each Edge defines a new face (along with the new vertex) to be added.
		\param newFacesIDs If not NULL, will contain the IDs of the new faces which were created during the process 
		\retval OK Operation succeeded
		\retval VertexNotFound One of the given neighboring vertices' IDs was not found.
		\note if \e neighbouringVertices is NULL, the vertex is added with no adjacent vertices \n
			The output list needs to be allocated before use. Also, the function does not empty the lists before use, so make sure the lists are empty (unless you want the new data to be added to previous list elements).
	*/
	RESULT addVertex(Coord coordinate, VertexID& vID, const LinkedList<Edge>* neighbouringVertices = NULL, LinkedList<FaceID>* newFacesIDs = NULL);

	/** Adds a face to the mesh
		\param face Specifies the new face to be added, according to its 3 vertices' IDs
		\param fID \out The ID of the new added face 
		\param check Whether to check the mesh for the existence of such a face
		\param color The color of the new face, or NULL for the default face color
		\retval OK Operation succeeded
		\retval FaceExists \e check is true, and the face already exists in the mesh (the new face was NOT added to the mesh as a result).
		\retval VertexNotFound  No vertex was found with one of the given face's vertices' IDs
	*/
	RESULT addFace(Face face, FaceID& fID, bool check=false, const Color* color=NULL);

	//@}

	/************************************************/
	/** \name Validation */
	//@{

	// indicates whether the model is empty
#ifndef SKIP_THIS
	bool isEmpty();
#endif
	/** Checks whether the model is empty (no vertices).
		\retval true The model is empty (contains no vertices)
		\retval false The model is not empty (has at least one vertex)
	*/
	bool isModelEmpty();
	/** Checks whether the data structures are empty (model is empty, and there are no lines, spheres and cylinders to draw)
		\retval true The data structures are empty
		\retval false The data structures are not empty
	*/
	bool isTotallyEmpty();
	/** Checks whether there are any lines to draw
		\retval true There are lines to draw
		\retval false There aren't any lines to draw
	*/
	bool hasLines();
	/** Checks whether there are any spheres to draw
		\retval true There are spheres to draw
		\retval false There aren't any spheres to draw
	*/
	bool hasSpheres();
	/** Checks whether there are any cylinders to draw
		\retval true There are cylinders to draw
		\retval false There aren't any cylinders to draw
	*/
	bool hasCylinders();
	/** Checks whether the given vertex id is valid
		\retval true \e vid is valid
		\retval false \e vid is not valid
	*/
	bool isValidVertexIndex(VertexID vID);
	/** Checks whether the given edge id is valid
		\retval true \e eID is valid
		\retval false \e eID is not valid
	*/
	bool isValidEdgeIndex(EdgeID eID);
	/** Checks whether the given face id is valid
		\retval true \e fID is valid
		\retval false \e fID is not valid
	*/
	bool isValidFaceIndex(FaceID fID);
	/** Checks whether the given line id is valid
		\retval true \e lID is valid
		\retval false \e lID is not valid
	*/
	bool isValidLineIndex(LineID lID);
	/** Checks whether the given sphere id is valid
		\retval true \e sID is valid
		\retval false \e sID is not valid
	*/
	bool isValidSphereIndex(SphereID sID);
	/** Checks whether the given cylinder id is valid
		\retval true \e cID is valid
		\retval false \e cID is not valid
	*/
	bool isValidCylinderIndex(CylinderID cID);

	/** Checks whether the given vertex is on a boundary
		\param vID The vertex ID
		\param boundary \out If true, the vertex is on a boundary. Otherwise, false.
		\retval OK Operation succeeded
		\retval VertexNotFound No vertex was found with one of the given IDs
	*/
	RESULT isBoundaryVertex(VertexID vID, bool& boundary);

	/** Checks whether the given edge is on a boundary
		\param eID The edge ID
		\param boundary \out If true, the edge is on a boundary. Otherwise, false.
		\retval OK Operation succeeded
		\retval EdgeNotFound No edge was found with one of the given IDs
	*/
	RESULT isBoundaryEdge(EdgeID eID, bool& boundary);

	/** Checks whether the given face is on a boundary
		\param fID The face ID
		\param boundary \out If true, the face is on a boundary. Otherwise, false.
		\retval OK Operation succeeded
		\retval FaceNotFound No face was found with one of the given IDs
	*/
	RESULT isBoundaryFace(FaceID fID, bool& boundary);

	//@}

	/********* Misc. ***********/

#ifndef SKIP_THIS
	// set the renderer instance
	RESULT setRenderer(Renderer* renderer);
#endif

private:

	// The renderer instance
	Renderer* renderer;
	
	
	/********* projection settings. ***********/

	VRViewpoint*	vrViewpoint;
	VRSFFloat	clipNear;
	VRSFFloat	clipFar;
	VRSFFloat	frustumHalfWidth;
	VRSFFloat	paramScale;
	double		scalefactor;
	double		znear, zfar;


	/********* Normals ***********/

	// calculate normals
	void getNormals();
	void getFaceNormal(FaceID face, VRSFVec3f normal);


	/********* Bounding box ***********/

	VRSFVec3f bBoxMin; // min point
	VRSFVec3f bBoxMax; // max point
	VRSFVec3f bBoxDim; // bb dimensions
	VRSFVec3f center;  // bb center

	// calculate the bounding box
	void getBoundingBox();
	bool bboxValid;


	/********* I/O ***********/

	// current VRML file name
	char* fileName;

	// actually write the output file
	RESULT writeVRMLFile(bool export = false);
	// combine all IFS's into one IFS
	VRIndexedFaceSet* combineIFS(VRMFNode IFS_list);
	void getCubeVertices(VertexID v1, VertexID v2, Coord *coords, double dist);
	void getCubeVertices(Coord coord1, Coord coord2, Coord *coords, double dist);


	/********* mesh geometry ***********/

	// The combined IndexedFaceSet of the model
	VRIndexedFaceSet* vrIFS;
	// database (corner table)
	Corner*			cornerTable;		// corner table
	// keeping validation + colors
	VertexInfo*		verticesInfo;		// vertices info: colors + validation + a list of corners
	EdgeInfo*		edgesInfo;			// edges info: colors + validation
	FaceInfo*		facesInfo;			// faces info: colors + validation
	// total size of vertices/edges/facesInfo (including empty places)
	long	verticesInfoSize;	
	long	edgesInfoSize;		
	long	facesInfoSize;		
	// lists of the empty vertices/faces places in the vertices/facesInfo array
	LinkedList<VertexID>	emptyVertices;	
	LinkedList<FaceID>		emptyFaces;	
	// counting vertices/edges/faces
	VertexID	numOfVertices;
	EdgeID		numOfEdges;
	FaceID		numOfPolys;

	// check validation for a vertex/edge/face ID
	bool	checkVertexIndex(VertexID vID);
	bool	checkEdgeIndex(EdgeID eID);
	bool	checkFaceIndex(FaceID fID);
	EdgeID	countDoubleEdges();
	RESULT	m_edgeCollapse(VertexID vertexID1, VertexID vertexID2, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);
	RESULT	m_addFace(Face face, FaceID& faceId, bool check=false);
	RESULT	m_removeFace(FaceID faceID, LinkedList<VertexID>* affectedVertices=NULL, LinkedList<EdgeID>* removedEdges=NULL);
	RESULT	m_removeVertex(VertexID vertexID, LinkedList<FaceID>* removedFaces=NULL, LinkedList<EdgeID>* removedEdges=NULL);
	RESULT	m_addVertex(Coord coordinate, VertexID& newVertex, const LinkedList<Edge>* neighbouringVertices = NULL, LinkedList<FaceID>* newFacesIDs = NULL);
	bool	areNeighbours(VertexID vertexIndex1, VertexID vertexIndex2, FaceID &t1, FaceID &t2);
	FaceID	getNewFaceIndex();
	VertexID getNewVertexIndex();
	 
	// reallocate memory for databases
	void	allocCornerTable(int oldSize, int newSize);
	void	allocFacesInfo(int oldSize, int newSize);
	void	allocEdgesInfo(int oldSize, int newSize);
	void	allocVerticesInfo(int oldSize, int newSize);
	void	allocLines(int oldSize, int newSize);
	void	allocLinesInfo(int oldSize, int newSize);
	void	allocSpheres(int oldSize, int newSize);
	void	allocCylinders(int oldSize, int newSize);
	void	allocSpheresInfo(int oldSize, int newSize);
	void	allocCylindersInfo(int oldSize, int newSize);
 

	/********* lines ***********/

	Line*		lines;					// lines
	LineInfo*	linesInfo;			// lines info: colors + validation
	long		linesInfoSize;			// the total size of linesInfo (including empty places)
	LineID		numOfLines;				// total number of non-empty lines
	LinkedList<LineID> emptyLines;		// a list of the empty lines places in the lines array

	// check validation for a line ID
	bool	checkLineIndex(LineID lID);	
	LineID	getNewLineIndex();

	/********* spheres ***********/

	Sphere*			spheres;				// lines
	SphereInfo*		spheresInfo;			// lines info: colors + validation
	long			spheresInfoSize;		// the total size of linesInfo (including empty places)
	SphereID		numOfSpheres;			// total number of non-empty lines
	LinkedList<SphereID>	emptySpheres;	// a list of the empty lines places in the lines array

	// check validation for a sphere ID
	bool		checkSphereIndex(SphereID sID);	
	SphereID	getNewSphereIndex();

	/********* cylinders ***********/

	Cylinder*			cylinders;				// lines
	CylinderInfo*		cylindersInfo;			// lines info: colors + validation
	long			cylindersInfoSize;		// the total size of linesInfo (including empty places)
	CylinderID		numOfCylinders;			// total number of non-empty lines
	LinkedList<CylinderID>	emptyCylinders;	// a list of the empty lines places in the lines array

	// check validation for a sphere ID
	bool		checkCylinderIndex(CylinderID cID);	
	CylinderID	getNewCylinderIndex();
	double		getAngle(Coord v1, Coord v2);

	/********* mutex ***********/

	HANDLE mutex;

	void lock();
	void unlock();
};

extern Mesh *mesh;

#endif