// LinkedList.h

#ifndef LINKEDLIST_H
#define LINKEDLIST_H

#include <stdlib.h>

/*** ListElement ***/

#ifndef SKIP_THIS
template<class T> 
class ListElement {
public:
	ListElement() : next(NULL), prev(NULL) {}
	ListElement(const T& _value) : value(_value), next(NULL), prev(NULL) {}
	~ListElement() {}
	T value;
	ListElement* next;
	ListElement* prev;
};
#endif

/*** LinkedList ***/

/** Defines a template for a linked list. A list can only hold values of one type, T, defined when the list is created. For example, the following list: 
\code LinkedList<int> mylist \endcode
can only hold values of type int 
*/

template<class T> 
class LinkedList {

private:

	ListElement<T>* head;
	ListElement<T>* tail;
	long Size;
	
public:

	/** Defines an iterator of a list of type LinkedList. 
		The iterator is used to iterate over the elements of a list. The iterator at each moment is pointing to an element in the list. 
		Example of usage: 
 \code	
LinkedList<int> list;
LinkedList<int>::iterator iter;
int val;
for (iter=list.begin(); iter!=list.end(); iter++) { // iterating over the elements
	val = (int)(*iter); // extracting the value
	.
	.
	.
} \endcode
	*/
	class iterator { 
		friend LinkedList<T>;
	public:
		/** Default constructor. Initializes the iterator to point to nothing. */
		iterator() : ptr(NULL) {}
		#ifndef SKIP_THIS
		/** Constructor */
		iterator(ListElement<T>* _ptr) : ptr(_ptr) {}
		#endif
		/** Copy constructor. Initializes the iterator to point at the same element as \e iter 
			\param iter A reference to a list iterator to copy 
		*/
		iterator(const iterator& iter) {
			ptr = iter.ptr;
		}
		/** Returns the value of the element pointed to by this iterator. 
			\return A reference of the value pointed by the iterator
		*/
		T& operator*() {
			return ptr->value;
		}
		/** Updates the iterator to point on the next list element (prefix operator). 
			\return A reference of the current iterator
		*/
		iterator& operator++() {
			ptr = ptr->next;
			return (*this); 
		}
		/** Updates the iterator to point on the next list element (postfix operator). 
			\return A copy of the iterator
		*/
		iterator operator++(int) { 
			iterator iter = *this;
			++(*this);
			return (iter); 
		}
		/** Updates the iterator to more forward \e inc elements in the list (postfix operator). 
			\param inc The number of steps to move forward in the list  
			\return A copy of the iterator
		*/
		iterator operator+(int inc) {  
			iterator iter = *this;
			for (int i=0; i<inc; i++)
				iter++;
			return (iter); 
		}
		/** Updates the iterator to point on the previous list element (prefix operator). 
			\return A reference of the current iterator
		*/
		iterator& operator--() {
			ptr = ptr->prev;
			return (*this); 
		}
		/** Updates the iterator to point on the previous list element (postfix operator). 
			\return A copy of the iterator
		*/
		iterator operator--(int) {  
			iterator iter = *this;
			--(*this);
			return (iter); 
		}
		/** Updates the iterator to more backwards \e inc elements in the list (postfix operator). 
			\param inc The number of steps to move backwards in the list  
			\return A copy of the iterator
		*/
		iterator operator-(int inc) {  // postfix
			iterator iter = *this;
			for (int i=0; i<inc; i++)
				iter--;
			return (iter); 
		}
		/** Updates the current iterator to point to the same element pointed by the given iterator. 
			\param iter  The iterator to copy  
			\return A reference of the current iterator
		*/
		iterator& operator=(const iterator& iter) {
			ptr = iter.ptr;
			return (*this); 
		}
		/** Compares the 2 iterators (by the elements they're pointing to. 
			\param iter The iterator to compare to  
			\retval true The 2 iterators point to the same element  
			\retval false The 2 iterators point to different elements 
		*/
		bool operator==(const iterator& iter) {
			return (ptr==iter.ptr);
		}
		/** Compares the 2 iterators (by the elements they're pointing to. 
			\param iter The iterator to compare to  
			\retval true The 2 iterators point to different elements  
			\retval false The 2 iterators point to the same element 
		*/
		bool operator!=(const iterator& iter) {
			return (ptr!=iter.ptr);
		}
	private:
		ListElement<T>* ptr;
	};

	/** Constructor. Initializes an empty list */
	LinkedList() {
		head = new ListElement<T>();
		tail = new ListElement<T>();
		head->next = tail;
		tail->prev = head;
		Size=0;
	}
	/** Copy constructor. Initializes the list with a copy of \e list */
	LinkedList(const LinkedList<T>& list) {
		head = new ListElement<T>();
		tail = new ListElement<T>();
		head->next = tail;
		tail->prev = head;
		Size=0;
		push_back_list((LinkedList<T>*)&list);
	}
	/** Destructor, clears the list and frees memory. */
	~LinkedList() {
		clear();
		delete head;
		delete tail;
	}

	/** Copies all elements of \e list to the current list (clears current list first). 
		\param list The list to copy 
	*/
	void operator=(LinkedList<T>& list) {
		clear();
		push_back_list(&list);
	}
	/** Checks if list is empty.
		\retval true The list is empty  
		\retval false The list is not empty (at least one element) 
	*/
	bool empty() const {
		return (Size==0);
	}
	/** Returns the list size (number of elements in the list). 
		\return The list size
	*/
	long size() const {
		return Size;
	}
	/** Checks if the given value is found in the list. 
		\param value The value to check for existence in the list  
		\retval true \e value is found in the list  
		\retval false \e value is not found in the list 
	*/
	bool contains(const T& value) {
		for (iterator iter = begin(); iter != end(); ++iter) {
			if (*iter == value) {
				return true;
			}
		}
		return false;
	}
	/** Checks if the given value is found in the list, and if so, returns an iterator pointing to it. 
		\param value  The value to check for existence in the list  
		\param iter \out An iterator pointing to the element in the list with the given value  
		\retval true \e value is found in the list  
		\retval false \e value is not found in the list 
	*/
	bool contains(const T& value, iterator &iter) {
		for (iterator iter1 = begin(); iter1 != end(); ++iter1) {
			if (*iter1 == value) {
				iter=iter1;
				return true;
			}
		}
		return false;
	}
	/** Clears the list (removes all elements). */
	void clear() {
		if (Size==0) return;
		iterator iter(head->next);
		ListElement<T>* val;
		while (iter.ptr!=tail) {
			val = iter.ptr;
			++iter;
			delete val;
		}
		Size=0;
		head->next = tail;
		tail->prev = head;
	}
	/** Remove the element pointed by \e iter. 
		\param iter An iterator pointing to the element to erase  
		\retval true Element removed succesfully  
		\retval false Operation failed (\e iter was not pointing to a valid element) 
	*/
	bool erase(iterator iter) {
		if (iter.ptr==NULL) return false;
		iter.ptr->prev->next = iter.ptr->next;
		iter.ptr->next->prev = iter.ptr->prev;
		delete iter.ptr;
		Size--;
		return true;
	}
	/** Remove all elements in the list between the element pointed by \e iterBegin, and the element pointed by \e iterEnd (including both). 
		\param iterBegin An iterator pointing to the first element to erase  
		\param iterEnd An iterator pointing to the last element to erase  
		\retval true Elements removed succesfully  
		\retval false Operation failed (\e iterBegin or \e iterEnd were not pointing to valid elements) 
	*/
	bool erase(iterator iterBegin, iterator iterEnd) {
		if (iterBegin.ptr==NULL || iterEnd.ptr==NULL) return false;
		iterator iter, iter1;
		for (iter = iterBegin; iter!=iterEnd; ) {
			iter1=iter++;
			erase(iter1);
		}
		erase(iter);
		return true;
	}
	/** Remove all the elements in the list whose value is equal to the given value. 
		\param value The value to remove from the list  
		\retval true Operation succeeded  
		\retval false No element was removed (none with the given value) 
	*/
	bool remove(const T& value) {
		iterator endIter = end();
		bool removed = false;
		for (iterator iter = begin(); iter != endIter; ) {
			if (*iter == value) {
				erase(iter++);
				removed = true;
			}
			else
				++iter; 
		}
		return removed;
	}
	/** Remove all the elements in the list whose value is equal to the given value, and count the number of elements removed. 
		\param value The value to remove from the list  
		\param count \out The number of elements removed from the list  
		\retval true Operation succeeded  
		\retval false No element was removed (none with the given value) 
	*/
	bool removeAndCount(const T& value, long &count) {
		iterator endIter = end();
		count = 0;
		bool removed = false;
		for (iterator iter = begin(); iter != endIter; ) {
			if (*iter == value) {
				erase(iter++);
				removed = true;
				count++;
			}
			else
				++iter; 
		}
		return removed;
	}
	/** Copies the given list at the end of the current list. 
		\param list The list to copy  
		\retval true Operation succeeded  
		\retval false Operation failed (\e list is NULL) 
	*/
	bool push_back_list(LinkedList<T> *list) {
		LinkedList<T>::iterator iter;
		T element;
		for (iter = list->begin(); iter!=list->end(); iter++) {
			element = (T)(*iter);
			this->push_back(element);
		}
		return true;
	}
	/** Insert a new element with the given value to the head of the list. 
		\param value The value of the element to add  
		\retval true Operation succeeded  
		\retval false Operation failed 
	*/
	bool push_front(const T& value) {
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->next = head->next;
		newElement->next->prev = newElement;
		newElement->prev = head;
		newElement->prev->next = newElement;
		Size++;
		return true;
	}

	/** Insert a new element with the given value to the end of the list. 
		\param value The value of the element to add  
		\retval true Operation succeeded  
		\retval false Operation failed 
	*/
	bool push_back(const T& value) {
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->prev = tail->prev;
		newElement->prev->next = newElement;
		newElement->next = tail;
		newElement->next->prev = newElement;
		Size++;
		return true;
	}
	/** Insert a new element with the given value to the head of the list, only if the same value is not already contained in the list. 
		\param value The value of the element to add  
		\retval true Operation succeeded (element added)  
		\retval false Operation failed (\e value already exists in the list) 
	*/
	bool push_front_unique(const T& value) {
		if (contains(value))
			return false;
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->next = head->next;
		newElement->next->prev = newElement;
		newElement->prev = head;
		newElement->prev->next = newElement;
		Size++;
		return true;
	}
	/** Insert a new element with the given value to the end of the list, only if the same value is not already contained in the list. 
		\param value The value of the element to add  
		\retval true Operation succeeded (element added)  
		\retval false Operation failed (\e value already exists in the list) 
	*/
	bool push_back_unique(const T& value) {
		if (contains(value))
			return false;
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->prev = tail->prev;
		newElement->prev->next = newElement;
		newElement->next = tail;
		newElement->next->prev = newElement;
		Size++;
		return true;
	}

	/** Remove the first element of the list and return it. 
		If the list is already empty, the function fails and calls exit(1) (after printing an error message) 
		\return The value of the first element of the list (which was removed)
	*/
	T pop_front() {
		if (Size>0) {
			ListElement<T>* removeElement = head->next;
			T removeVal = removeElement->value;
			head->next = removeElement->next;
			removeElement->next->prev =	head;
			delete removeElement;
			Size--;
			return removeVal;
		}
		else {
			MessageBox(NULL,"LinkedList::pop_front - no element to pop, exiting ...\n", "Error", MB_OK);
			exit(1);
		}
	}
	/** Remove the last element of the list and return it. 
		If the list is already empty, the function fails and calls exit(1) (after printing an error message) 
		\return The value of the last element of the list (which was removed)
	*/
	T pop_back() {
		if (Size>0) {
			ListElement<T>* removeElement = tail->prev;
			T removeVal = removeElement->value;
			tail->prev = removeElement->prev;
			removeElement->prev->next =	tail;
			delete removeElement;
			Size--;
			return removeVal;
		}
		else {
			MessageBox(NULL,"LinkedList::pop_back - no element to pop, exiting ...\n", "Error", MB_OK);
			exit(1);
		}
	}
	/** Returns the first element of the list (without removing it). 
		If the list is empty, the function fails and calls exit(1) 
		\return A reference of the first element of the list
	*/
	T& front() {
		if (Size>0)
			return head->next->value;
		else {
			MessageBox(NULL,"LinkedList::front - list is empty, exiting ...\n", "Error", MB_OK);
			exit(1);
		}
	}
	/** Returns the last element of the list (without removing it). 
		If the list is empty, the function fails and calls exit(1) 
		\return A reference of the last element of the list
	*/
	T& back() {
		if (Size>0)
			return tail->prev->value;
		else {
			MessageBox(NULL,"LinkedList::back - list is empty, exiting ...\n", "Error", MB_OK);
			exit(1);
		}
	}

	/** Returns an iterator pointing to the first element of the list. 
		\return An iterator pointing to first element of the list 
		\sa Usage for begin() and end() in the documentation for LinkedList::iterator
	*/
	iterator begin() {
		iterator iter(head->next);
		return iter;
	}

	/** Returns an iterator pointing to a dummy element, which is an element after the last element of the list. 
		\return An iterator pointing to first element of the list 
		\sa Usage for begin() and end() in the documentation for LinkedList::iterator
	*/
	iterator end() {
		iterator iter(tail);
		return iter;
	}

	/** Inserts a new element with the given value to the list, after the element pointed to by \e iter. 
		\param iter The iterator after which the element should be added  
		\param value The value to add  
		\return If \e iter is NULL, the value is not inserted and the function returns NULL; else, the function returns an iterator pointing to the element holding the new value
	*/
	iterator* insertAfter(iterator* iter, const T& value) {
		if (iter==NULL)
			return NULL;
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->next = iter->ptr->next;
		newElement->next->prev = newElement;
		newElement->prev = iter->ptr;
		newElement->prev->next = newElement;
		Size++;
		iterator* newIter = new iterator(newElement);
		return newIter;
	}

	/** Inserts a new element with the given value to the list, before the element pointed to by \e iter. 
		\param iter The iterator after which the element should be added  
		\param value The value to add  
		\return If \e iter is NULL, the value is not inserted and the function returns NULL; else, the function returns an iterator pointing to the element holding the new value 
	*/
	iterator* insertBefore(iterator* iter, const T& value) {
		if (iter==NULL)
			return NULL;
		ListElement<T>* newElement = new ListElement<T>(value);
		newElement->prev = iter->ptr->prev;
		newElement->prev->next = newElement;
		newElement->next = iter->ptr;
		newElement->next->prev = newElement;
		Size++;
		iterator* newIter = new iterator(newElement);
		return newIter;
	}

};

#endif