// face.h

#include "params.h"

#ifndef FACE_H
#define FACE_H

/** Defines a face. The face is defined by the VertexID's of the face's vertices. 
	Currently, only triangles are allowed. The orientation of the vertices is: v1 v2 v3 (ccw).
	In order for the software to work properly, all faces have to be in the same orientation.
*/
class Face {

private:
	VertexID v[MAX_FACE_VERTICES];

public:
	/** Constructor. Initializes the vertices IDs.
		\param v An array of the face's vertices IDs. If NULL, all VertexID's are initialized to -1
	*/
	Face(VertexID v[MAX_FACE_VERTICES] = NULL) { 
		if (v==NULL) {
			for (int i=0; i<MAX_FACE_VERTICES; i++)
				this->v[i] = -1; 
		} else {
			for (int i=0; i<MAX_FACE_VERTICES; i++)
				this->v[i] = v[i]; 
		}
	}
	/** Sets the vertices IDs.
		\param v An array of the face's vertices IDs. If NULL, all VertexID's are initialized to -1
	*/
	void set(VertexID v[MAX_FACE_VERTICES] = NULL) { 
		if (v==NULL) {
			for (int i=0; i<MAX_FACE_VERTICES; i++)
				this->v[i] = -1; 
		} else {
			for (int i=0; i<MAX_FACE_VERTICES; i++)
				this->v[i] = v[i]; 
		}
	}
	/** Get the ID of the vertex in v[vNum]
		\param vNum The index of the ID in the array v
		\return The ID of the vertex in v[vNum]
	*/
	VertexID getV(int vNum) const {
		if (vNum>=MAX_FACE_VERTICES || vNum<0)
			return -1;
		return v[vNum];
	}

	/** Compares 2 Faces. Faces are the same if they have the same VertexID's and the same orientation (not necessarily in the same order).
		For example, a face with VertexID's {1, 3, 4} is the same as the one with {3, 4, 1}, but different from {1, 4, 3}.
		\param face The Face to compare to
		\retval true The faces are the same
		\retval false The faces are different
	*/
	bool operator==(const Face &face) const {
		for (int i=0; i<3; i++) {
			if (face.v[i]==v[0] && face.v[(i+1)%3]==v[1] && face.v[(i+2)%3]==v[2]) return true;
		}
		return false;
	}
	/** Sets the face according to \a face
		\param face The Face to copy
	*/
	void operator=(const Face &face) {
		for (int i=0; i<3; i++)
			v[i]=face.v[i];
	}
};

#endif