// cylinder.h

#ifndef CYLINDER_H
#define CYLINDER_H

#include "coord.h"

/**
* Defines a cylinder, whose ends can be rendered as cones. The cylinder is defined by 2 coordinates
  (top and bottom), a radius, and the lengths of cones that can be drawn at its top or bottom. 
  The height of each cone can be no more than half the cylinder's length (if it is longer, the cone will be drawn as having half the length of the cylinder). 

  Examples for cylinders with different parameters:

  \li bottom=(-3,0,0), top=(4,0,0), radius=2, bottomCone=topCone=0
      \image html cyl1.jpg
  \li bottom=(-3,0,0), top=(4,0,0), radius=2, bottomCone=3, topCone=0
      \image html cyl2.jpg
  \li bottom=(-3,0,0), top=(4,0,0), radius=2, bottomCone=3, topCone=1
      \image html cyl3.jpg
  \li bottom=(-3,0,0), top=(4,0,0), radius=2, bottomCone>=3.5, topCone>=3.5;
      \image html cyl4.jpg
*/

class Cylinder {

private:
	Coord	bottom;
	Coord	top;
	double	radius;
	double	bottomCone;		// The bottom cone length
	double	topCone;		// The top cone length
public:

	/** Default constructor. Initializes the radius to be DEFAULT_CYLINDER_RADIUS, and no cones (topCone=bottomCone=0). */
	Cylinder():radius(DEFAULT_CYLINDER_RADIUS),bottomCone(0),topCone(0) {} 

	/** Constructor.
		\param bottom The bottom coordinate
		\param top The top coordinate
		\param radius The radius of the cylinder
		\param topCone The length of the top cone (starts at the top coordinate inwards)
		\param bottomCone The length of the bottom cone (starts at the bottom coordinate inwards)
	*/
	Cylinder(Coord bottom, Coord top, double radius, double topCone=0, double bottomCone=0) {
		this->bottom = bottom;
		this->top = top;
		this->radius = radius;
		this->topCone = topCone;
		this->bottomCone = bottomCone;
	}

	/** Set the bottom and top coordinates
		\param bottom The bottom coordinate
		\param top The top coordinate
	*/
	void setLocations(Coord bottom, Coord top) { 
		this->bottom = bottom;
		this->top = top;
	}

	/** Set the radius
		\param radius The radius of the cylinder
	*/
	void setRadius(double radius) { 
		this->radius = radius;
	}

	/** Set the bottom and top cone lengths
		\param topCone The length of the top cone (starts at the top coordinate inwards)
		\param bottomCone The length of the bottom cone (starts at the bottom coordinate inwards)
	*/
	void setCones(double topCone=0, double bottomCone=0) { 
		this->topCone = topCone;
		this->bottomCone = bottomCone;
	}

	/** Set all parameters 
		\param bottom The bottom coordinate
		\param top The top coordinate
		\param radius The radius of the cylinder
		\param topCone The length of the top cone (starts at the top coordinate inwards)
		\param bottomCone The length of the bottom cone (starts at the bottom coordinate inwards)
	*/
	void setParams(Coord top, Coord bottom, double radius, double topCone=0, double bottomCone=0) { 
		this->bottom = bottom;
		this->top = top;
		this->radius = radius;
		this->topCone = topCone;
		this->bottomCone = bottomCone;
	}

	/** Get the bottom and top coordinates.
		\param bottom \out The bottom coordinate
		\param top \out The top coordinate
	*/
	void getLocations(Coord& top, Coord& bottom) const  { 
		bottom = this->bottom;
		top = this->top;
	}

	/** Set the radius
		\param radius \out The radius of the cylinder
	*/
	void getRadius(double& radius) const { 
		radius = this->radius;
	}

	/** Get the bottom and top cone lengths
		\param topCone \out The length of the top cone (starts at the top coordinate inwards)
		\param bottomCone \out The length of the bottom cone (starts at the bottom coordinate inwards)
	*/
	void getCones(double &topCone, double &bottomCone) const { 
		topCone = this->topCone;
		bottomCone = this->bottomCone;
	}

	/** Get all parameters
		\param bottom \out The bottom coordinate
		\param top \out The top coordinate
		\param radius \out The radius of the cylinder
		\param topCone \out The length of the top cone (starts at the top coordinate inwards)
		\param bottomCone \out The length of the bottom cone (starts at the bottom coordinate inwards)
	*/
	void getParams(Coord& top, Coord& bottom, double& radius, double &topCone, double &bottomCone) const { 
		bottom = this->bottom;
		top = this->top;
		radius = this->radius;
		topCone = this->topCone;
		bottomCone = this->bottomCone;
	}

};

#endif