// coord.h

#ifndef COORD_H
#define COORD_H

#include "vr.h"

/** Defines a 3D coordinate. Every component (x, y, and z) is a double. 
	The coordinate can also be used as a vector, starting at (0,0,0) and ending at (x,y,z). */

class Coord  
{ 
public:
	/** The x coordinate */
    double x;
	/** The y coordinate */
	double y; 
	/** The z coordinate */
	double z;

public:
	/** Default constructor. Initializes the coordinate to (0, 0, 0). */
	Coord();
	/** Constructor. Initializes the coordinate to (x, y, z). 
		\param x The x coordinate	
		\param y The y coordinate
		\param z The z coordinate
	*/
	Coord(double x, double y = 0.0, double z = 0.0);
	/** Set the coordinate
		\param coord A 3D coordinate
	*/
	Coord(VRSFVec3f coord);
	/** Set the coordinate
		\param x The x coordinate	
		\param y The y coordinate
		\param z The z coordinate
	*/
	void setCoords(double x = 0.0, double y = 0.0, double z = 0.0);

	/** Adds \a coord to current coordinate. 
		\param coord The coordinate to add
		\return A reference to this object
	*/
	Coord& operator +=(const Coord &coord);
	/** Subtract \a coord from current coordinate. 
		\param coord The coordinate to subtract
		\return A reference to this object
	*/
	Coord& operator -=(const Coord &coord);
	/** Adds \a coord to current coordinate
		\param coord The coordinate to add
		\return A new Coord object with the result
	*/
	Coord operator +(const Coord &coord) const;
	/** Subtract \a coord from current coordinate
		\param coord The coordinate to subtract
		\return A new Coord object with the result
	*/
	Coord operator -(const Coord &coord) const;
	/** Returns the current coordinate multiplied by -1
		\return A new Coord object with the result
	*/
	Coord operator -() const;
	/** Compares 2 coordinates and returns true if they are the same
		\param coord The coordinate to compare to
		\retval true The 2 coordinates are the same
		\retval false The coordinates are different
	*/
	Coord& operator =(const Coord &coord);
	/** Compares 2 coordinates and returns true if they are the same
		\param coord The coordinate to compare to
		\retval true The 2 coordinates are the same
		\retval false The coordinates are different
	*/
	bool operator ==(const Coord &coord) const;

	/** Multiplies coordinate by \a d
		\param d The parameter to multiply by
		\return A reference to this object
	*/
	Coord& operator *=(double d);
	/** Divides coordinate by \a d
		\param d The parameter to divide by
		\return A reference to this object
	*/
	Coord& operator /=(double d);
	/** Multiplies coordinate by \a d
		\param d The parameter to multiply by
		\return A new Coord object with the result
	*/
	Coord operator *(double d) const;
	/** Divides coordinate by \a d
		\param d The parameter to divide by
		\return A new Coord object with the result
	*/
	Coord operator /(double d) const;

	/** Returns the x/y/z coordinate (depends on \a i). 
		\param i Specifies the x/y/z coordinate (0 for x, 1 for y, 2 for z)
		\return The current x/y/z coordinate
	*/
	double& operator [](int i) ;
	/** Returns the x/y/z coordinate (depends on \a i). 
		\param i Specifies the x/y/z coordinate (0 for x, 1 for y, 2 for z)
		\return The current x/y/z coordinate
	*/
	double operator [](int i) const;

	/** Returns the dot product between the 2 coordinates
		\param c1 First coordinate
		\param c2 Second coordinate
		\return The dot product result
	*/
	friend double dot(const Coord& c1, const Coord& c2);
	/** Returns the cross product between the 2 coordinates
		\param c1 First coordinate
		\param c2 Second coordinate
		\return The cross product result
	*/
	friend Coord cross(const Coord& c1, const Coord& c2);

	/** Returns the norm of the vector.
		It is also the dot product between the vector and itself 
		\return The norm of the vector
	*/
	double sqrabs() const;
	/** Returns the length of the vector.
		It is also the square root of the dot product between the vector and itself 
		\return The length of the vector
	*/
	double abs() const;
	/** Returns a normalized vector (same direction as original vector, but with length=1) 
		\return Normalized vector
	*/
	Coord unit() const;
	/** Normalizes the vector to be of length=1 */
    void normalize();

};

/** Defines a 2D coordinate. Every component (x and y) is a double. 
	The coordinate can also be used as a vector, starting at (0,0) and ending at (x,y). */

class Coord2D 
{ 
public:
	/** The x coordinate */
    double x;
	/** The y coordinate */
	double y; 

public:
	/** Default constructor. Initializes the coordinate to (0, 0). */
	Coord2D();
	/** Constructor. Initializes the coordinate to (x, y). 
		\param x The x coordinate	
		\param y The y coordinate
	*/
	Coord2D(double x, double y = 0.0);
	/** Set the coordinate
		\param x The x coordinate	
		\param y The y coordinate
	*/
	void setCoords(double x = 0.0, double y = 0.0);

	/** Adds \a coord to current coordinate. 
		\param coord The coordinate to add
		\return A reference to this object
	*/
	Coord2D& operator +=(const Coord2D &coord);
	/** Subtract \a coord from current coordinate. 
		\param coord The coordinate to subtract
		\return A reference to this object
	*/
	Coord2D& operator -=(const Coord2D &coord);
	/** Adds \a coord to current coordinate
		\param coord The coordinate to add
		\return A new Coord2D object with the result
	*/
	Coord2D operator +(const Coord2D &coord) const;
	/** Subtract \a coord from current coordinate
		\param coord The coordinate to subtract
		\return A new Coord2D object with the result
	*/
	Coord2D operator -(const Coord2D &coord) const;
	/** Returns the current coordinate multiplied by -1
		\return A new Coord2D object with the result
	*/
	Coord2D operator -() const;
	/** Compares 2 coordinates and returns true if they are the same
		\param coord The coordinate to compare to
		\retval true The 2 coordinates are the same
		\retval false The coordinates are different
	*/
	bool operator ==(const Coord2D &coord) const;
	/** Compares 2 coordinates and returns true if they are the same
		\param coord The coordinate to compare to
		\retval true The 2 coordinates are the same
		\retval false The coordinates are different
	*/
	Coord2D& operator =(const Coord2D &coord);

	/** Multiplies coordinate by \a d
		\param d The parameter to multiply by
		\return A reference to this object
	*/
	Coord2D& operator *=(double d);
	/** Divides coordinate by \a d
		\param d The parameter to divide by
		\return A reference to this object
	*/
	Coord2D& operator /=(double d);
	/** Multiplies coordinate by \a d, returns a new Coord2D object
		\param d The parameter to multiply by
		\return A new Coord2D object with the result
	*/
	Coord2D operator *(double d) const;
	/** Divides coordinate by \a d, returns a new Coord2D object
		\param d The parameter to divide by
		\return A new Coord2D object with the result
	*/
	Coord2D operator /(double d) const;

	/** Returns the x/y coordinate (depends on \a i). 
		\param i Specifies the x/y coordinate (0 for x, 1 for y)
		\return The current x/y coordinate
	*/
	double& operator [](int i) ;
	/** Returns the x/y/z coordinate (depends on \a i). 
		\param i Specifies the x/y/z coordinate (0 for x, 1 for y)
		\return The current x/y coordinate
	*/
	double operator [](int i) const;

	/** Returns the dot product between the 2 coordinates
		\param c1 First coordinate
		\param c2 Second coordinate
		\return The dot product result
	*/
	friend double dot(const Coord2D& c1, const Coord2D& c2);

	/** Returns the norm of the vector.
		It is also the dot product between the vector and itself 
		\return The norm of the vector
	*/
	double sqrabs() const;
	/** Returns the length of the vector.
		It is also the square root of the dot product between the vector and itself 
		\return The length of the vector
	*/
	double abs() const;
	/** Returns a normalized vector (same direction as original vector, but with length=1) 
		\return Normalized vector
	*/
	Coord2D unit() const;
	/** Normalizes the vector to be of length=1 */
    void normalize();

};

#endif