// color.h

#ifndef COLOR_H
#define COLOR_H

#include <stdlib.h>
#include <gl/glaux.h>
#include "params.h"

/** Defines an RGB color. Every component is a double at the range [0.0, 1.0]. 
	After each operation an overflow check is made for every component, and if necessary, the component is updated.
	For example, if the R component is larger than 1.0, it will be updated to 1.0; if smaller than 0.0, it will be updated to 0.0
*/
class Color {

public:
	
	/** Default constructor. Initializes the color to black (0, 0, 0). */
	Color();
	/** Constructor. Initializes the color to (r, g, b). 
		\param r The red component	
		\param g The green component
		\param b The blue component
	*/
	Color(double r, double g, double b); 
	/** Constructor. Initializes the color according to the given color. 
		\param color A color
	*/
	Color(COLORREF color);

	/** Set the color components
		\param r The red component	
		\param g The green component
		\param b The blue component
	*/
	void setColor(double r, double g, double b);

	/** Returns the R/G/B value (depends on \a col). 
		\param col Specifies the R/G/B component
		\return The current R/G/B value
	*/
	double& operator[](COLOR col);
	/** Returns the R/G/B value (depends on \a col). 
		\param col Specifies the R/G/B component
		\return The current R/G/B value
	*/
	double operator[](COLOR col) const;

	/** Sets the current color according to the given color 
		\param color The color to copy
		\return A reference to this object
	*/
	Color& operator=(const Color &color);

	/** Adds \a color to current color. 
		\param color The color to add
		\return A reference to this object
	*/
	Color& operator +=(const Color &color);
	/** Subtracts \a color to current color.
		\param color The color to subtract
		\return A reference to this object
	*/
	Color& operator -=(const Color &color);
	/** Returns the sum of current color and \a color
		\param color The color to add
		\return A new Color object with the result
	*/
	Color operator +(const Color &color) const;
	/** Subtracts \a color from current color, returns a new Color object.
		\param color The color to subtract
		\return A new Color object with the result
	*/
	Color operator -(const Color &color) const;
	/** Returns the current color multiplied by -1
		\return A new Color object with the result
	*/
	Color operator -() const;

	/** Multiplies every color component by \a d
		\param d The parameter to multiply by
		\return A reference to this object
	*/
	Color& operator *=(double d);
	/** Divides every color component by \a d
		\param d The parameter to divide by
		\return A reference to this object
	*/
	Color& operator /=(double d);
	/** Multiplies every color component by \a d, returns a new Color object
		\param d The parameter to multiply by
		\return A new Color object with the result
	*/
	Color operator *(double d) const;
	/** Divides every color component by \a d, returns a new Color object
		\param d The parameter to divide by
		\return A new Color object with the result
	*/
	Color operator /(double d) const;
	/** Compares 2 colors and returns true if they are the same
		\param color The color to compare to
		\retval true The 2 colors are the same
		\retval false The colors are different
	*/
	bool operator ==(const Color &color) const;

	/** Returns the absolute value of the color, i.e. sqrt(r*r+g*g+b*b) 
		\return The absolute value
	*/
	double abs() const;
	/** Sets a random color */
	void setRandomColor();

private:
	double rgb[3];
	void checkOverflow();
};

#endif